/*****************************************************************************
 * $Id: ix-debug.c,v 1.3 2005/08/26 10:44:10 killabyte Exp $
 *
 * Some tests used in debug mode to check the consistency of ELF structures in
 * memory.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>
#include<irix/elfops.h>

/*****************************************************************************
 * All known sections names (well... almost all) for this platform
 *
 *   Used to dump ELF structures in a human readable way. I really wish that
 *   macros used to build the list don't give too much problems on the
 *   different C compilers of this universe...
 *
 *****************************************************************************/

typedef struct {
  int     d_tag;
  char    *name;
  char    *desc;
} DT_NAMES;
  
#define BDTDESC(x,y)            { x, #x, y }
#define BDTDESC_END             { 0, NULL, NULL }

static DT_NAMES dt_names[] = {
  BDTDESC(DT_NULL,                       "Marks end of dynamic section"),
  BDTDESC(DT_NEEDED,                     "Name of needed library"),
  BDTDESC(DT_PLTRELSZ,                   "Size in bytes of PLT relocs"),
  BDTDESC(DT_PLTGOT,                     "Processor defined value"),
  BDTDESC(DT_HASH,                       "Address of symbol hash table"),
  BDTDESC(DT_STRTAB,                     "Address of string table"),
  BDTDESC(DT_SYMTAB,                     "Address of symbol table"),
  BDTDESC(DT_RELA,                       "Address of Rela relocs"),
  BDTDESC(DT_RELASZ,                     "Total size of Rela relocs"),
  BDTDESC(DT_RELAENT,                    "Size of one Rela reloc"),
  BDTDESC(DT_STRSZ,                      "Size of string table"),
  BDTDESC(DT_SYMENT,                     "Size of one symbol table entry"),
  BDTDESC(DT_INIT,                       "Address of init function"),
  BDTDESC(DT_FINI,                       "Address of termination function"),
  BDTDESC(DT_SONAME,                     "Name of shared object"),
  BDTDESC(DT_RPATH,                      "Library search path (deprecated)"),
  BDTDESC(DT_SYMBOLIC,                   "Start symbol search here"),
  BDTDESC(DT_REL,                        "Address of Rel relocs"),
  BDTDESC(DT_RELSZ,                      "Total size of Rel relocs"),
  BDTDESC(DT_RELENT,                     "Size of one Rel reloc"),
  BDTDESC(DT_PLTREL,                     "Type of reloc in PLT"),
  BDTDESC(DT_DEBUG,                      "For debugging; unspecified"),
  BDTDESC(DT_TEXTREL,                    "Reloc might modify .text"),
  BDTDESC(DT_JMPREL,                     "Address of PLT relocs"),
  BDTDESC(DT_LOPROC,                     "Start of processor-specific"),  
  BDTDESC(DT_HIPROC,                     "End of processor-specific"),  

  BDTDESC(DT_IRIX_LTR,                   "DT_IRIX_LTR"),
  BDTDESC(DT_IRIX_LPT,                   "DT_IRIX_LPT"),
  BDTDESC(DT_IRIX_LPT_IDX,               "DT_IRIX_LPT_IDX"),
  BDTDESC(DT_IRIX_LPT_NO,                "DT_IRIX_LPT_NO"),
  BDTDESC(DT_IRIX_PLT,                   "DT_IRIX_PLT"),
  BDTDESC(DT_IRIX_PLT_NON_PREEMPT_IDX,   "DT_IRIX_PLT_NON_PREEMPT_IDX"),
  BDTDESC(DT_IRIX_PLT_NON_PREEMPT_NO,    "DT_IRIX_PLT_NON_PREEMPT_NO"),
  BDTDESC(DT_IRIX_PLT_IN_DYNSYM_IDX,     "DT_IRIX_PLT_IN_DYNSYM_IDX"),
  BDTDESC(DT_IRIX_PLT_IN_DYMSYM_NO,      "DT_IRIX_PLT_IN_DYMSYM_NO"),
  BDTDESC(DT_IRIX_PLT_PROTECTED_IDX,     "DT_IRIX_PLT_PROTECTED_IDX"),
  BDTDESC(DT_IRIX_PLT_PROTECTED_NO,      "DT_IRIX_PLT_PROTECTED_NO"),
  BDTDESC(DT_IRIX_PLT_PREEMPT_IDX,       "DT_IRIX_PLT_PREEMPT_IDX"),
  BDTDESC(DT_IRIX_PLT_PREEMPT_NO,        "DT_IRIX_PLT_PREEMPT_NO"),
  BDTDESC(DT_IRIX_DLT,                   "DT_IRIX_DLT"),
  BDTDESC(DT_IRIX_DLT_NON_PREEMPT_IDX,   "DT_IRIX_DLT_NON_PREEMPT_IDX"),
  BDTDESC(DT_IRIX_DLT_NON_PREEMPT_NO,    "DT_IRIX_DLT_NON_PREEMPT_NO"),
  BDTDESC(DT_IRIX_DLT_IN_DYNSYM_IDX,     "DT_IRIX_DLT_IN_DYNSYM_IDX"),
  BDTDESC(DT_IRIX_DLT_IN_DYMSYM_NO,      "DT_IRIX_DLT_IN_DYMSYM_NO"),
  BDTDESC(DT_IRIX_DLT_PROTECTED_IDX,     "DT_IRIX_DLT_PROTECTED_IDX"),
  BDTDESC(DT_IRIX_DLT_PROTECTED_NO,      "DT_IRIX_DLT_PROTECTED_NO"),
  BDTDESC(DT_IRIX_DLT_PREEMPT_IDX,       "DT_IRIX_DLT_PREEMPT_IDX"),
  BDTDESC(DT_IRIX_DLT_PREEMPT_NO,        "DT_IRIX_DLT_PREEMPT_NO"),
  BDTDESC(DT_IRIX_DSYM_PLT_IDX,          "DT_IRIX_DSYM_PLT_IDX"),
  BDTDESC(DT_IRIX_DSYM_DLT_IDX,          "DT_IRIX_DSYM_DLT_IDX"),

  BDTDESC(DT_MIPS_RLD_VERSION,           "DT_MIPS_RLD_VERSION"),
  BDTDESC(DT_MIPS_TIME_STAMP,            "DT_MIPS_TIME_STAMP"),
  BDTDESC(DT_MIPS_ICHECKSUM,             "DT_MIPS_ICHECKSUM"),
  BDTDESC(DT_MIPS_IVERSION,              "DT_MIPS_IVERSION"),
  BDTDESC(DT_MIPS_FLAGS,                 "DT_MIPS_FLAGS"),
  BDTDESC(DT_MIPS_BASE_ADDRESS,          "DT_MIPS_BASE_ADDRESS"),
  BDTDESC(DT_MIPS_MSYM,                  "DT_MIPS_MSYM"),
  BDTDESC(DT_MIPS_CONFLICT,              "DT_MIPS_CONFLICT"),
  BDTDESC(DT_MIPS_LIBLIST,               "DT_MIPS_LIBLIST"),
  BDTDESC(DT_MIPS_LOCAL_GOTNO,           "DT_MIPS_LOCAL_GOTNO"),
  BDTDESC(DT_MIPS_CONFLICTNO,            "DT_MIPS_CONFLICTNO"),
  BDTDESC(DT_MIPS_LIBLISTNO,             "DT_MIPS_LIBLISTNO"),
  BDTDESC(DT_MIPS_SYMTABNO,              "DT_MIPS_SYMTABNO"),
  BDTDESC(DT_MIPS_UNREFEXTNO,            "DT_MIPS_UNREFEXTNO"),
  BDTDESC(DT_MIPS_GOTSYM,                "DT_MIPS_GOTSYM"),
#ifndef __osf__
  BDTDESC(DT_MIPS_HIPAGENO,              "DT_MIPS_HIPAGENO"),
  BDTDESC(DT_MIPS_RLD_MAP,               "DT_MIPS_RLD_MAP"),
# if (defined(_DELTA_C_PLUS_PLUS) && (_DELTA_C_PLUS_PLUS==1))
  BDTDESC(DT_MIPS_DELTA_CLASS,           "DCC class definitions"),
  BDTDESC(DT_MIPS_DELTA_CLASS_NO,        "No. DCC class definitions"),
  BDTDESC(DT_MIPS_DELTA_INSTANCE,        "DCC class instances"),
  BDTDESC(DT_MIPS_DELTA_INSTANCE_NO,     "No. DCC class instances"),
  BDTDESC(DT_MIPS_DELTA_RELOC,           "DCC relocations"),
  BDTDESC(DT_MIPS_DELTA_RELOC_NO,        "No.  DCC relocations"),
  BDTDESC(DT_MIPS_DELTA_SYM,             "DCC symbols Refered to by DCC relocations"),
  BDTDESC(DT_MIPS_DELTA_SYM_NO           "No. DCC symbols"),
  BDTDESC(DT_MIPS_DELTA_CLASSSYM         "DCC class declarations"),
  BDTDESC(DT_MIPS_DELTA_CLASSSYM_NO      "No. DCC class declarations"),
# endif	/* _DELTA_C_PLUS_PLUS */
  BDTDESC(DT_MIPS_CXX_FLAGS,             "Flags:  C++ flavor"),
  BDTDESC(DT_MIPS_PIXIE_INIT,            "Init code for pixie"),
  BDTDESC(DT_MIPS_SYMBOL_LIB,            "DT_MIPS_SYMBOL_LIB"),
  BDTDESC(DT_MIPS_LOCALPAGE_GOTIDX,      "DT_MIPS_LOCALPAGE_GOTIDX"),
  BDTDESC(DT_MIPS_LOCAL_GOTIDX,          "DT_MIPS_LOCAL_GOTIDX"),
  BDTDESC(DT_MIPS_HIDDEN_GOTIDX,         "DT_MIPS_HIDDEN_GOTIDX"),
  BDTDESC(DT_MIPS_PROTECTED_GOTIDX,      "DT_MIPS_PROTECTED_GOTIDX"),
  BDTDESC(DT_MIPS_OPTIONS,               "Address of .options"),
  BDTDESC(DT_MIPS_INTERFACE,             "Address of .interface"),
  BDTDESC(DT_MIPS_DYNSTR_ALIGN,          "DT_MIPS_DYNSTR_ALIGN"),
  BDTDESC(DT_MIPS_INTERFACE_SIZE,        "size of the .interface"),
  BDTDESC(DT_MIPS_RLD_TEXT_RESOLVE_ADDR, "Addr of rld_text_resolve"),
  BDTDESC(DT_MIPS_PERF_SUFFIX,           "suffix to be added to dso name before dlopen() call"),
  BDTDESC(DT_MIPS_COMPACT_SIZE,          "(O32)Size of .compact_rel"),
  BDTDESC(DT_MIPS_GP_VALUE,              "gp value for aux gots"),
  BDTDESC(DT_MIPS_AUX_DYNAMIC,           "Address of aux .dynamic"),
#else /* __osf__ */
  BDTDESC(DT_MIPS_PACKAGE,               "DT_MIPS_PACKAGE"),
  BDTDESC(DT_MIPS_PACKAGENO,             "DT_MIPS_PACKAGENO"),
  BDTDESC(DT_MIPS_PACKSYM,               "DT_MIPS_PACKSYM"),
  BDTDESC(DT_MIPS_PACKSYMNO,             "DT_MIPS_PACKSYMNO"),
  BDTDESC(DT_MIPS_IMPACKNO,              "DT_MIPS_IMPACKNO"),
  BDTDESC(DT_MIPS_EXPACKNO,              "DT_MIPS_EXPACKNO"),
  BDTDESC(DT_MIPS_IMPSYMNO,              "DT_MIPS_IMPSYMNO"),
  BDTDESC(DT_MIPS_EXPSYMNO,              "DT_MIPS_EXPSYMNO"),
  BDTDESC(DT_MIPS_HIPAGENO,              "DT_MIPS_HIPAGENO"),
#endif /* __osf__ */
  BDTDESC_END
};

/*****************************************************************************
 * Section check table
 * 
 *   In this table is expressed which sections should exist always and which
 *   are optional. Only these sections will be examined, any other section
 *   found will be ignored. Note that you can add a pointer to a function to
 *   check deeply a section. The order in this table is important.
 *****************************************************************************/

typedef struct _tag_CHECK_DT_SECT {
  int          d_tag;
  int          d_un;
#define D_UN_IGNORED    0
#define D_UN_VAL        1
#define D_UN_PTR        2
#define D_UN_UNSPEC     3
  int          pres_ex;
  int          pres_so;
#define PRES_IGNORED     0
#define PRES_OPTIONAL    1
#define PRES_SEMIMANDA   2
#define PRES_MANDATORY   3
  int          found;
  int          (*func)(struct _tag_CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi);
} CHECK_DT_SECT;

static int showSOName(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi);
static int checkNeeded(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi);
static int checkRelSections(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi);

/* NOTE: The order in this table is important because */
/* some sections depend on the existence of other!    */
static CHECK_DT_SECT sections[] = {
  { DT_STRTAB,            D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_STRSZ,             D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_SONAME,            D_UN_VAL,     PRES_IGNORED,   PRES_OPTIONAL,  0, showSOName },
  { DT_SYMTAB,            D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_SYMENT,            D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_NEEDED,            D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkNeeded },
  { DT_RELA,              D_UN_PTR,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELASZ,            D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELAENT,           D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_REL,               D_UN_PTR,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELSZ,             D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELENT,            D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_PLTREL,            D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkRelSections },
  { DT_PLTRELSZ,          D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkRelSections },
  { DT_JMPREL,            D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkRelSections },
  { DT_PLTGOT,            D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_HASH,              D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_INIT,              D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_FINI,              D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_RPATH,             D_UN_VAL,     PRES_OPTIONAL,  PRES_IGNORED,   0, NULL },
  { DT_SYMBOLIC,          D_UN_IGNORED, PRES_IGNORED,   PRES_OPTIONAL,  0, NULL },
  { DT_DEBUG,             D_UN_PTR,     PRES_OPTIONAL,  PRES_IGNORED,   0, NULL },
  { DT_TEXTREL,           D_UN_IGNORED, PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_RLD_VERSION,  D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_MIPS_FLAGS,        D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_MIPS_BASE_ADDRESS, D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_MIPS_LOCAL_GOTNO,  D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_MIPS_SYMTABNO,     D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_MIPS_GOTSYM,       D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_MIPS_RLD_MAP,      D_UN_VAL,     PRES_MANDATORY, PRES_IGNORED,   0, NULL },
  { DT_MIPS_TIME_STAMP,   D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_ICHECKSUM,    D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_IVERSION,     D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_CONFLICT,     D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_LIBLIST,      D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_CONFLICTNO,   D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_LIBLISTNO,    D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_UNREFEXTNO,   D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_MIPS_HIPAGENO,     D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_NULL,              D_UN_IGNORED, PRES_MANDATORY, PRES_MANDATORY, 1, NULL }
};

static char *getSectionName(ElfW(Sword) d_tag)
{
  DT_NAMES *act = dt_names;
  for(act = dt_names; act->name; act++)
    if(act->d_tag == d_tag)
      return act->name;

  return "-unknown-";
}

static ElfW(Dyn) *searchDynSection(ElfW(Obj_Info) *oi, ElfW(Sword) d_tag)
{
  ElfW(Dyn) *dyn;

  if(d_tag == DT_NULL)
    return NULL;

  for(dyn = _pdi_irix_getFirstDynEntry(oi); dyn->d_tag != DT_NULL; dyn++)
    if(dyn->d_tag == d_tag)
      return dyn;

  return NULL;
}

static ElfW(Dyn) *nextDynSection(ElfW(Dyn) *dyn, ElfW(Sword) d_tag)
{
  if(d_tag == DT_NULL) return NULL;
  if(dyn->d_tag != DT_NULL) dyn++;

  for(; dyn->d_tag != DT_NULL; dyn++)
    if(dyn->d_tag == d_tag)
      return dyn;

  return NULL;
}

static CHECK_DT_SECT *getSection(ElfW(Sword) d_tag)
{
  CHECK_DT_SECT *sect;
  for(sect = sections; sect->d_tag != DT_NULL; sect++)
    if(sect->d_tag == d_tag)
      return sect;
  return NULL;
}

/* This function requires that DT_STRTAB has been checked */
static int checkNeeded(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi)
{
  ElfW(Dyn) *dyn;
  char *strtab;

  if(sect->found)
  {
    strtab = (char *) searchDynSection(oi, DT_STRTAB)->d_un.d_val;
    _pdi_debug(__FILE__, NULL, "    + Library dependencies (DT_NEEDED):");

    for(dyn = searchDynSection(oi, sect->d_tag);
        dyn;
        dyn = nextDynSection(dyn, sect->d_tag))
    {
      _pdi_debug(__FILE__, NULL, "      - '%s'",
                 strtab + dyn->d_un.d_val);
    }
  }

  return 0;
}

static int showSOName(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi)
{
  if(!sect->found)
    return 0;

  _pdi_debug(__FILE__, NULL, "    + DT_SONAME = '%s'",
             (char *) (
                 searchDynSection(oi, DT_STRTAB)->d_un.d_ptr
               + searchDynSection(oi, sect->d_tag)->d_un.d_ptr)
             );

  return 0;
}

static int checkRelSections(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi)
{
  ElfW(Dyn) *dyn;

  dyn = searchDynSection(oi, sect->d_tag);
  if(sect->found)
    switch(sect->d_tag)
    {
      case DT_JMPREL:
        _pdi_debug(__FILE__, NULL, "    + Found section DT_JMPREL.");
        _pdi_debug(__FILE__, NULL, "      - d_ptr = 0x%lx", dyn->d_un.d_ptr);
        _pdi_error(__FILE__, NULL, "      - ON MIPS SECTION DT_JMPREL DOES NOT EXISTS!");
        /* Check required sections */
        if(!getSection(DT_PLTREL)->found)
        {
          _pdi_error(__FILE__, NULL, "      - DT_JMPREL needs DT_PLTREL presence!");
          return -1;
        }
        if(!getSection(DT_PLTRELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - DT_JMPREL needs DT_PLTRELSZ presence!");
          return -1;
        }
        return -1;
      case DT_PLTREL:
        _pdi_debug(__FILE__, NULL,
                   "    + Found DT_PLTREL. DT_JMPREL uses structures of type '%s'",
                   dyn->d_un.d_val == DT_REL ? "DT_REL" : "DT_RELA");
        if(!getSection(DT_PLTRELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_PLTREL needs DT_PLTRELSZ presence!");
          return -1;
        }
        break;
      case DT_RELA:
        _pdi_debug(__FILE__, NULL, "    + Found DT_RELA. (0x%lx)", dyn->d_un.d_ptr);
        if(!getSection(DT_RELASZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - DT_RELA needs DT_RELASZ presence!");
          return -1;
        }
        if(!getSection(DT_RELAENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_RELA needs DT_RELAENT presence!");
          return -1;
        }
        break;
      case DT_REL:
        _pdi_debug(__FILE__, NULL, "    + Found DT_REL. (0x%lx)", dyn->d_un.d_ptr);
        if(!getSection(DT_RELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - DT_REL needs DT_RELSZ presence!");
          return -1;
        }
        if(!getSection(DT_RELENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - DT_REL needs DT_RELENT presence!");
          return -1;
        }
        break;
      case DT_RELASZ:
      case DT_RELAENT:
      case DT_RELSZ:
      case DT_RELENT:
      case DT_PLTRELSZ:
        break;
      default:
        _pdi_error(THIS, "Unknown section (%s) %d.",
                   getSectionName(sect->d_tag), sect->d_tag);
        abort();
    }
  else
    switch(sect->d_tag)
    {
      case DT_RELA:
        if(getSection(DT_REL)->found)
          _pdi_debug(__FILE__, NULL, "      - Not found DT_RELA, but DT_REL exists. All ok.");
        else {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELA nor DT_REL.");
          return -1;
        }
        break;
      case DT_RELASZ:
        if(!getSection(DT_RELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELASZ nor DT_RELSZ.");
          return -1;
        }
        break;
      case DT_RELAENT:
        if(!getSection(DT_RELENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELAENT nor DT_RELENT.");
          return -1;
        }
        break;
      case DT_REL:
        if(getSection(DT_RELA)->found)
          _pdi_debug(__FILE__, NULL, "      - Not found DT_REL, but DT_RELA exists. All ok.");
        else {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELA nor DT_REL.");
          return -1;
        }
        break;
      case DT_RELSZ:
        if(!getSection(DT_RELASZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELASZ nor DT_RELSZ.");
          return -1;
        }
        break;
      case DT_RELENT:
        if(!getSection(DT_RELAENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELAENT nor DT_RELENT.");
          return -1;
        }
        break;
      case DT_JMPREL:
        _pdi_warning(__FILE__, NULL, "      - WARNING: Not found DT_JMPREL.");
        break;
      case DT_PLTREL:
      case DT_PLTRELSZ:
        break;
      default:
        _pdi_error(THIS, "Unknown section (%s) %d.",
                   getSectionName(sect->d_tag), sect->d_tag);
        abort();
    }

  return 0;
}

static void showSectionValues(CHECK_DT_SECT *sect, ElfW(Obj_Info) *oi)
{
  ElfW(Dyn) *dyn;

  if(sect->found > 0)
  {
    _pdi_debug(__FILE__, NULL,
               sect->found > 1
                 ? "    + Section '%s' (0x%x), %d times."
                 : "    + Section '%s' (0x%x).",
               getSectionName(sect->d_tag), sect->d_tag,
               sect->found);
    if(sect->d_un != D_UN_IGNORED)
    {
      for(dyn = searchDynSection(oi, sect->d_tag);
          dyn != NULL;
          dyn = nextDynSection(dyn, sect->d_tag))
        _pdi_debug(__FILE__, NULL, "      - %s = 0x%lx",
                   sect->d_un == D_UN_VAL ? "d_val" : "d_ptr",
                   sect->d_un == D_UN_VAL ? dyn->d_un.d_val : dyn->d_un.d_ptr);
    }
  }
}

static void resetSections(void)
{
  int i;

  for(i = 0; sections[i].d_tag != DT_NULL; i++)
    sections[i].found = 0;
}

static void countSections(ElfW(Obj_Info) *oi)
{
  ElfW(Dyn) *dyn;
  int i;

  for(dyn = _pdi_irix_getFirstDynEntry(oi); dyn->d_tag != DT_NULL; dyn++)
  {
    for(i = 0; sections[i].d_tag != DT_NULL && sections[i].d_tag != dyn->d_tag; i++)
      ;
    if(sections[i].d_tag == dyn->d_tag)
      sections[i].found ++;
    else
      _pdi_debug(__FILE__, NULL,
                 "    + Unknown section '%s' (0x%x).",
                 getSectionName(dyn->d_tag),
                 dyn->d_tag);
  }
}

static void checkSections(ElfW(Obj_Info) *oi, int shared_object)
{
  int i, presence;

  for(i = 0; sections[i].d_tag != DT_NULL; i++)
  {
    presence = shared_object
                 ? sections[i].pres_so
                 : sections[i].pres_ex;

    if(presence == PRES_IGNORED)
      continue;
    if(presence == PRES_MANDATORY && sections[i].found == 0)
    {
      _pdi_debug(__FILE__, NULL,
                 "    + Section '%s' (0x%x) is obligatory and it is NOT present.",
                 getSectionName(sections[i].d_tag),
                 sections[i].d_tag);
      exit(1);
    }
    if(presence == PRES_SEMIMANDA && sections[i].found == 0)
      _pdi_debug(__FILE__, NULL,
                 "    + Section '%s' (0x%x) is obligatory and SHOULD BE present.",
                 getSectionName(sections[i].d_tag),
                 sections[i].d_tag);

    if(sections[i].func != NULL)
    {
      if(sections[i].func(&(sections[i]), oi))
      {
        ElfW(Dyn) *dyn;
        _pdi_debug(__FILE__, NULL,
                   "  ---> ELF STRUCTURE OF THIS DSO DOES NOT SEEM TO BE CORRECT!");
        _pdi_debug(__FILE__, NULL, "      + Quick list of present sections:");
        for(dyn = _pdi_irix_getFirstDynEntry(oi); dyn->d_tag != DT_NULL; dyn++)
          _pdi_debug(__FILE__, NULL, "           %s (0x%x)",
                     getSectionName(dyn->d_tag), dyn->d_tag);
        _pdi_debug(__FILE__, NULL, "           [end of list]");
        return;
      }
    } else
      showSectionValues(&(sections[i]), oi);
  }
}

void _pdi_irix_debug_checkSections(void)
{
  ElfW(Obj_Info) *oi;
  int shared_object;

  /* the first element of the list is the executable binary */
  shared_object = 0;

  _pdi_debug(__FILE__, NULL, "Check ELF structures in memory:");
  for(oi = __rld_obj_head; oi; oi = (ElfW(Obj_Info) *) oi->oi_next)
  {
    _pdi_debug(__FILE__, NULL,
               "  # Checking %s '%s' (ehdr=0x%lx, delta=0x%lx).",
               shared_object ? "object" : "executable binary",
               oi->oi_pathname,
               oi->oi_ehdr,
               oi->oi_ehdr - oi->oi_orig_ehdr);
    resetSections();
    countSections(oi);
    checkSections(oi, shared_object);

    shared_object = 1;
  }
}

