/*****************************************************************************
 * $Id: ix-alterplt.c,v 1.2 2005/08/25 13:06:52 killabyte Exp $
 *
 * This file implements a mechanism used to alter easily the content of the
 * GOT/PLT tables.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>
#include<irix/elfops.h>

/*****************************************************************************
 * int _pdi_irix_alterPlt(PDI_ELFOBJ *eo, unsigned int symndx, void *target,
 *                        PDI_IRIX_PLT_UNDO_DATA *undo_data)
 *
 * Description:
 *   Modify the PLT/GOT entry selected by 'symndx' in the object 'eo',
 *   substituting the current pointer by 'target', and saving to 'undo_data'
 *   (if it is not NULL) the current data.
 *
 * Parameters:
 *   eo        - target object
 *   symndx    - index in the symbol table of the target function
 *   target    - a pointer to the new definition
 *   undo_data - A pointer to a buffer where save the original PLT/GOT data. If
 *               it is NULL the current PLT/GOT info is discarded.
 *
 * Returns:
 *   0 if all ok, other value otherwise.
 *
 *****************************************************************************/

int _pdi_irix_alterPlt(PDI_ELFOBJ *eo, unsigned int symndx, void *target,
                       PDI_IRIX_PLT_UNDO_DATA *undo_data)
{
  ElfW(Addr) got_address;
  unsigned int got_index;

  /***************************************************************************
   * MIPS 32/64 Bits
   ***************************************************************************/
#if PLATFORM_ASM == PDI_ASM_MIPS
  /* Check that index 'symndx' is valid in GOT */
  if(symndx < eo->o_arch->gotSym)
  {
    _pdi_error(THIS,
               "Symbol %s::%s() (index %d) has no correspondence in GOT.",
               eo->o_name,
               (char *) (eo->o_arch->stringTable
                        + ((ElfW(Sym) *) (eo->o_arch->symbolTable + symndx * eo->o_arch->symbolSize))->st_name),
               symndx);
    return -1;
  }

  /* Compute symbol's index in GOT */
  got_index = eo->o_arch->localGotElems + symndx - eo->o_arch->gotSym;

  /* Calculate symbol's SLOT address in GOT */
  got_address = eo->o_arch->gotTable + (got_index * sizeof(ElfW(Addr)));

  /* Save undo data */
  if(undo_data)
  {
    undo_data->slot_addr = (void **) got_address;
    undo_data->addr = *((void **) got_address);
  }

  /* Install interposition */
  *((void **) got_address) = target;

  /***************************************************************************
   * Unknowm platforms
   ***************************************************************************/
#else
#error "I don't know how to install a interposition on this platform"
#endif

  return 0;
}

/*****************************************************************************
 * void _pdi_irix_restorePlt(PDI_IRIX_PLT_UNDO_DATA *undo_data)
 *
 * Description:
 *   Restore a GOT slot with info stored in 'undo_data'.
 *
 * Parameters:
 *   undo_data - Undo data :)
 *
 * Returns:
 *   -nothing-
 *
 *****************************************************************************/

void _pdi_irix_restorePlt(PDI_IRIX_PLT_UNDO_DATA *undo_data)
{
  /***************************************************************************
   * MIPS 32/64 Bits
   ***************************************************************************/
#if PLATFORM_ASM == PDI_ASM_MIPS
  /* Restore GOT contents */
  *(undo_data->slot_addr) = undo_data->addr;

  /***************************************************************************
   * Unknown platforms
   ***************************************************************************/
#else
#error "I don't know how to uninstall a interposition on this platform"
#endif

  /* Set undo data to zero */
  memset(undo_data, 0, sizeof(PDI_IRIX_PLT_UNDO_DATA));
}

