/*****************************************************************************
 * $Id: files.c,v 1.1 2005/08/02 10:41:03 killabyte Exp $
 *
 * Functions used to work with file paths and to search files in search path
 * lists. With help of this functions is simplified file searching and absolute
 * path resolution.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<log.h>

/*****************************************************************************
 * static int getUserHome(char *user, char *out)
 *
 * Descripcin:
 *   Busca al usuario 'user' en el sistema y devuelve el path a su directorio
 *   home.
 *
 * Parmetros:
 *   user - username
 *   out  - cadena donde copiaremos el home del usuario. Su longitud no puede
 *          ser inferior a PATH_MAX carcteres.
 *
 * Devuelve:
 *   0 en caso de que no ocurriese ningn error. -1 en caso de error.
 *   Si el usuario existe se copia su directorio home a 'out', en caso de no
 *   existir se devuelve error.
 *
 *****************************************************************************/

static int getUserHome(char *user, char *out)
{
  char *s;
  int c, i, ret;
  FILE *f;

  /* Si no se ha especificado un usuario intentamos resolver la 'home'   */
  /* mediante la variable de entorno HOME. Si esta no existe, intentamos */
  /* resolver el nombre de usuario mediante la variable de entorno USER  */
  /* y seguimos con el procedimiento normal...                           */
  if(!user || !strlen(user))
  {
    if((s = getenv("HOME")) != NULL)
    {
      if(strlen(s) == 0)
        s = "/";
      strncpy(out, s, PATH_MAX);
      return 0;
    }

    if((user = getenv("USER")) == NULL)
    {
      _pdi_error(THIS, "Environment variable USER is not defined.");
      return -1;
    }
  }

  /* Abrimos el fichero de passwords para lectura... */
  if((f = fopen("/etc/passwd", "r")) == NULL)
  {
    _pdi_error(THIS, "Cannot open '/etc/passwd' file.");
    return -1;
  }

  /* Buscamos el usuario 'user', y si lo encontramos nos copiamos su 'home' */
  ret = -1;
  c = 0;
  while(ret && c != EOF)
  {
    /* Leemos el nombre de usuario y comparamos */
    for(i = 0, s = user;
        (c = fgetc(f)) != EOF && c != '\n' && c != ':' && *s && c == *s;
        i++, s++)
      ;

    /* Si en ambos lados hemos llegado al final de cadena ... */
    if(c == ':' && *s == '\0')
    {
      /* lo hemos encontrado... recogemos el home */
      /* --> primero nos saltamos los otros parmetros feos */
      for(i = 0; c != EOF && c != '\n' && i < 4; i++)
        while((c = fgetc(f)) != EOF && c != ':' && c != '\n')
          ;
      if(c != EOF)
      {
        /* --> leemos el home */
        for(i = 0;
            i < PATH_MAX-1 && (c = fgetc(f)) != EOF && c != ':' && c != '\n';
            i++)
          out[i] = c;
        out[i] = '\0';
        /* --> copiamos el home */
        if(strlen(out) == 0)
          strcpy(out, "/");
        ret = 0;
      }
    }

    /* debemos saltarnos el resto de la lnea */
    while(c != '\n' && c != EOF)
      c = fgetc(f);
  }

  fclose(f);

  return ret;
}

static int canonicalizepath(char *src, char *out)
{
  int i, j, k;
  int fin, ndir;
  char **p;
  char tmp[PATH_MAX];

  /* Inicializamos 'tmp' con el directorio actual si 'src' */
  /* no es un path absoluto, o una cadena vacia si lo es.  */
  /* Acto seguido le aadimos el directorio de 'src'       */
  /* Y por ltimo le quitamos la barra del final (si hay)  */
  if(*src != '/')
  {
    if(!getcwd(tmp, PATH_MAX))
    {
      _pdi_error(THIS, "getcwd() failed.");
      return -1;
    }
    strcat(tmp, "/");
  } else
    *tmp = '\0';
  strcat(tmp, src);
  if(tmp[strlen(tmp)-1] == '/')
    tmp[strlen(tmp)-1] = '\0';

  /* Eliminamos todos las barras repetidas (//+) */
  for(i = 0; tmp[i]; i++)
    if(tmp[i] == '/' && tmp[i+1] == '/')
    {
      for(j = i+1; tmp[j] && tmp[j] == '/'; j++)
        ;
      for(k = i+1; tmp[j]; k++, j++)
        tmp[k] = tmp[j];
      tmp[k] = '\0';
    }

  /* Eliminamos todos los directorios actuales (/./) */
  for(i = 0; tmp[i]; )
    if(tmp[i] == '/' && tmp[i+1] == '.' && tmp[i+2] == '/')
    {
      for(j = i; tmp[j+2]; j++)
        tmp[j] = tmp[j+2];
      tmp[j] = '\0';
    } else
      i++;

  /* Contamos cuantas componentes hay */
  ndir = 0;
  for(i = 0; tmp[i]; i++)
    if(tmp[i] == '/')
      ndir++;

  if(ndir <= 1)
  {
    strcpy(out, tmp);
    return 0;
  }

  /* Creamos una tabla de punteros en p */
  if((p = malloc(sizeof(char *) * (ndir+1))) == NULL)
  {
    _pdi_error(THIS, "No memory.");
    return -1;
  }
  j = 0;
  for(i = 0; tmp[i]; i++)
    if(tmp[i] == '/')
    {
      p[j++] = tmp + i + 1;
      p[j] = NULL;
      tmp[i] = '\0';
    }

  /* Sustitumos las cadenas /dir/../ por / */
  fin = 0;
  while(!fin)
  {
    fin = -1;
    for(i = 0; fin && p[i]; i++)
      if(!strcmp("..", p[i]))
      {
        fin = 0;
        if(i == 0)
          for(i = 0; p[i]; i++)
            p[i] = p[i+1];
        else {
          for( ; p[i+1]; i++)
            p[i-1] = p[i+1];
          p[i-1] = NULL;
        }
      }
  }

  /* Construmos el path */
  *out = '\0';
  for(i = 0; p[i]; i++)
  {
    strcat(out, "/");
    strcat(out, p[i]);
  }

  free(p);

  return 0;
}

/*****************************************************************************
 * int _pdi_getCanonicalPath(char *src_path, char *out, int real)
 *
 * Descripcin:
 *   Coge una ruta 'src_path' y copia su ruta cannica equivalente en 'out'.
 *
 * Parmetros:
 *   src_path - ruta a un fichero/directorio
 *   out      - cadena donde copiaremos la ruta absoluta. Su longitud no puede
 *              ser inferior a PATH_MAX carcteres.
 *   real     - si se activa sigue los enlaces simblicos
 *
 * Devuelve:
 *   0 en caso de que no ocurriese ningn error. -1 en caso de error.
 *
 *****************************************************************************/

int _pdi_getCanonicalPath(char *src_path, char *out, int real)
{
  char scratch[PATH_MAX];
  int i;

  /* Si indican una home de usuario intentamos resolverla...  */
  /* (dejamos en 'scratch' la ruta al directorio del usuario) */
  if(src_path[0] == '~')
  {
    char username[PATH_MAX];

    for(i = 0; src_path[i+1] && src_path[i+1] != '/'; i++)
      username[i] = src_path[i+1];
    username[i++] = '\0';
    if(!getUserHome(username, scratch) && strlen(scratch) > 0)
    {
      if(scratch[strlen(scratch)-1] != '/')
        strcat(scratch, "/");
      src_path += i;
      if(*src_path == '/')
        src_path++;
    } else
      scratch[0] = '\0';
  } else
    scratch[0] = '\0';

  /* Aadimos el resto del path a 'scratch' */
  strcat(scratch, src_path);

  /* Resolvemos el path */
  if(real)
  {
    if(!realpath(scratch, out))
      return -1;
  } else {
    if(canonicalizepath(scratch, out))
      return -1;
  }
  
  return 0;
}

/*****************************************************************************
 * int _pdi_fileIsRegular(char *fname)
 *
 * Descripcin:
 *   Comprueba si el fichero 'fname' es un fichero regular.
 *
 * Parmetros:
 *   fname - fichero a comprobar
 *
 * Devuelve:
 *   -1 en caso de que el fichero exista y sea regular, 0 en caso contrario.
 *
 *****************************************************************************/

int _pdi_fileIsRegular(char *fname)
{
  struct stat buf;

  return (!stat(fname, &buf) && S_ISREG(buf.st_mode));
}

/*****************************************************************************
 * int _pdi_findFile(char **paths, char *file, char *out, int real)
 *
 * Descripcin:
 *   Coge una ruta 'src_path' y copia su ruta absoluta equivalente en 'out'.
 *
 * Parmetros:
 *   paths - lista de directorios
 *   file  - ruta al fichero a buscar
 *   out   - cadena donde copiaremos la ruta cannica al fichero. Su longitud
 *           no puede ser inferior a PATH_MAX carcteres.
 *   real  - resuelve enlaces simblicos
 *
 * Devuelve:
 *   0 en caso de que no ocurriese ningn error. -1 en caso de error.
 *
 *****************************************************************************/

int _pdi_findFile(char **paths, char *file, char *out, int real)
{
  char tmp[PATH_MAX];

  if(*file != '/' && *file != '~')
  {
    for(; *paths; paths++)
    {
      if(strlen(*paths) > 0)
      {
        if(snprintf(tmp, PATH_MAX,"%s%s%s",
                    *paths, (*paths)[strlen(*paths)-1] != '/' ? "/" : "", file) > PATH_MAX)
        {
          _pdi_error(THIS, "String '%s/%s' is longer than PATH_MAX characters (%d).",
                     *paths, file, PATH_MAX);
          return -1;
        }
      } else
        strcpy(tmp, file);
               
      if(!_pdi_getCanonicalPath(tmp, out, real)
      && _pdi_fileIsRegular(out))
        return 0;
    }
  } else
    if(!_pdi_getCanonicalPath(file, out, real)
    && _pdi_fileIsRegular(out))
      return 0;

  *out = '\0';
  return -1;
}

