/*****************************************************************************
 * $Id: begest.c,v 1.2 2005/10/11 18:32:11 killabyte Exp $
 *
 * This file contains functions that will be used to handle backends and access
 * to their methods (backend entry points as initialization, finalization,
 * etc).
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *****************************************************************************/

#include<config.h>
#include<pdiconfig.h>
#include<ebeif.h>
#include<log.h>

/*****************************************************************************
 * static void finalizeBackend(PDI_ELFOBJ *backend)
 *
 * Descripcin:
 *   Ejecuta el cdigo de finalizacin de un backend (si se debe ejecutar).
 *
 * Parmetros:
 *   backend - backend a finalizar.
 *
 * Devuelve:
 *   Nada.
 *
 *****************************************************************************/

static void finalizeBackend(PDI_ELFOBJ *backend)
{
  if(!backend->o_beFini)
    return;

  if(backend->o_beInitialized)
    backend->o_beFini();
  else
    _pdi_warning(_pdi_ebe_getObjectName(backend), NULL, "Finalization code "
                 "is not executed because initialization failed.");
}

/*****************************************************************************
 * static int closeBackend(PDI_ELFOBJ *backend)
 *
 * Descripcin:
 *   Cierra el backend 'backend' (lo quita de memoria)
 *
 * Parmetros:
 *   backend - backend a cerrar
 *
 * Devuelve:
 *   0 si hubo xito, otro valor en caso de error.
 *
 *****************************************************************************/

static int closeBackend(PDI_ELFOBJ *backend)
{
  if(backend->o_dl_handler && dlclose(backend->o_dl_handler))
  {
    _pdi_error(THIS, "Error at backend '%s' unloading:%s",
               _pdi_ebe_getObjectName(backend), dlerror());
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * PDI_ELFOBJ *_pdi_ebe_loadBackend(char *path)
 *
 * Descripcin:
 *   Permite cargar e inicializar un backend a partir de la ruta a su binario.
 *
 * Parmetros:
 *   path - ruta al binario con el backend.
 *
 * Devuelve:
 *   Un puntero al PDI_ELFOBJ que representa a este backend si ha habido xito
 *   en la carga. En caso contrario la funcin devuelve NULL.
 *
 *****************************************************************************/

PDI_ELFOBJ *_pdi_ebe_loadBackend(char *path)
{
  void *handler;
  PDI_ELFOBJ *eo;

  /* Comprobamos que no est cargado en memoria */
#warning Ojo: que pasa si el backend estaba preloaded?
  if(_pdi_ebe_searchObjectByPath(path) != NULL)
  {
    _pdi_error(THIS, "Backend '%s' was previously loaded in memory.", path);
    return NULL;
  }

  /* Intentamos cargar el backend en memoria */
  if((handler = dlopen(path, RTLD_LAZY)) == NULL)
  {
    _pdi_error(THIS, "Cannot open backend '%s': %s.", path, dlerror());
    return NULL;
  }

  /* Actualizamos la lista de objetos */
  if(_pdi_arch_refreshObjectList())
  {
    _pdi_error(THIS, "Cannot refresh object list.");
    return NULL;
  }

  /* Buscamos el nuevo objeto */
  if((eo = _pdi_ebe_searchObjectByPath(path)) == NULL)
  {
    _pdi_error(THIS, "Backend '%s' was not loaded in memory.", path);
    return NULL;
  }

  /* Marcamos este cacharro como backend */
  eo->o_isBackend     = -1;
  eo->o_beInitialized = 0;
  eo->o_dl_handler    = handler;

  /* Buscamos sus smbolos y los cacheamos */
  eo->o_beInit              = dlsym(eo->o_dl_handler, PDI_STR_BE_FUNC_INIT);
  eo->o_beFini              = dlsym(eo->o_dl_handler, PDI_STR_BE_FUNC_FINI);
  eo->o_bePreEventCallback  = dlsym(eo->o_dl_handler, PDI_STR_BE_FUNC_PRE_CB);
  eo->o_bePostEventCallback = dlsym(eo->o_dl_handler, PDI_STR_BE_FUNC_POST_CB);
  eo->o_beCallbackRequired  = dlsym(eo->o_dl_handler, PDI_STR_BE_FUNC_CB_REQ);

  /* Resolve symbols on this backend so it will not be affected by */
  /* any interposition (if 'donttouch_backends' option is enabled) */
  if(PDICFG.donttouch_backends)
    if(_pdi_ebe_protectBackend(eo))
    {
      _pdi_error(THIS, "Error while protecting backend '%s'.", path);
      return NULL;
    }

  /* Tratamos de inicializar el backend */
  if(eo->o_beInit)
    eo->o_beInitialized = eo->o_beInit();
  else
    eo->o_beInitialized = -1;

  return eo;
}

/*****************************************************************************
 * void *_pdi_ebe_getBackendSymbol(PDI_ELFOBJ *backend, char *symbol)
 *
 * Descripcin:
 *   Obtiene un puntero a un smbolo 'symbol' del backend 'backend'.
 *
 * Parmetros:
 *   backend - Estructura PDI_ELFOBJ con informacin sobre el backend
 *   symbol  - smbolo que deseamos encontrar en el backend
 *
 * Devuelve:
 *   Si todo fue bien, y el smbolo existe en el backend, se devuelve un
 *   puntero al mismo, en caso contrario se devuelve NULL.
 *
 *****************************************************************************/

void *_pdi_ebe_getBackendSymbol(PDI_ELFOBJ *backend, char *symbol)
{
  if(!backend->o_isBackend)
  {
    _pdi_error(THIS, "Object '%s' is not a backend.",
               _pdi_ebe_getObjectName(backend));
    return NULL;
  }

  if(!symbol)
    return NULL;

  return dlsym(backend->o_dl_handler, symbol);
}

/*****************************************************************************
 * int _pdi_ebe_unloadBackend(PDI_ELFOBJ *be)
 *
 * Descripcin:
 *   Descarga y finaliza el backend 'be'.
 *
 * Parmetros:
 *   be - backend a descargar.
 *
 * Devuelve:
 *   0 si todo fue bien, otro valor en caso de error.
 *
 *****************************************************************************/

int _pdi_ebe_unloadBackend(PDI_ELFOBJ *be)
{
  if(!be->o_isBackend)
  {
    _pdi_error(THIS, "Object '%s' is not a backend.",
               _pdi_ebe_getObjectName(be));
    return -1;
  }

  /* primero de todo quitamos las interposiciones que van hacia este backend */
  /* y (si las hay) quitamos las interposiciones instaladas sobre el backend */
  if(_pdi_ebe_uninstallBackendInterpositions(be)
  || _pdi_ebe_uninstallInterpositions(be))
  {
    _pdi_error(THIS, "An error happened while uninstalling interpositions "
               "related to '%s'.\n", _pdi_ebe_getObjectName(be));
    return -1;
  }

  /* Llamamos al cdigo de terminacin del backend (si existe) */
  finalizeBackend(be);

  /* Descargamos el objeto de memoria */
  if(closeBackend(be))
  {
    _pdi_error(THIS, "Cannot close backend '%s'", _pdi_ebe_getObjectName(be));
    return -1;
  }

  /* Actualizamos la lista de objetos */
  if(_pdi_arch_refreshObjectList())
  {
    _pdi_error(THIS, "Cannot refresh object list.");
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * void _pdi_ebe_finalizeAllBackends(void)
 *
 * Descripcin:
 *   Ejecuta el cdigo de finalizacin de todos los backends en memoria.
 *
 * Parmetros:
 *   ninguno
 *
 * Devuelve:
 *   nada
 *
 *****************************************************************************/

void _pdi_ebe_finalizeAllBackends(void)
{
  PDI_ELFOBJ *eo;

  for(eo = _pdi_objlist; eo; eo = eo->o_next)
    finalizeBackend(eo);
}

/*****************************************************************************
 * int _pdi_ebe_unloadAllBackends(void)
 *
 * Descripcin:
 *   Descarga de memoria todos los backends. PERO OJO! No ejecuta el cdigo
 *   de finalizacin!
 *
 * Parmetros:
 *   ninguno
 *
 * Devuelve:
 *   0 si todo fue bien, otro valor en caso de error.
 *
 *****************************************************************************/

int _pdi_ebe_unloadAllBackends(void)
{
  PDI_ELFOBJ *eo;

  for(eo = _pdi_objlist; eo; eo = eo->o_next)
    if(eo->o_isBackend && closeBackend(eo))
    {
      _pdi_error(THIS, "Cannot close backed '%s'", _pdi_ebe_getObjectName(eo));
      return -1;
    }

  return 0;
}

