/*****************************************************************************
 * $Id: beconfig.c,v 1.2 2005/09/01 21:31:11 killabyte Exp $
 *
 * This object gives a set of tools to use structures that will simplify
 * loading backends and the installation of the interposition commands. These
 * structures specify which objects and functions will be affected by
 * interpositions and which objects (backends and functions) will be
 * interposed.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>
#include<beconfig.h>
#include<pdiconfig.h>
#include<files.h>

/*****************************************************************************
 * BECFG_OBJ_INFO *_pdi_becfg_getObjectByPath(BECFG_CONFIG *cfg, char *path)
 *
 * Description:
 *   Search an object by its path in the object list of configuration 'cfg'.
 *
 * Parameters:
 *   cfg  - configuration
 *   path - path to the searched object
 *
 * Returns:
 *   This function returns a pointer to a BECFG_OBJ_INFO structure if exists a
 *   DSO with path 'path', otherwise NULL is returned.
 *
 *****************************************************************************/

BECFG_OBJ_INFO *_pdi_becfg_getObjectByPath(BECFG_CONFIG *cfg, char *path)
{
  BECFG_OBJ_INFO *oi;

  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
    if(oi->co_path && !strcmp(oi->co_path, path))
      return oi;

  return NULL;
}

/*****************************************************************************
 * BECFG_OBJ_INFO *_pdi_becfg_getObjectByAlias(BECFG_CONFIG *cfg, char *alias)
 *
 * Description:
 *   Search an object by its alias in the object list of configuration 'cfg'.
 *
 * Parameters:
 *   cfg  - configuration
 *   path - alias of the searched object
 *
 * Returns:
 *   This function returns a pointer to a BECFG_OBJ_INFO structure if exists a
 *   DSO with alias 'alias', otherwise NULL is returned.
 *
 *****************************************************************************/

BECFG_OBJ_INFO *_pdi_becfg_getObjectByAlias(BECFG_CONFIG *cfg, char *alias)
{
  BECFG_OBJ_INFO *oi;

  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
    if(oi->co_alias && !strcmp(oi->co_alias, alias))
      return oi;

  return NULL;
}

/*****************************************************************************
 * BECFG_OBJ_INFO *_pdi_becfg_getObject(BECFG_CONFIG *cfg, char *x)
 *
 * Description:
 *   Search an object by its alias/path 'x' in the object list contained in
 *   config 'cfg'. This function tries to search an object with an alias 'x',
 *   and if it doesn't exist then retries searching by path.
 *
 * Parameters:
 *   cfg  - configuration
 *   path - alias or path of the searched object
 *
 * Returns:
 *   If the object exists then it returns a pointer to its BECFG_OBJ_INFO
 *   structure, or returns NULL if search failed.
 *
 *****************************************************************************/

BECFG_OBJ_INFO *_pdi_becfg_getObject(BECFG_CONFIG *cfg, char *x)
{
  BECFG_OBJ_INFO *o;

  if((o = _pdi_becfg_getObjectByAlias(cfg, x)) == NULL)
    o = _pdi_becfg_getObjectByPath(cfg, x);

  return o;
}

/*****************************************************************************
 * char *_pdi_becfg_getObjectName(BECFG_OBJ_INFO *object)
 *
 * Description:
 *   Returns the name of object 'object' (path or alias)
 *
 * Parameters:
 *   object - the object
 *
 * Returns:
 *   A pointer to a string with the name of this object.
 *
 *****************************************************************************/

char *_pdi_becfg_getObjectName(BECFG_OBJ_INFO *object)
{
  return object->co_path
           ? object->co_path
           : object->co_alias;
}

/*****************************************************************************
 * int _pdi_becfg_setObjectAlias(BECFG_OBJ_INFO *object, char *new_alias)
 *
 * Description:
 *   Change the alias of object 'object'.
 *
 *   NOTE: This function does not check if alias 'new_alias' exists or is equal
 *         to some object path.
 *
 * Parameters:
 *   object    - the object
 *   new_alias - new alias
 *
 * Returns:
 *   Returns 0 if succesful, -1 otherwise.
 *
 *****************************************************************************/

int _pdi_becfg_setObjectAlias(BECFG_OBJ_INFO *object, char *new_alias)
{
  char *old_alias;
  char *p = NULL;
 
  /* Copy the new alias */
  if(new_alias)
  {
    if((p = malloc(strlen(new_alias)+1)) == NULL)
    {
      _pdi_error(THIS, "No memory.");
      return -1;
    }
    strcpy(p, new_alias);
  }

  /* Change the alias */
  old_alias = object->co_alias;
  object->co_alias = p;

  /* Free memory used by the old alias */
  if(old_alias)
    free(old_alias);


  return 0;
}

/*****************************************************************************
 * static BECFG_OBJ_INFO *addObject(BECFG_CONFIG *cfg,
 *                                  char *o_path, char *o_alias,
 *                                  int is_backend)
 *
 * Description:
 *   Internal function of this module, used by '_pdi_becfg_addObject()', that
 *   allos memory and initializes a new BECFG_OBJ_INFO structure without making
 *   any check and following orders strictly.
 *
 * Parameters:
 *   cfg        - configuration being loaded
 *   o_path     - path to the object
 *   o_alias    - object alias
 *   is_backend - it should be different from zero if the object is a backend
 *
 * Returns:
 *   A pointer to the info of the new object if succesful, or NULL otherwise.
 *
 *****************************************************************************/

static BECFG_OBJ_INFO *addObject(BECFG_CONFIG *cfg,
                                 char *o_path, char *o_alias,
                                 int is_backend)
{
  BECFG_OBJ_INFO *oi;
  char canonical_path[PATH_MAX];
  
  /* alloc memory for the structure */
  if((oi = malloc(sizeof(BECFG_OBJ_INFO))) == NULL)
  {
    if(oi) free(oi);
    _pdi_error(THIS, "Cannot malloc memory for object '%s'.", o_path);
    return NULL;
  }

  /* Alloc memory for the canonized path and copy it */
  if(o_path)
  {
    if(!PDICFG.no_check_on_config)
    {
      if(_pdi_findFile(is_backend ? PDICFG.be_path : PDICFG.lib_path,
                       o_path, canonical_path, 0))
      {
        _pdi_error(THIS, "File '%s' not found.", o_path);
        return NULL;
      }
    } else
      canonical_path[0] = '\0';

    if(!strlen(canonical_path))
      strcpy(canonical_path, o_path);

    if((oi->co_path = malloc(strlen(canonical_path) + 1)) == NULL)
    {
      free(oi);
      _pdi_error(THIS, "Cannot malloc memory for object path '%s'.",
                 canonical_path);
      return NULL;
    }
    strcpy(oi->co_path, canonical_path);
  } else
    oi->co_path = NULL;

  if(o_alias)
  {
    if((oi->co_alias = malloc(strlen(o_alias) + 1)) == NULL)
    {
      if(oi->co_path) free(oi->co_path);
      free(oi);
      _pdi_error(THIS, "Cannot malloc memory for object alias '%s'.", o_path);
      return NULL;
    }
    strcpy(oi->co_alias, o_alias);
  } else
    oi->co_alias = NULL;

  /* Initialize the structure */
  oi->co_isBackend = is_backend;
  oi->co_next = NULL;
  oi->co_lineno = cfg->bc_lineno;

  /* Add the new object to the list of known objects */
  if(cfg->bc_objlist)
    cfg->bc_objlist_last = cfg->bc_objlist_last->co_next = oi;
  else
    cfg->bc_objlist = cfg->bc_objlist_last = oi;

  return oi;
}

/*****************************************************************************
 * BECFG_OBJ_INFO *_pdi_becfg_addObject(BECFG_CONFIG *cfg,
 *                                      char *o_path, char *o_alias,
 *                                      int is_backend)
 *
 * Description:
 *   Add a new object to the list of objects in a configuration.
 *
 * Parameters:
 *   cfg        - configuration being build
 *   o_path     - path to the object
 *   o_alias    - alias of object
 *   is_backend - different of zero if the object is a backend
 *
 * Returns:
 *   A pointer to the info of the new object if succesful, or NULL otherwise.
 *
 *****************************************************************************/

BECFG_OBJ_INFO *_pdi_becfg_addObject(BECFG_CONFIG *cfg,
                                     char *o_path, char *o_alias,
                                     int is_backend)
{
  BECFG_OBJ_INFO *oi;
  char canonical_path[PATH_MAX];
  
  /* If o_path is NULL then error! */
  if(!o_path)
  {
    _pdi_error(THIS, "Object path is NULL.");
    return NULL;
  }

  /* Canonicalize path */
  if(_pdi_getCanonicalPath(o_path, canonical_path, 0))
  {
    _pdi_error(THIS, "Error in getAbsolutePath().");
    strcpy(canonical_path, o_path);
  }

  /* If alias length is zero don't admit it */
  if(o_alias && strlen(o_alias) == 0)
    o_alias = NULL;

  /* Check 'o_alias' is not a reserved or used alias */
  if(o_alias)
  {
    /* Check it is not a reserved alias */
    if(!strcmp(o_alias, PDI_ALIAS_LIBC)
    || !strcmp(o_alias, PDI_ALIAS_MAIN)
    || !strcmp(o_alias, PDI_ALIAS_PDI))
    {
      _pdi_error(THIS, "Cannot assign reserved alias '%s' to object '%s'.",
                 o_alias, canonical_path);
      return NULL;
    }

    /* Check it is not used by another object nor equal to any object path */
    for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
    {
      if(oi->co_alias && !strcmp(oi->co_alias, o_alias))
      {
        _pdi_error(THIS, "Object '%s' is using alias '%s'.",
                   oi->co_path, o_alias);
        return NULL;
      }
      if(oi->co_path && !strcmp(oi->co_path, o_alias))
      {
        _pdi_error(THIS, "Alias '%s' matches with object path '%s'.",
                   o_alias, oi->co_path);
        return NULL;
      }
    }
  }

  /* Check this object it is not in list */
  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
    if(oi->co_path && !strcmp(oi->co_path, canonical_path))
    {
      _pdi_error(THIS, "Object '%s' was registered.", o_path);
      return NULL;
    }

  /* Create the new object and return it */
  return addObject(cfg, o_path, o_alias, is_backend);
}

/*****************************************************************************
 * int _pdi_becfg_addCommand(BECFG_CONFIG *cfg, int type,
 *                           char *trgObj, char *trgFunc,
 *                           char *backend, char *wrapper)
 *
 * Description:
 *   Adds a new interposition command to 'cfg' configuration.
 *
 * Parameters:
 *   cfg        - configuration
 *   type       - command type
 *                (BECFG_IT_RELINK, BECFG_IT_REDEFINITION or BECFG_IT_CALLBACK)
 *   trgObj     - object where will be installed thir interposition (or NULL if
 *                wildcard)
 *   trgFunc    - function affected by this interposition (or NULL for a
 *                wildcard)
 *   backend    - backend that will manage this interpostion
 *   wrapper    - interposed function (wrapper)
 *
 * Returns:
 *   Returns 0 if succesful, -1 otherwise.
 *
 *****************************************************************************/

int _pdi_becfg_addCommand(BECFG_CONFIG *cfg,
                          int type,
                          BECFG_OBJ_INFO *object, char *symbol,
                          BECFG_OBJ_INFO *backend, char *wrapper)
{
  BECFG_INTERCEPT *bi;
  
  if(type == BECFG_IT_CALLBACK)
  {
    _pdi_debug(THIS,
               "%s:Registering command 'C %s * %s%s%s'",
               cfg->bc_name,
               object ? _pdi_becfg_getObjectName(object) : "*",
               _pdi_becfg_getObjectName(backend),
               wrapper ? " " : "",
               wrapper ? wrapper : "");
    if(symbol)
    {
      _pdi_error(cfg->bc_name, NULL, "A callback cannot affect to only a symbol.");
      return -1;
    }
  } else {
    _pdi_debug(THIS,
               "%s:Registering command '%c %s %s %s %s'",
               cfg->bc_name,
               type == BECFG_IT_RELINK ? 'R' : 'D',
               object ? _pdi_becfg_getObjectName(object) : "*",
               symbol    ? symbol    : "*",
               _pdi_becfg_getObjectName(backend),
               wrapper);
    if(!symbol)
    {
      _pdi_error(cfg->bc_name, NULL, "%s cannot accept an undefined symbol.",
                 type == BECFG_IT_RELINK
                   ? "Relinks"
                   : "Redefinitions");
      return -1;
    }
  }

  /* If this interposition is a callback and it defines an alternative */
  /* generic wrapper check that configuration allows this              */
  if(type == BECFG_IT_CALLBACK && wrapper && !(PDICFG.cb_allow_handler))
  {
    _pdi_error(cfg->bc_name, NULL,
               "%d:Cannot use '%s()' as a generic wrapper if "
               "option 'cb_allow_handler' is disabled.",
               cfg->bc_lineno, wrapper);
    return -1;
  }

  /* Check if 'object' can be instrumented */
  if(object)
  {
    if(object->co_isBackend)
    {
      if(PDICFG.donttouch_backends)
      {
        _pdi_error(cfg->bc_name, NULL,
                   "%d:Cannot instrument object '%s' if option "
                   "'donttouch_backends' is enabled.",
                   cfg->bc_lineno, _pdi_becfg_getObjectName(object));
        return -1;
      } else
        _pdi_warning(cfg->bc_name, NULL,
                     "%d:Tracing a backend ('%s') is not a good idea.",
                     cfg->bc_lineno, _pdi_becfg_getObjectName(object));
    }
    if(object->co_alias && !strcmp(object->co_alias, PDI_ALIAS_PDI))
    {
      if(PDICFG.donttouch_pdi)
      {
        _pdi_error(__FILE__, cfg->bc_name,
                   "%d:Cannot instrument '%s' if option 'donttouch_pdi' is enabled.",
                   cfg->bc_lineno, _pdi_becfg_getObjectName(object));
        return -1;
      } else
        _pdi_warning(__FILE__, cfg->bc_name,
                     "%d:Is VERY DANGEROUS to instrument pDI-Tools itself ('%s').",
                     cfg->bc_lineno, _pdi_becfg_getObjectName(object));
    }
  }

  /* If object 'backend' it is not a backend then shit, ass, fart, and error */
  if(!backend->co_isBackend)
  {
    if(PDICFG.allow_lib_as_be)
    {
      _pdi_warning(cfg->bc_name, NULL,
                   "%d:Regular object '%s' is used as a backend.",
                 cfg->bc_lineno, _pdi_becfg_getObjectName(backend));
    } else {
      _pdi_error(cfg->bc_name, NULL, "%d:Object '%s' is not a backend.",
                 cfg->bc_lineno, _pdi_becfg_getObjectName(backend));
      return -1;
    }
  }

  /* alloc memory for and initialize the new intercept */
  if((bi = malloc(sizeof(BECFG_INTERCEPT))) == NULL)
  {
    if(bi) free(bi);
    _pdi_error(THIS, "Cannot alloc memory for a BECFG_INTERCEPT struct.");
    return -1;
  }

  bi->ci_type = type;
  bi->ci_object = object;
  bi->ci_backend = backend;
  bi->ci_lineno = cfg->bc_lineno;
  bi->ci_next = NULL;
  if(wrapper)
  {
    if((bi->ci_wrapper = malloc(strlen(wrapper) + 1)) == NULL)
    {
      _pdi_error(THIS,
                 "Cannot alloc memory for string '%s'.", wrapper);
      return -1;
    }
    strcpy(bi->ci_wrapper, wrapper);
  } else
    bi->ci_wrapper = NULL;
  if(symbol)
  {
    if((bi->ci_symbol  = malloc(strlen(symbol)  + 1)) == NULL)
    {
      _pdi_error(THIS, "Cannot alloc memory for string '%s'.", symbol);
      return -1;
    }
    strcpy(bi->ci_symbol, symbol);
  } else
    bi->ci_symbol = NULL;

  /* Add this structure to the list of interpositions */
  if(cfg->bc_rdeflist)
    cfg->bc_rdeflist_last = cfg->bc_rdeflist_last->ci_next = bi;
  else
    cfg->bc_rdeflist = cfg->bc_rdeflist_last = bi;

  return 0;
}

/*****************************************************************************
 * BECFG_CONFIG *_pdi_becfg_newConfig(char *name)
 *
 * Description:
 *   Alloc memory for a new configuration.
 *
 * Parameters:
 *   name - config name (usually the name of the readed file)
 *
 * Returns:
 *   A pointer to the configuration or NULL if error.
 *
 *****************************************************************************/

BECFG_CONFIG *_pdi_becfg_newConfig(char *name)
{
  BECFG_CONFIG *cfg;

  /* alloc and initialize the structure */
  if((cfg = malloc(sizeof(BECFG_CONFIG))) == NULL
  || (cfg->bc_name = malloc(strlen(name) + 1)) == NULL)
  {
    if(cfg && cfg->bc_name) free(cfg->bc_name);
    if(cfg) free(cfg);

    _pdi_error(THIS, "Cannot alloc memory for a new BECFG_CONFIG.");
    return NULL;
  }

  strcpy(cfg->bc_name, name);
  cfg->bc_lastChar      = '\0';
  cfg->bc_lineno        = 0;
  cfg->bc_buffSize      = 0;
  cfg->bc_f             = NULL;
  cfg->bc_objlist       = NULL;
  cfg->bc_rdeflist      = NULL;
  cfg->bc_objlist_last  = NULL;
  cfg->bc_rdeflist_last = NULL;
  cfg->bc_buffer        = NULL;
  cfg->bc_deps          = NULL;

  /* Add the objects PDI_ALIAS_LIBC, PDI_ALIAS_MAIN, PDI_ALIAS_PDI */
  if(!addObject(cfg, NULL, PDI_ALIAS_LIBC, 0)
  || !addObject(cfg, NULL, PDI_ALIAS_MAIN, 0)
  || !addObject(cfg, NULL, PDI_ALIAS_PDI, 0))
  {
    _pdi_error(THIS, "An error happened while assigning alias "
                     PDI_ALIAS_LIBC ", "
                     PDI_ALIAS_MAIN " and/or "
                     PDI_ALIAS_PDI ".");
    _pdi_becfg_destroyConfig(cfg);
    return NULL;
  }

  return cfg;
}

/*****************************************************************************
 * void _pdi_becfg_destroyConfig(BECFG_CONFIG *cfg)
 *
 * Description:
 *   This functions free memory used by config 'cfg'.
 *
 * Parameters:
 *   cfg - config to destroy
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

void _pdi_becfg_destroyConfig(BECFG_CONFIG *cfg)
{
  BECFG_INTERCEPT *bi, *bi_next;
  BECFG_OBJ_INFO *bo, *bo_next;
  int i;

  /* Destroy backend dependency info */
  if(cfg->bc_deps)
  {
    for(i = 0; i < cfg->bc_deps->g_nnodes; i++)
    {
      if(cfg->bc_deps->g_nodes[i]->gn_to) free(cfg->bc_deps->g_nodes[i]->gn_to);
      free(cfg->bc_deps->g_nodes[i]);
    }
    if(cfg->bc_deps->g_nodes) free(cfg->bc_deps->g_nodes);
    free(cfg->bc_deps);
  }

  /* free buffers */
  if(cfg->bc_name)   free(cfg->bc_name);
  if(cfg->bc_buffer) free(cfg->bc_buffer);

  /* free interpositions */
  bi = cfg->bc_rdeflist;
  while(bi != NULL)
  {
    bi_next = bi->ci_next;
    if(bi->ci_symbol)  free(bi->ci_symbol);
    if(bi->ci_wrapper) free(bi->ci_wrapper);
    free(bi);
    bi = bi_next;
  }

  /* destroy object list */
  bo = cfg->bc_objlist;
  while(bo != NULL)
  {
    bo_next = bo->co_next;
    if(bo->co_path)  free(bo->co_path);
    if(bo->co_alias) free(bo->co_alias);
    free(bo);
    bo = bo_next;
  }

  /* destroy the config structure */
  free(cfg);
}

/*****************************************************************************
 * int _pdi_becfg_applyBackendConfig(BECFG_CONFIG *cfg)
 *
 * Description:
 *   Load and initialize backends declared in 'cfg'.
 *
 * Parameters:
 *   cfg - config to apply
 *
 * Returns:
 *   If succesful returns 0, otherwise returns -1.
 *
 *****************************************************************************/

int _pdi_becfg_applyBackendConfig(BECFG_CONFIG *cfg)
{
  BECFG_OBJ_INFO *oi;
  BECFG_INTERCEPT *bi;
  PDI_ELFOBJ *eo;

  /* Check:                                                          */
  /*   - all objects are in memory (warn user otherwise)             */
  /*   - backends are loaded in memory (otherwise load them)         */
  /* in the other hand this function checks that the config 'cfg' is */
  /* coherent with current configuration (backends are backends in   */
  /* config 'cfg' and in current running config).                    */
  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
  {
    /* first try to get the object by its path, and if it cannot */
    /* be find then try to search it by its alias                */
    oi->co_eo = NULL;
    if(!oi->co_path || (oi->co_eo = _pdi_ebe_searchObjectByPath(oi->co_path)) == NULL)
      oi->co_eo = _pdi_ebe_searchObjectByAlias(oi->co_alias);

    if(!oi->co_eo)
    {
      /* The object is not in memory! if this is a regular object then print */
      /* a warning and continue, otherwise try to load this backend          */
      if(!oi->co_isBackend)
      {
        _pdi_warning(cfg->bc_name, NULL, "Cannot found object '%s' in memory.",
                     _pdi_becfg_getObjectName(oi));
        continue;
      }

      /* if we have not a path to the object then issue a fatal error */
      if(!oi->co_path)
      {
        _pdi_error(cfg->bc_name, NULL,
                   "Cannot load a backend only knowing its alias '%s'.",
                   oi->co_alias);
        return -1;
      }

      /* ok... we're going to try to load and initialize this backend */
      _pdi_log(THIS, "Loading and initializing backend '%s'.",
               _pdi_becfg_getObjectName(oi));
      if((oi->co_eo = _pdi_ebe_loadBackend(oi->co_path)) == NULL)
      {
        _pdi_error(cfg->bc_name, NULL, "Cannot load backend '%s' in memory.",
                   _pdi_becfg_getObjectName(oi));
        return -1;
      }
    } else {
      /* this object is currently in memory. Check that its role in config */
      /* (backend/object) is the same in the current session               */
      if((oi->co_isBackend && !oi->co_eo->o_isBackend)
      || (!oi->co_isBackend && oi->co_eo->o_isBackend))
      {
        _pdi_error(cfg->bc_name, NULL,
                   "Object '%s' expected to be a %s, but actually it is a %s.",
                   _pdi_becfg_getObjectName(oi),
                   oi->co_isBackend ? "backend" : "regular object",
                   oi->co_eo->o_isBackend ? "backend" : "regular object");
        return -1;
      }

      /* if it is a backend warn that was not expected in memory */
      if(oi->co_isBackend)
        _pdi_warning(THIS, "Backend '%s' was previously loaded and it have "
                     "not been loaded by pDI-Tools.",
                     _pdi_becfg_getObjectName(oi));
    }

    /* assign its alias to this object */
    if(oi->co_alias
    && (!oi->co_eo->o_alias || strcmp(oi->co_eo->o_alias, oi->co_alias))
    && _pdi_ebe_setObjectAlias(oi->co_eo, oi->co_alias))
    {
      _pdi_error(cfg->bc_name, NULL, "Cannot asign alias '%s' to '%s'.",
                 oi->co_alias, _pdi_becfg_getObjectName(oi));
      return -1;
    }
  }

  /* Apply interposition commands */
  for(bi = cfg->bc_rdeflist; bi; bi = bi->ci_next)
  {
    /* Check required objects are available */
    if((bi->ci_object && !bi->ci_object->co_eo) || !bi->ci_backend->co_eo)
    {
      _pdi_warning(cfg->bc_name, NULL,
                   "Ignoring interposition command: %s '%s' is not present.",
                   !bi->ci_backend->co_eo ? "backend" : "object",
                   _pdi_becfg_getObjectName(!bi->ci_backend->co_eo
                                              ? bi->ci_backend
                                              : bi->ci_object));
      continue;
    }

    /* Check wrapper */
    if(bi->ci_type != BECFG_IT_CALLBACK && !bi->ci_wrapper)
    {
      _pdi_error(cfg->bc_name, NULL,
                 "Ignoring interposition command: NULL wrapper.");
      continue;
    }

    /* if user used a wildcard on a redefinition ignore this interposition */
    if(bi->ci_type != BECFG_IT_REDEFINITION && !bi->ci_object)
    {
      _pdi_error(cfg->bc_name, NULL, "Ignoring interposition command: "
                 "cannot use a wildcard un a redefinition.");
      continue;
    }

    if(!bi->ci_object)
    {
      /* WILDCARD */
      /* Apply this interposition on all instrumentable objects (usually all */
      /* objects but backends and libpdi.so [see 'donttouch_backends' and    */
      /* 'donttouch_pdi' pDI-Tools configuration options])                   */
      for(eo = _pdi_objlist; eo; eo = eo->o_next)
        if((!PDICFG.donttouch_backends || !eo->o_isBackend)
        && (!PDICFG.donttouch_pdi      || eo != _pdi_pdiobj))
          if(_pdi_ebe_installInterposition(bi->ci_type,
                                           eo, bi->ci_symbol,
                                           bi->ci_backend->co_eo, bi->ci_wrapper))
            _pdi_warning(cfg->bc_name, NULL,
                         "Cannot install correctly this interposition.");
    } else {
      /* Apply this interposition */
      if(_pdi_ebe_installInterposition(bi->ci_type,
                                       bi->ci_object->co_eo, bi->ci_symbol,
                                       bi->ci_backend->co_eo, bi->ci_wrapper))
        _pdi_warning(cfg->bc_name, NULL,
                     "Cannot install correctly this interposition.");
    }
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_becfg_initializeRead(BECFG_CONFIG *cfg, FILE *f)
 *
 * Description:
 *   Prepare config structure 'cfg' to begin to read the config from file 'f'.
 *
 * Parameters:
 *   cfg - config
 *   f   - input file
 *
 * Returns:
 *   If succesful returns 0, otherwise returns -1.
 *
 *****************************************************************************/

int _pdi_becfg_initializeRead(BECFG_CONFIG *cfg, FILE *f)
{
  if((cfg->bc_buffer = malloc(BECFG_MAX_LINE + 1)) == NULL)
  {
    _pdi_error(THIS, "No memory for reading buffer.");
    return -1;
  }
  cfg->bc_buffSize = BECFG_MAX_LINE;
  cfg->bc_f = f;

  return 0;
}

/*****************************************************************************
 * void _pdi_becfg_finalizeRead(BECFG_CONFIG *cfg)
 *
 * Description:
 *   Finish file reading on 'cfg'.
 *
 * Parameters:
 *   cfg - config
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

void _pdi_becfg_finalizeRead(BECFG_CONFIG *cfg)
{
  if(cfg->bc_buffer) free(cfg->bc_buffer);
  cfg->bc_buffer = NULL;
  cfg->bc_buffSize = 0;
  cfg->bc_f = NULL;
}

/*****************************************************************************
 * void _pdi_becfg_showObjects(int level, BECFG_CONFIG *cfg)
 *
 * Description:
 *   Show object list contained in config 'cfg'. The parameter 'level' can be
 *   one of LOG_LEVEL_* values defined in file '/include/log.h' and it sets
 *   from which verbosity level this list should be shown.
 *
 * Parameters:
 *   level - verbosity level
 *   cfg   - config
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

void _pdi_becfg_showObjects(int level, BECFG_CONFIG *cfg)
{
  BECFG_OBJ_INFO *oi;
  int lobj, lali, lname, t;
  char buffer[100];

  if(level > 0 && PDICFG.verbose < level)
    return;

  /* Calculate how much space is needed for objects and aliases */
  lobj = 6; lali = 5;
  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
  {
    if(oi->co_path  && (t = strlen(oi->co_path))  > lobj) lobj = t;
    if(oi->co_alias && (t = strlen(oi->co_alias)) > lali) lali = t;
  }
  lname = strlen(cfg->bc_name);
  if(lobj + lali - 2 < lname)
  {
    t = lname - (lobj + lali - 2);
    lobj += (t >> 1) + (t & 1);
    lali += (t >> 1);
  }
  lname = lobj + lali - 2;

  /* Print to separators and config name */
  fputc('+', PDICFG.logfile);
  for(t = 0; t < 15 + lobj + lali; t++) fputc('-', PDICFG.logfile);
  fprintf(PDICFG.logfile, "+\n");

  snprintf(buffer, 100, "| Configuration: %%-%ds |\n", lname);
  fprintf(PDICFG.logfile, buffer, cfg->bc_name);

  fputc('+', PDICFG.logfile); for(t = 0; t < 9; t++) fputc('-', PDICFG.logfile);
  fputc('+', PDICFG.logfile); for(t = 0; t < 2 + lobj; t++) fputc('-', PDICFG.logfile);
  fputc('+', PDICFG.logfile); for(t = 0; t < 2 + lali; t++) fputc('-', PDICFG.logfile);
  fprintf(PDICFG.logfile, "+\n");

  /* Prepare format string to print the list */
  snprintf(buffer, 100, "| %%-7s | %%-%ds | %%-%ds |\n", lobj, lali);

  /* Print header and a separator */
  fprintf(PDICFG.logfile, buffer, "TYPE", "OBJECT", "ALIAS");

  fputc('+', PDICFG.logfile);
  for(t = 0; t < 9; t++) fputc('-', PDICFG.logfile);
  fputc('+', PDICFG.logfile);
  for(t = 0; t < 1 + lobj + 1; t++) fputc('-', PDICFG.logfile);
  fputc('+', PDICFG.logfile);
  for(t = 0; t < 1 + lali + 1; t++) fputc('-', PDICFG.logfile);
  fprintf(PDICFG.logfile, "+\n");

  /* print the list */
  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
    fprintf(PDICFG.logfile, buffer,
             oi->co_isBackend ? "BACKEND" : "object",
             oi->co_path  ? oi->co_path  : "-",
             oi->co_alias ? oi->co_alias : "");

  /* print final separator */
  fputc('+', PDICFG.logfile);
  for(t = 0; t < 9; t++) fputc('-', PDICFG.logfile);
  fputc('+', PDICFG.logfile);
  for(t = 0; t < 1 + lobj + 1; t++) fputc('-', PDICFG.logfile);
  fputc('+', PDICFG.logfile);
  for(t = 0; t < 1 + lali + 1; t++) fputc('-', PDICFG.logfile);
  fprintf(PDICFG.logfile, "+\n");
}

