/*****************************************************************************
 * $Id: bcfgread.c,v 1.1 2005/07/23 22:31:16 killabyte Exp $
 *
 * This file implements some tools for reading interposition commands files (or
 * backend config files). In those files are specified which on which files we
 * are going to install interpositions, which functions will be affected by
 * them, and who (backend and wrapper) is going to receive the interposition.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<beconfig.h>
#include<log.h>

/*****************************************************************************
 * static int readLine(BECFG_CONFIG *cfg)
 *
 * Description:
 *   Read a line from a file, and trim spaces.
 *
 * Parameters:
 *   cfg - config file
 *
 * Returns:
 *   -1 if have readed correctly a line or 0 if have found the end-of-file
 *
 *****************************************************************************/

static int readLine(BECFG_CONFIG *cfg)
{
  char *p;
  int i = '\0';

  while(cfg->bc_lastChar != EOF)
  {
    /* Read a line */
    cfg->bc_lineno++;
    for(p = cfg->bc_buffer;
        p < (cfg->bc_buffer + cfg->bc_buffSize) && (*p = (i = fgetc(cfg->bc_f))) != '\n' && i != EOF;
        p++)
      ;

    /* Remember last readed char and close the string */
    cfg->bc_lastChar = i;
    *p = '\0';

    /* If the read was a succes trim the string and exit */
    if(p < cfg->bc_buffer + cfg->bc_buffSize || cfg->bc_lastChar == EOF || cfg->bc_lastChar == '\n')
    {
      if(strlen(cfg->bc_buffer) > 0)
      {
        /* chop last white spaces */
        for(p = cfg->bc_buffer + strlen(cfg->bc_buffer) - 1;
            p > cfg->bc_buffer && (*p == ' ' || *p == '\t');
            p--)
          *p = '\0';

        /* del begining spaces */
        for(p = cfg->bc_buffer, i = 0;
            p < cfg->bc_buffer + cfg->bc_buffSize && (*p == ' ' || *p == '\t');
            p++, i++)
          ;
        if(i > 0)
        {
          for(p = cfg->bc_buffer; *(p + i) != '\0'; p++)
            *p = *(p + i);
          *p = '\0';
        }
      }

      return -1;
    }

    /* oh! long line. Ignore file contents until the next line */
    while((cfg->bc_lastChar = fgetc(cfg->bc_f)) != EOF && cfg->bc_lastChar != '\n')
      ;
  }

  return 0;
}

/*****************************************************************************
 * static int readString(char **p, char **ret)
 *
 * Description:
 *   From position 'p' in a buffer this function try to read a whole word or a
 *   string delimited by quotes ("). It returns in 'p' the address of the last
 *   char found and the beginning of the string in 'ret'. If the string was
 *   quoted it interprets escape characters, so things like \" and \\ are
 *   substituted by " and \.
 *
 * Parameters:
 *   p   - (in and out parameter) position from it begins to read, and when
 *         exit it returns in this parameter the last readed character
 *         position.
 *   ret - (out) where the string starts.
 *
 * Returns:
 *   -1 if there was an error (quoted string without end quote, for instance),
 *   or the lenght of the string.
 *
 *****************************************************************************/

static int readString(char **p, char **ret)
{
  register char *t = *p;
  register int i;

  for(; *t && (*t == ' ' || *t == '\t'); t++)
    ;
  if(!*t)
    return 0;

  if(*t == '"')
  {
    /* Extract the string */
    *ret = ++t;
    for(; *t && *t != '"'; t++)
      if(*t == '\\')
        if(*++t == '\0')
          break;
    if(*t != '"')
      return -1;
    if(*t != '\0')
      *t++ = '\0';
    *p = t;

    /* Process escape sequences (\x => x) */
    for(t = *ret, i = 0; *(t + i); )
    {
      if(*t == '\\')
        i++;
      if(i > 0)
        *t = *(t + i);
      t++;
    }
    *t = '\0';
  } else {
    *ret = t;
    for(; *t && *t != '\t' && *t != ' '; t++)
      ;
    if(*t != '\0')
      *t++ = '\0';
    *p = t;
  }

  return strlen(*ret);
}

/*****************************************************************************
 * static int parseModuleLine(BECFG_CONFIG *cfg, char **str1, char **str2)
 *
 * Description:
 *   Parse the line in the buffer and returns
 *
 *     - in 'str1' a pointer to a string with the name of the object, and in
 *       'str2' its alias, if found.
 *     - nothing if a comentary was found.
 *     - a special signal if found '#commands'
 *
 *   The value returned by the function is a numeric identifier telling what
 *   thing was found (#define, #object, #commands, comentary, error...)
 *
 * Parameters:
 *   cfg    - Config file being readed.
 *   str1   - (out) string with object path or an error description.
 *   str2   - (out) string with object alias or NULL
 *
 * Returns:
 *   -1 if error, or:
 *
 *         0   Void line or comentary
 *         1   A regular object was declared
 *         2   A backend declaration
 *         3   End of this section (#commands).
 *
 *  Other value should be considered as an error.
 *
 *****************************************************************************/

static int parseModuleLine(BECFG_CONFIG *cfg, char **str1, char **str2)
{
  char *p = cfg->bc_buffer,
       *t;
  int backend = 0, l;

  /* If it is a comentary or a void line return now */
  if(*p == ';' || *p == '\0')
    return 0;

  /* If the line begins by #xxx, try to identify which directive is */
  if(*p == '#')
  {
    /* Jump white spaces and delimit the string to identify it. */
    for(p++; *p && (*p == ' ' || *p == '\t'); ++p)
      ;
    t = p;
    for(; *p && *p != ' ' && *p != '\t'; ++p)
      ;

    /* If the directive found mark the begining of a the next section, */
    /* exit from this fuction returning code 3                         */
    if(*p == '\0')
    {
      if(strcmp(t, "commands") && strcmp(t, "relinks"))
      {
        _pdi_error(NULL, cfg->bc_name, "%d:Unknown directive '%s'.", cfg->bc_lineno, t);
        return -1;
      } else
        return 3;
    }
    *p++ = '\0';

    /* Check that it is a known directive */
    if(strcmp(t, "define")   && strcmp(t, "backend") && strcmp(t, "object") &&
       strcmp(t, "commands") && strcmp(t, "relinks"))
    {
      _pdi_error(NULL, cfg->bc_name, "%d:Unknown directive '%s'.", cfg->bc_lineno, t);
      return -1;
    }

    /* If this directive is 'commands' or 'relinks' and the execution  */
    /* has arrived here it means that there is trash after it. This is */
    /* a syntax error.                                                 */
    if(!strcmp(t, "commands") || !strcmp(t, "relinks"))
    {
      _pdi_warning(NULL, cfg->bc_name, "%d:Trailing trash after '%s'.", cfg->bc_lineno, t);
      return 3;
    }

    /* At this point, 't' only can be 'define', 'backend' or 'object',  */
    /* and depending on this we decide the value of the flag 'backend'. */
    backend = !strcmp(t, "backend");
  } else
    backend = 1;

  /* Get the path to the library and, if it exist, its alias     */
  /* (start assuming that first comes alias and second the path) */
  if((l = readString(&p, str2)) < 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Unfinished string.", cfg->bc_lineno);
    return -1;
  }
  if(l == 0)
  {
    _pdi_error(NULL, cfg->bc_name,
               "%d:Path to the object/backend is needed.",
               cfg->bc_lineno);
    return -1;
  }
  if((l = readString(&p, str1)) < 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Unfinished string.", cfg->bc_lineno);
    return -1;
  }

  /* If the second string doesn't exist, we don't have an alias and */
  /* the first string should be the path to the object              */
  if(l == 0)
  {
    *str1 = *str2;
    *str2 = NULL;
  }

  return backend ? 2 : 1;
}

/*****************************************************************************
 * static int parseCommandLine(BECFG_CONFIG *cfg,
 *                             int *type,
 *                             char **trgObj,
 *                             char **trgFunc,
 *                             char **backend,
 *                             char **wrapper)
 *
 * Description:
 *   Parse buffer contents (a line) and it can detect
 *
 *     - a command in parameters type, trgObj, trgFunc, backend and wrapper.
 *     - or a void line or a comentary
 *
 *   The value returned by this function is a identifier telling what type of
 *   line has been detected.
 *
 * Parameters:
 *   cfg      - Config file being readed
 *   type     - it could be BECFG_IT_RELINK, BECFG_IT_REDEFINITION or
 *              BECFG_IT_CALLBACK.
 *   trgObj   - target object (NULL is a wildcard)
 *   trgFunc  - target function (NULL is a wildcard)
 *   backend  - backend
 *   wrapper  - function that will receive this interposition (in callback's it
 *              is usually NULL)
 *
 * Returns:
 *   -1 if error, in other cases:
 *
 *         0   Void line or comentary
 *         1   Interposition command
 *
 *  Another value is an error.
 *
 *****************************************************************************/

static int parseCommandLine(BECFG_CONFIG *cfg,
                            int *type,
                            char **trgObj,
                            char **trgFunc,
                            char **backend,
                            char **wrapper)
{
  int l;
  char *p = cfg->bc_buffer;

  /* If it is a comentary or a void line then exit from this function */
  if(*p == ';' || *p == '\0')
    return 0;

  /* The line should beging with interposition type ([FRDC] in uppercase) */
  if(*p == 'f' || *p == 'r' || *p == 'd' || *p == 'c')
  {
    _pdi_warning(NULL, cfg->bc_name,
                 "%d:Type of interposition should be in uppercase.",
                 cfg->bc_lineno);
    *p += 'A' - 'a';
  }

  switch(*p)
  {
    case 'F': /* type F is an R alias */
    case 'R': *type = BECFG_IT_RELINK;       break;
    case 'D': *type = BECFG_IT_REDEFINITION; break;
    case 'C': *type = BECFG_IT_CALLBACK;     break;
    default:
      _pdi_error(NULL, cfg->bc_name, "%d:Unknown interposition type '%c'.",
                 cfg->bc_lineno, *p);
      return -1;
  }

  /* a space is expected, and the rest of spaces are ignored */
  p++;
  if(*p != ' ' && *p != '\t')
  {
    _pdi_error(NULL, cfg->bc_name, "%d:White space expected.", cfg->bc_lineno);
    return -1;
  }

  for(p++; *p && (*p == ' ' || *p == '\t'); ++p)
    ;

  /* Now we should have 4 strings: */
  /* 1 - target object (or wildcard) */
  if((l = readString(&p, trgObj)) < 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Unfinished string.", cfg->bc_lineno);
    return -1;
  }
  if(l == 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Target object name forgotten.",
               cfg->bc_lineno);
    return -1;
  }
  if(!strcmp(*trgObj, "*"))
  {
    /* Redefinitions are incompatible with wildcards */
    if(*type == BECFG_IT_REDEFINITION)
    {
      _pdi_error(NULL, cfg->bc_name,
                 "%d:Wildcards not allowed on redefinitions.", cfg->bc_lineno);
      return -1;
    }

    *trgObj = NULL;
  }

  /* 2 - target function (or wildcard) */
  if((l = readString(&p, trgFunc)) < 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Unfinished string.", cfg->bc_lineno);
    return -1;
  }
  if(l == 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Expected function name to redefine.",
               cfg->bc_lineno);
    return -1;
  }
  if(!strcmp(*trgFunc, "NULL"))
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Target function cannot be NULL.",
               cfg->bc_lineno);
    return -1;
  }
  if(!strcmp(*trgFunc, "*"))
  {
    /* Redefinitions doesn't accept wildcards */
    if(*type == BECFG_IT_REDEFINITION)
    {
      _pdi_error(NULL, cfg->bc_name,
                 "%d:You cannot create a callback with redefinitions.",
                 cfg->bc_lineno);
      return -1;
    }

    /* If this interposition is a relink, this wildcard */
    /* promote it to a callback                         */
    *type = BECFG_IT_CALLBACK;
    *trgFunc = NULL;
  } else
    if(*type == BECFG_IT_CALLBACK)
    {
      _pdi_error(NULL, cfg->bc_name,
                 "%d:You cannot use a callback on only one function.",
                 cfg->bc_lineno);
      return -1;
    }

  /* 3 - backend alias or path */
  if((l = readString(&p, backend)) < 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Unfinished string.", cfg->bc_lineno);
    return -1;
  }
  if(l == 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Expected backend name or alias.",
               cfg->bc_lineno);
    return -1;
  }
  if(!strcmp(*backend, "*"))
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Expected a backend, not a wildcard.",
               cfg->bc_lineno);
    return -1;
  }

  /* 4 - function wrapper (or generic wrapper or NULL in case of callback) */
  if((l = readString(&p, wrapper)) < 0)
  {
    _pdi_error(NULL, cfg->bc_name, "%d:Unfinished string.", cfg->bc_lineno);
    return -1;
  }
  if(l == 0)
  {
    if(*type != BECFG_IT_CALLBACK)
    {
      _pdi_error(NULL, cfg->bc_name, "%d:Expected wrapper function name.",
                 cfg->bc_lineno);
      return -1;
    }
    *wrapper = NULL;
  } else {
    if(!strcmp(*wrapper, "*"))
    {
      _pdi_error(NULL, cfg->bc_name,
                 "%d:Expected wrapper function name, not a wildcard.",
                 cfg->bc_lineno);
      return -1;
    }
    if(!strcmp(*wrapper, "NULL"))
      *wrapper = NULL;
  }

  /* ignore the rest of the line */
  for(; *p && (*p == ' ' || *p == '\t'); ++p)
    ;
  if(*p)
    _pdi_warning(NULL, cfg->bc_name, "%d:Ignoring the rest of the line.",
                 cfg->bc_lineno);

  return 1;
}

/*****************************************************************************
 * static int loadModulesList(BECFG_CONFIG *cfg)
 *
 * Description:
 *   Read the first part of interposition commands file interpreting the list
 *   of objects and alias.
 *
 * Parameters:
 *   cfg - interposition commands file readed
 *
 * Returns:
 *   -1 if error, 0 otherwise.
 *
 *****************************************************************************/

static int loadModulesList(BECFG_CONFIG *cfg)
{
  char *str1, *str2;
  int status;

  while(readLine(cfg))
    switch(status = parseModuleLine(cfg, &str1, &str2))
    {
      case 0: /* void line or comentary */
        break;
      case 1: /* object */
      case 2: /* backend */
        if(!_pdi_becfg_addObject(cfg, str1, str2, status == 2))
        {
          _pdi_error(THIS, "Cannot add '%s' to the object list.", str1);
          return -1;
        }
        break;
      case 3: /* commands */
        return 0;
      case -1: /* syntax error */
        return -1;
      default: /* strange value */
        _pdi_error(THIS,
                   "Function parseModuleLine() returned a unknown code (%d).",
                   status);
        return -1;
    }

  return 0;
}

/*****************************************************************************
 * static int loadCommands(BECFG_CONFIG *cfg)
 *
 * Description:
 *   Process the second part of a interposition commands file: interposition
 *   commands.
 *
 * Parameters:
 *   cfg - interposition commands file readed
 *
 * Returns:
 *   -1 if error, 0 otherwise.
 *
 *****************************************************************************/

static int loadCommands(BECFG_CONFIG *cfg)
{
  char *trgObj, *trgFunc, *strBackend, *wrapper;
  BECFG_OBJ_INFO *object, *backend;
  int status, type;

  while(readLine(cfg))
    switch(status = parseCommandLine(cfg, &type, &trgObj, &trgFunc, &strBackend, &wrapper))
    {
      case 0: /* void line or comentary */
        break;
      case 1: /* interposition command */
        /* Try to find 'trgObj' in the current object list. If cannot find */
        /* it, then it will be added to the 'cfg' object list.             */
        if(trgObj)
        {
          if((object = _pdi_becfg_getObject(cfg, trgObj)) == NULL)
          {
            _pdi_warning(cfg->bc_name, NULL,
                         "%d:Object '%s' was not previously declared.",
                         cfg->bc_lineno, trgObj);
            if((object = _pdi_becfg_addObject(cfg, trgObj, NULL, 0)) == NULL)
            {
              _pdi_error(THIS,
                         "There was an error while adding '%s' to the object list.",
                         trgObj);
              return -1;
            }
          }
        } else
          object = NULL;

        /* Search the object 'strBackend'. If it not exist in the object  */
        /* list then the object will be added to it, but the program will */
        /* shout a lot of horrible adjectives to the luser.               */
        if((backend = _pdi_becfg_getObject(cfg, strBackend)) == NULL)
        {
          /* Shit! This interposition is making a reference to a backend  */
          /* totally unknown to us! This backend shouldn't exist! We will */
          /* add it, but first we're going to say a crude.                */
          _pdi_warning(cfg->bc_name, NULL,
                       "%d:Backend '%s' should have been declared in the object list.",
                       cfg->bc_lineno, strBackend);
          if((backend = _pdi_becfg_addObject(cfg, strBackend, NULL, 1)) == NULL)
          {
            _pdi_error(THIS,
                       "There was an error while adding '%s' to the object list.",
                       strBackend);
            return -1;
          }
        }

        /* Add this command */
        if(_pdi_becfg_addCommand(cfg, type, object, trgFunc, backend, wrapper))
        {
          _pdi_error(THIS,
                     "%s:%d:An error ocurred while adding a interposition.",
                     cfg->bc_name, cfg->bc_lineno);
          return -1;
        }
        break;
      case -1: /* syntax error */
        return -1;
      default: /* strange value */
        _pdi_error(THIS,
                   "Function parseCommandLine() returned and unknown code (%d).",
                   status);
        return -1;
    }

  return 0;
}

/*****************************************************************************
 * static BECFG_CONFIG *readBackendConfig(char *name, FILE *f)
 *
 * Description:
 *   Parse a interposition commands file from file 'f', assuming that its name
 *   is 'name'.
 *
 * Parameters:
 *   name - name of the config file
 *   f    - stream from we are going to read the file
 *
 * Returns:
 *   NULL if an error ocurred, or a pointer to a config if all was ok.
 *
 *****************************************************************************/

static BECFG_CONFIG *readBackendConfig(char *name, FILE *f)
{
  BECFG_CONFIG *cfg;

  /* create a new config */
  if((cfg = _pdi_becfg_newConfig(name)) == NULL)
  {
    _pdi_error(THIS, "An error ocurred while creating a config for '%s'.",
               name);
    return NULL;
  }

  /* link the new config with file 'f' and initialize the read buffer */
  if(_pdi_becfg_initializeRead(cfg, f))
  {
    _pdi_becfg_destroyConfig(cfg);
    _pdi_error(THIS, "Cannot initialize the reading.");
    return NULL;
  }

  /* parse the object and backend list */
  if(loadModulesList(cfg))
  {
    _pdi_error(THIS, "An error ocurred while reading object list from '%s'.",
               cfg->bc_name);
    _pdi_becfg_destroyConfig(cfg);
    return NULL;
  }

  /* parse the interposition commands */
  if(loadCommands(cfg))
  {
    _pdi_error(THIS,
               "An error ocurred while reading interposition commands from '%s'.",
               cfg->bc_name);
    _pdi_becfg_destroyConfig(cfg);
    return NULL;
  }

  /* terminate read */
  _pdi_becfg_finalizeRead(cfg);

  return cfg;
}

/*****************************************************************************
 * int _pdi_becfg_loadBackendConfig(char *fname);
 * int _pdi_becfg_loadBackendConfigFromFile(FILE *f);
 *
 * Description:
 *   Those functions allow to parse and load from an interposition commands
 *   file a configuration. They doesn't check that the readed config fit with
 *   the real objects and functions in memory, only build the config
 *   structures.
 *
 * Parameters:
 *   fname - path to the file to parse
 *   f     - file from the interposition commands will be readed
 *
 * Returns:
 *   NULL if an error ocurred, or a pointer to a config if all was ok.
 *
 *****************************************************************************/

BECFG_CONFIG *_pdi_becfg_loadBackendConfigFromFile(FILE *f)
{
  BECFG_CONFIG *cfg;

  /* read the config directly */
  if((cfg = readBackendConfig(BECFG_ANON_FILE, f)) == NULL)
  {
    _pdi_error(THIS, "Cannot read the interposition commands file from the stream.");
    return NULL;
  }

  return cfg;
}

BECFG_CONFIG *_pdi_becfg_loadBackendConfig(char *fname)
{
  FILE *f;
  BECFG_CONFIG *cfg;

  /* try to open the interposition commands file */
  if((f = fopen(fname, "r")) == NULL)
  {
    _pdi_error(THIS, "Cannot open the file '%s'.", fname);
    return NULL;
  }

  /* read it */
  if((cfg = readBackendConfig(fname, f)) == NULL)
    _pdi_error(THIS, "Cannot read the file '%s'.", fname);

  fclose(f);

  return cfg;
}

