/*****************************************************************************
 * $Id: bcfgmrge.c,v 1.1 2005/07/23 15:16:41 killabyte Exp $
 *
 * Solution for combining several interposition command files in a final
 * config.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<beconfig.h>
#include<ebeif.h>
#include<log.h>

/*****************************************************************************
 * BECFG_OBJ_TRANSF
 *
 *   This structure is used to build a table that allows to search quickly a
 *   BECFG_OBJ_INFO of a config equivalent to a BECFG_OBJ_INFO of an another
 *   config.
 *
 *****************************************************************************/

typedef struct _tag_BECFG_OBJ_TRANSF {
  BECFG_OBJ_INFO *os;
  BECFG_OBJ_INFO *od;
} BECFG_OBJ_TRANSF;

/*****************************************************************************
 * static BECFG_GRAPH *new_graph(void)
 *
 * Description:
 *   Build a new graph.
 *
 * Parameters:
 *   none
 *
 * Returns:
 *   A pointer to the new graph, or NULL if happened an error.
 *
 *****************************************************************************/

static BECFG_GRAPH *new_graph(void)
{
  BECFG_GRAPH *g;

  if((g = malloc(sizeof(BECFG_GRAPH))) == NULL)
    return NULL;

  g->g_nnodes = 0;
  g->g_nodes = NULL;

  return g;
}

/*****************************************************************************
 * static BECFG_GRAPH_NODE *new_graph_node(BECFG_GRAPH *g,
 *                                         BECFG_OBJ_INFO *object)
 *
 * Description:
 *   Adds a new node to graph 'g' and associates it to object 'object'.
 *
 * Parameters:
 *   g      - graph
 *   object - object that will be associate to the new node
 *
 * Returns:
 *   A pointer to the new node or NULL if error.
 *
 *****************************************************************************/

static BECFG_GRAPH_NODE *new_graph_node(BECFG_GRAPH *g, BECFG_OBJ_INFO *object)
{
  BECFG_GRAPH_NODE **nt;

  if((nt = realloc(g->g_nodes, sizeof(BECFG_GRAPH_NODE *) * (g->g_nnodes + 1))) == NULL)
  {
    _pdi_error(THIS, "No memory (1).");
    return NULL;
  }
  g->g_nodes = nt;

  if((g->g_nodes[g->g_nnodes] = malloc(sizeof(BECFG_GRAPH_NODE))) == NULL)
  {
    _pdi_error(THIS, "No memory (2).");
    return NULL;
  }

  g->g_nodes[g->g_nnodes]->gn_object = object;
  g->g_nodes[g->g_nnodes]->gn_weight = 0;
  g->g_nodes[g->g_nnodes]->gn_hasPar = 0;
  g->g_nodes[g->g_nnodes]->gn_nto    = 0;
  g->g_nodes[g->g_nnodes]->gn_to     = NULL;

  g->g_nnodes++;

  return g->g_nodes[g->g_nnodes - 1];
}

/*****************************************************************************
 * static BECFG_GRAPH_NODE *get_graph_node_by_object(BECFG_GRAPH *g,
 *                                                   BECFG_OBJ_INFO *object)
 *
 * Description:
 *   Search in graph 'g' a node associated to object 'object'.
 *
 * Parameters:
 *   g      - graph
 *   object - searched object
 *
 * Returns:
 *   A pointer to the node associated with object 'object', or NULL if it does
 *   not exist.
 *
 *****************************************************************************/

static BECFG_GRAPH_NODE *get_graph_node_by_object(BECFG_GRAPH *g,
                                                  BECFG_OBJ_INFO *object)
{
  int i;
  for(i = 0; i < g->g_nnodes; i++)
    if(g->g_nodes[i]->gn_object == object)
      return g->g_nodes[i];
  return NULL;
}

/*****************************************************************************
 * static int push_parent(BECFG_GRAPH *g,
 *                        BECFG_GRAPH_NODE *n, BECFG_GRAPH_NODE *p)
 *
 * Description:
 *   In graph 'g', this function set node 'p' as father of node 'n'.
 *
 * Parameters:
 *   g      - working graph
 *   n      - node that will be son of 'p'.
 *   p      - node that will be father of 'n'.
 *
 * Returns:
 *   0 if ok, a negative value otherwise.
 *
 *****************************************************************************/

static int push_parent(BECFG_GRAPH *g,
                       BECFG_GRAPH_NODE *n, BECFG_GRAPH_NODE *p)
{
  BECFG_GRAPH_NODE **rels, **e;

  /* If 'n' or 'p' is NULL, finish now */
  if(!n || !p)
    return 0;

  /* Check that they are not the same node nor exists this relationship */
  if(n == p)
    return -1;

  if(p->gn_to)
    for(e = p->gn_to; *e; e++)
      if(*e == n)
        return 0;
  
  /* alloc memory for a new son and initialize it */
  if((rels = realloc(p->gn_to, sizeof(BECFG_GRAPH_NODE *) * (p->gn_nto + 2))) == NULL)
  {
    _pdi_error(THIS, "No memory!");
    return -1;
  }
  p->gn_to = rels;

  p->gn_to[p->gn_nto++] = n;
  p->gn_to[p->gn_nto] = NULL;

  /* now 'n' has a father */
  n->gn_hasPar = -1;

  return 0;
}

/*****************************************************************************
 * static int assign_weights(BECFG_GRAPH *g)
 *
 * Description:
 *   Update field 'weight' in all nodes in 'g'. This value is calculated in
 *   this way:
 *
 *     (i)  Nodes without parents have 0 weight.
 *     (ii) If node 'X' is son of group of nodes C, its weight will be:
 *          X.weight = MAX(C(1).weight, C(2).weight), ..., C(N).weight) + 1
 *
 *   If there is a cycle in this graph, this algorithm will fall in a infinite
 *   loop, and then returning an error.
 *
 * Parameters:
 *   g - graph
 *
 * Returns:
 *   0 if all ok, a negative value otherwise.
 *
 *****************************************************************************/

static int assign_weights(BECFG_GRAPH *g)
{
  int i, wt, flowers_found;
  BECFG_GRAPH_NODE **n;

  /* initialize all nodes with parents with weight -1, but nodes */
  /* without parent will be initialized to 0                     */
  for(i = 0; i < g->g_nnodes; i++)
    g->g_nodes[i]->gn_weight = g->g_nodes[i]->gn_hasPar ? -1 : 0;

  /* Now set 'wt' in each iteration starting from 0 to 'g->nnodes'. In each */
  /* iteration we set 'flowers_found' to 0, and set it to -1 if we found a  */
  /* node with weight 'wt'. If a iteration ends and we haven't found any    */
  /* node with weight 'wt', the flag 'flowers_found' will force the exit    */
  /* from the main loop. For all sons of a node with weight 'wt' set them   */
  /* with weight 'wt+1'.                                                    */
  /* If the condition (wt >= g->nnodes) is true, we can suspect that there  */
  /* is a cycle in this graph.                                              */
  for(flowers_found = -1, wt = 0; flowers_found && wt < g->g_nnodes + 1; wt++)
  {
    flowers_found = 0;
    for(i = 0; i < g->g_nnodes; i++)
      if(g->g_nodes[i]->gn_weight == wt)
      {
        flowers_found = -1;
        if(g->g_nodes[i]->gn_to)
          for(n = g->g_nodes[i]->gn_to; *n; n++)
            (*n)->gn_weight = wt + 1;
      }
  }

  /* Test all is ok */
  if(flowers_found && wt > g->g_nnodes)
    return -1;

  for(i = 0; i < g->g_nnodes; i++)
    if(g->g_nodes[i] && g->g_nodes[i]->gn_weight < 0)
      return -1;

  return 0;
}

/*****************************************************************************
 * static int build_deps_graph(BECFG_CONFIG *cfg)
 *
 * Description:
 *   If it is not built the backend dependency graph of config 'cfg', this
 *   function build and sort it.
 *
 * Parameters:
 *   cfg - configuration
 *
 * Returns:
 *   0 if all ok, a negative value otherwise.
 *
 *****************************************************************************/

static int build_deps_graph(BECFG_CONFIG *cfg)
{
  BECFG_GRAPH_NODE *prev, *act;
  BECFG_OBJ_INFO *oi;

  /* If graph is built exit quietly */
  if(cfg->bc_deps)
    return 0;

  /* Alloc memory for a new graph */
  if((cfg->bc_deps = new_graph()) == NULL)
  {
    _pdi_error(THIS, "Cannot build dependency graph.");
    return -1;
  }

  /* build the graph */
  prev = NULL;
  for(oi = cfg->bc_objlist; oi; oi = oi->co_next)
    if(oi->co_isBackend)
    {
      if((act = new_graph_node(cfg->bc_deps, oi)) == NULL)
      {
        _pdi_error(THIS, "There is no memory for a new node.");
        return -1;
      }
      if(push_parent(cfg->bc_deps, act, prev))
      {
        _pdi_error(THIS, "Something weird happened when adding a son.");
        return -1;
      }
      prev = act;
    }

  /* assign weights to graph nodes */
  if(assign_weights(cfg->bc_deps))
  {
    _pdi_error(THIS, "A dependency cycle found in '%s'.", cfg->bc_name);
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * static BECFG_OBJ_TRANSF *mergeObjects(BECFG_CONFIG *ctrg,
 *                                       BECFG_CONFIG *csrc)
 *
 * Description:
 *   Add objects in config 'csrc' in configuration 'ctrg' and return a
 *   BECFG_OBJ_TRANSF table that allows to search quickly an equivalent object
 *   of 'csrc' in 'ctrg'.
 *
 * Parameters:
 *   ctrg - target configuration
 *   csrc - source configuration
 *
 * Returns:
 *   A pointer to a BECFG_OBJ_TRANSF table if all is ok, or NULL if something
 *   weird happened.
 *
 *****************************************************************************/

static BECFG_OBJ_TRANSF *mergeObjects(BECFG_CONFIG *ctrg, BECFG_CONFIG *csrc)
{
  int total_src_obj, i;
  BECFG_OBJ_TRANSF *trnsf;
  BECFG_OBJ_INFO *ois, *oid;

  /* Count how many objects there is in 'csrc' */
  for(total_src_obj = 0, ois = csrc->bc_objlist; ois; ois = ois->co_next)
    total_src_obj++;

  /* Alloc and initialize memory for the conversion table */
  if((trnsf = malloc(sizeof(BECFG_OBJ_TRANSF) * (total_src_obj + 1))) == NULL)
  {
    _pdi_error(THIS, "No memory!");
    return NULL;
  }
  memset(trnsf, 0, sizeof(BECFG_OBJ_TRANSF) * (total_src_obj + 1));

  /* Mix objects and build the conversion table */
  for(i = 0, ois = csrc->bc_objlist; ois; ois = ois->co_next, i++)
  {
    /* Try to get a pointer to the object */
    oid = _pdi_becfg_getObject(ctrg, _pdi_becfg_getObjectName(ois));

    /* If this object does not exist try to add it. If it exist we have */
    /* to check that is is the same type of object (regular or backend) */
    /* in both configurations.                                          */
    if(oid != NULL)
    {
      if((ois->co_isBackend && !oid->co_isBackend)
      || (!ois->co_isBackend && oid->co_isBackend))
      {
        _pdi_error(THIS,
                   "Found a contradiction in '%s': "
                   "%s '%s' was declared before as %s.",
                   csrc->bc_name,
                   ois->co_isBackend ? "Backend" : "Object",
                   _pdi_becfg_getObjectName(ois),
                   ois->co_isBackend ? "an object"  : "a backend");
        free(trnsf);
        return NULL;
      }
    } else {
      if((oid = _pdi_becfg_addObject(ctrg,
                                     ois->co_path,
                                     ois->co_alias,
                                     ois->co_isBackend)) == NULL)
      {
        _pdi_error(THIS, "There is no memory for %s '%s'.",
                   ois->co_isBackend ? "backend" : "object",
                   ois->co_path);
        free(trnsf);
        return NULL;
      }
    }

    /* have to change alias? */
    if((!ois->co_alias && oid->co_alias)
    || (ois->co_alias && !oid->co_alias)
    || (ois->co_alias && oid->co_alias && strcmp(ois->co_alias, oid->co_alias)))
    {
      _pdi_warning(csrc->bc_name, NULL,
                   "Alias of '%s' is changed from '%s' to '%s'.",
                   _pdi_becfg_getObjectName(ois),
                   oid->co_alias ? oid->co_alias : "(none)",
                   ois->co_alias ? ois->co_alias : "(none)");
#warning Se tendra que comprobar que es un alias vlido.
      _pdi_becfg_setObjectAlias(oid, ois->co_alias);
    }

    /* Add a new entry to the transformation table */
    trnsf[i].os = ois;
    trnsf[i].od = oid;
  }

  return trnsf;
}

/*****************************************************************************
 * static BECFG_OBJ_INFO *getTrgObject(BECFG_OBJ_TRANSF *trnsf,
 *                                     BECFG_OBJ_INFO *src)
 *
 * Description:
 *   Search object 'src' in table 'trnsf' and returns its equivalent.
 *
 * Parameters:
 *   trnsf - equivalence table
 *   src   - object searched
 *
 * Returns:
 *   A pointer to the equivalent object or NULL if it doesn't exist (it should
 *   never happen!)
 *
 *****************************************************************************/

static BECFG_OBJ_INFO *getTrgObject(BECFG_OBJ_TRANSF *trnsf, BECFG_OBJ_INFO *src)
{
  for(; trnsf->os; trnsf++)
    if(trnsf->os == src)
      return trnsf->od;

  _pdi_error(THIS, "Object '%s' not found in conversion table.",
             _pdi_becfg_getObjectName(src));
  return NULL;
}

/*****************************************************************************
 * static int apply_backend_order(BECFG_OBJ_TRANSF *trnsf,
 *                                BECFG_CONFIG *ctrg,
 *                                BECFG_CONFIG *csrc)
 *
 * Description:
 *   This function try to add backends dependencies of config 'csrc' in the
 *   backend dependency graph of 'ctrg'. This can lead to a graph with cicles.
 *
 * Parameters:
 *   trnsf - equivalence table between objects in 'csrc' and 'ctrg'
 *   ctrg  - file on which will be applied new dependencies
 *   csrc  - file with new dependencies
 *
 * Returns:
 *   0 if all ok, a negative value otherwise.
 *
 *****************************************************************************/

static int apply_backend_order(BECFG_OBJ_TRANSF *trnsf,
                               BECFG_CONFIG *ctrg,
                               BECFG_CONFIG *csrc)
{
  BECFG_GRAPH_NODE *prev, *act;
  BECFG_OBJ_INFO *os, *ot;

  prev = NULL;
  for(os = csrc->bc_objlist; os; os = os->co_next)
    if(os->co_isBackend)
    {
      /* In 'ctrg' should exist object 'os->path' */
      if((ot = getTrgObject(trnsf, os)) == NULL)
      {
        _pdi_error(THIS, "Cannot found object '%s' in '%s'",
                   _pdi_becfg_getObjectName(os), ctrg->bc_name);
        return -1;
      }

      /* Search for this node in graph */
      act = get_graph_node_by_object(ctrg->bc_deps, ot);

      /* and if it does not exist then try to add it */
      if(!act && (act = new_graph_node(ctrg->bc_deps, ot)) == NULL)
      {
        _pdi_error(THIS, "Cannot create a node for '%s'",
                   _pdi_becfg_getObjectName(ot));
        return -1;
      }

      /* set 'prev' as 'act' father (if it is not NULL) */
      if(push_parent(ctrg->bc_deps, act, prev))
      {
        _pdi_error(THIS, "Cannot set '%s' as father of '%s'",
                   prev ? _pdi_becfg_getObjectName(prev->gn_object) : "(none)",
                   _pdi_becfg_getObjectName(ot));
        return -1;
      }

      /* in next iteration 'act' will be 'prev' */
      prev = act;
    }

  return 0;
}

/*****************************************************************************
 * static int sort_backends(BECFG_CONFIG *cfg)
 *
 * Description:
 *   Sort backends of config 'cfg' in the order stablished by its dependency
 *   graph.
 *
 * Parameters:
 *   cfg - config to sort
 *
 * Returns:
 *   0 if all ok, a negative value otherwise.
 *
 *****************************************************************************/

static int sort_backends(BECFG_CONFIG *cfg)
{
  BECFG_OBJ_INFO *oi;
  int wt, i;

  /* delete all backends and leave only regular objects in 'cfg' */
  while(cfg->bc_objlist && cfg->bc_objlist->co_isBackend)
    cfg->bc_objlist = cfg->bc_objlist->co_next;

  if(cfg->bc_objlist)
  {
    for(oi = cfg->bc_objlist; oi->co_next; )
      if(oi->co_next->co_isBackend)
        oi->co_next = oi->co_next->co_next;
      else
        oi = oi->co_next;
    cfg->bc_objlist_last = oi;
  } else
    cfg->bc_objlist_last = NULL;

  /* Go through dependency graph adding in order the backends */
  for(wt = 0; wt < cfg->bc_deps->g_nnodes; wt++)
    for(i = 0; i < cfg->bc_deps->g_nnodes; i++)
      if(cfg->bc_deps->g_nodes[i]->gn_weight == wt)
      {
        cfg->bc_objlist_last->co_next = cfg->bc_deps->g_nodes[i]->gn_object;
        cfg->bc_objlist_last          = cfg->bc_deps->g_nodes[i]->gn_object;
      }
  cfg->bc_objlist_last->co_next = NULL;

  return 0;
}

/*****************************************************************************
 * static int merge_commands(BECFG_OBJ_TRANSF *trnsf,
 *                           BECFG_CONFIG *ctrg,
 *                           BECFG_CONFIG *csrc)
 *
 * Description:
 *   Add commands of 'csrc' to 'ctrg' config.
 *
 * Parameters:
 *   trnsf - equivalence table between objects of both configurations
 *   ctrg  - target configuration
 *   csrc  - source configuration
 *
 * Returns:
 *   0 if all ok, a negative value otherwise.
 *
 *****************************************************************************/

static int merge_commands(BECFG_OBJ_TRANSF *trnsf,
                          BECFG_CONFIG *ctrg,
                          BECFG_CONFIG *csrc)
{
  BECFG_INTERCEPT *r;
  BECFG_OBJ_INFO *object, *backend;

  for(r = csrc->bc_rdeflist; r; r = r->ci_next)
  {
    /* Get the equivalent object in 'ctrg' */
    if(r->ci_object)
    {
      if((object = getTrgObject(trnsf, r->ci_object)) == NULL)
      {
        _pdi_error(THIS, "I have not found object '%s' in final object list.",
                   _pdi_becfg_getObjectName(r->ci_object));
        return -1;
      }
    } else
      object = NULL;

    /* Get the equivalent backend in 'ctrg' */
    if((backend = getTrgObject(trnsf, r->ci_backend)) == NULL)
    {
      _pdi_error(THIS, "I haven't found backend '%s' in final object list.",
                 _pdi_becfg_getObjectName(r->ci_backend));
      return -1;
    }

    /* Copy this interposition in 'ctrg' */
    if(_pdi_becfg_addCommand(ctrg,
                             r->ci_type,
                             object, r->ci_symbol,
                             backend, r->ci_wrapper))
    {
      _pdi_error(THIS, "Error while adding a new interposition.");
      return -1;
    }
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_becfg_mergeBackendConfigFiles(BECFG_CONFIG *ctrg,
 *                                        BECFG_CONFIG *csrc)
 *
 * Description:
 *   Merge configurations 'csrc' and 'ctrg'.
 *
 * Parameters:
 *   ctrg  - target configuration
 *   csrc  - source configuration
 *
 * Returns:
 *   0 if all ok, a negative value otherwise.
 *
 * WARNING:
 *   If this function fails (returns a value different of zero), config 'ctrg'
 *   could be corrupted.
 *
 *****************************************************************************/

int _pdi_becfg_mergeBackendConfigFiles(BECFG_CONFIG *ctrg, BECFG_CONFIG *csrc)
{
  BECFG_OBJ_TRANSF *trnsf;

  /* If the backend dependency graph is not yet built, then build it now! */
  if(build_deps_graph(ctrg))
  {
    _pdi_error(THIS, "Cannot build dependency graph of '%s'.", ctrg->bc_name);
    return -1;
  }

  /* Try to merge all objects in 'ctrg'. At the same time check that these  */
  /* files are not contradictory and build  a conversion table (trnsf) that */
  /* allow to associate an object in 'csrc' with its equivalent in 'ctrg'   */
  if((trnsf = mergeObjects(ctrg, csrc)) == NULL)
  {
    _pdi_error(THIS, "Cannot merge objects lists.");
    return -1;
  }

  /* Apply dependencies in 'csrc' on config 'ctrg' */
  if(apply_backend_order(trnsf, ctrg, csrc))
  {
    _pdi_error(THIS, "Error when applying '%s' on graph of '%s'.",
               csrc->bc_name, ctrg->bc_name);
    free(trnsf);
    return -1;
  }

  /* Assign weights to graph nodes and check for cycles */
  if(assign_weights(ctrg->bc_deps))
  {
    _pdi_error(THIS,
               "Found a dependency cycle in '%s' after adding '%s' dependencies.",
               ctrg->bc_name, csrc->bc_name);
    free(trnsf);
    return -1;
  }

  /* Sort backends in the order stablished by the dependency graph */
  if(sort_backends(ctrg))
  {
    _pdi_error(THIS, "Error while sorting backends.");
    free(trnsf);
    return -1;
  }

  /* merge commands of both configs */
  if(merge_commands(trnsf, ctrg, csrc))
  {
    _pdi_error(THIS, "I cannot merge interposition commands.");
    free(trnsf);
    return -1;
  }

  free(trnsf);
  return 0;
}

