/*****************************************************************************
 * $Id: pdiconfig.h,v 1.1 2005/07/14 19:42:23 killabyte Exp $
 *
 * Header file that defines the API used to read/get pDI-Tools config from
 * several sources (config files, environment variables, LDAP directories, SQL
 * data bases, ...).
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#ifndef __PDI_PDICONFIG_H__
#define __PDI_PDICONFIG_H__

#ifdef __cplusplus
extern "C" {
#endif

#include<stdio.h>

/*****************************************************************************
 * Filenames, paths and environment variables
 *
 * PDI_CFG_DEF_FNAME
 *   Default filename of pDI-Tools config file.
 *
 * PDI_ENV_PDICFGFILE
 *   It is the environment variable that tells which pDI-Tools configuration
 *   file must be used.
 *
 * PDI_ENV_CONFIGFILE
 * PDI_ENV_RUNTIMEFILE
 *   Environment variables that specify which system (runtime) and user
 *   (config) interposition commands files must be processed.
 *
 * PDI_ENV_FEEDBACK
 *   If defined this environment variable sets maximum verbosity.
 *
 * PDI_ENV_CHAPMAN
 *   In the old DI-Tools, if defined, enable some hacks for Chapman (a
 *   computer). In pDI-Tools the presence of this environment variable only
 *   prints a warning message.
 *
 *****************************************************************************/

#define PDI_CFG_DEF_FNAME       "pdi.cfg"

#define PDI_ENV_PDICFGFILE      "DI_CFG_FILE"
#define PDI_ENV_CONFIGFILE      "DI_CONFIG_FILE"
#define PDI_ENV_DEBUG           "DI_DEBUG"
#define PDI_ENV_FEEDBACK        "DI_FEEDBACK"
#define PDI_ENV_CHAPMAN         "DI_FOR_CHAPMAN"
#define PDI_ENV_LOGFILE         "DI_LOG_FILE"
#define PDI_ENV_RUNTIMEFILE     "DI_RUNTIME_FILE"

typedef struct _tag_PDICONFIG {
  /* File where we are going to print all pDI-Tools messages */
  char          *log_filename;
  FILE          *logfile;
  
  /* verbose */
  /*   Stablish level verbosity of this tool */
  /*      0    total silence (except errors)                         */
  /*      1    only warnings and errors (default value)              */
  /*      2    cannot stop talking                                   */
  /*      3    the most annoying thing on the world (debug messages) */
  int           verbose;

  /* debug */
  /*   If enabled pDI-Tools will make a lot of extra work to verify that   */
  /*   everything is ok. We don't recommend to enable it during a true     */
  /*   instrumentation because it is slow and prints a lot of messages (it */
  /*   implies verbosity level 3).                                         */
  int           debug;

  /* be_path    */
  /* becfg_path */
  /*   These parameters specify the search paths for backends and */
  /*   interposition commands files.                              */
  char          **be_path;
  int           n_be_path;
  char          **becfg_path;
  int           n_becfg_path;

  /* lib_path   */
  /*   This parameter sets paths where pDI-Tools can find libraries (DSO's). */
  /*   Esentially is a copy from the environment variable 'LD_LIBRARY_PATH'. */
  char          **lib_path;
  int           n_lib_path;

  /* beconfig_files */
  /*   list of interposition commands files that will be processed by       */
  /*   pDItools. The first of all usually is the RUNTIME FILE, and the rest */
  /*   are always regular interposition commands files.                     */
  char          **beconfig_files;
  int           n_beconfig_files;

  /* allow_lib_as_be */
  /*   By default is not allowed to redirect one call from a DSO to another */
  /*   regular DSO (not backend). But sometimes could be interesting to     */
  /*   deactivate this restriction and use a regular DSO as a backend.      */
  /*   Enabling this parameter this type of checks are disabled.            */
  int           allow_lib_as_be;

  /* donttouch_backends */
  /*   If this parameter is enabled pDI-Tools will forbid to install      */
  /*   interpositions on backends. We strongly recommend to keep this     */
  /*   option enabled.                                                    */
  /*   NOTE: If this option is enabled wildcards will not include backend */
  /*         DSO's.                                                       */
  int           donttouch_backends;

  /* donttouch_pdi */
  /*   If this option is enabled pDI-Tools DSO's (libpdi.so) will not be    */
  /*   afected by any interposition. We strongly discourage to disable this */
  /*   protection: if you disable it the armaggedon will fall over          */
  /*   pDI-Tools, your backends and your program and at the end you will    */
  /*   discover that it was only a very Bad Idea (TM).                      */
  int           donttouch_pdi;

  /* max_objects */
  /*   This parameter set the maximum number of objects that pDI-Tools can  */
  /*   handle (program binary, library plus backends). It is recommended a  */
  /*   pretty high number, but not too much high to avoid an useless memory */
  /*   waste.                                                               */
  int           max_objects;

  /* max_threads */
  /*   (ONLY HAS EFFECT IF CALLBACK INTERPOSITIONS ARE USED)              */
  /*   The parameter 'max_threads' set the maximum quantity of threads    */
  /*   that pDI-Tools can instrument at the same time. It is recommended  */
  /*   a high limit because if it is exceeded the program execution will  */
  /*   be aborted. This parameter set the number of entries of a table of */
  /*   (sizeof(void *) * max_threads) bytes.                              */
  /*                                                                      */
  /*   NOTE: If it is planned to not use callbacks nor threads we         */
  /*         recommend to set this parameter to 0.                        */
  int           max_threads;

  /* num_threads */
  /*   (ONLY HAS EFFECT IF CALLBACK INTERPOSITIONS ARE USED)                 */
  /*   It cannot be larger than 'max_threads'. This parameter indicates how  */
  /*   many threads are expected to be used in the instrumented app at the   */
  /*   same time. In fact will be initialized and allocated 'num_threads'    */
  /*   structures during pDI-Tools initialization, and not during program    */
  /*   execution, reducing impact on performance.                            */
  /*                                                                         */
  /*   Accepted values:                                                      */
  /*                                                                         */
  /*        -1   Initialize all threads ('max_threads').                     */
  /*         0   Don't initialize any thread. The structures will be created */
  /*             during program instrumentation.                             */
  /*        +x   Initialize only x threads and the rest, to 'max_threads',   */
  /*             will be initialized during program execution, if needed.    */
  int           num_threads;

  /* cb_allow_handler */
  /*   (ONLY HAS EFFECT IF CALLBACK INTERPOSITIONS ARE USED)               */
  /*   Boolean. If it is true (I don't recommend that), is allowed to      */
  /*   install your own "generic wrapper". Generally this is a bestiality, */
  /*   so this flag is disabled by default. As always I let the user to    */
  /*   enable this "feature" if he/she wants, so he/she can't suicide      */
  /*   comfortably.                                                        */
  int           cb_allow_handler;

  /* cb_max_stubs */
  /*   (ONLY HAS EFFECT IF CALLBACK INTERPOSITIONS ARE USED)                */
  /*   This parameter sets stub quantity limit in memory. If we are going   */
  /*   to install a callback interposition on an object that calls to X     */
  /*   different functions and another object that calls to Y functions,    */
  /*   you must set this parameter to a value equal or bigger than (X + Y). */
  /*   If there are not sufficient stubs will be impossible to install all  */
  /*   callback interpositions.                                             */
  int           cb_max_stubs;
  
  /* cb_stack_size */
  /*   (ONLY HAS EFFECT IF CALLBACK INTERPOSITIONS ARE USED)              */
  /*   For each thread that is being instrumented with a callback         */
  /*   interposition exists one callback thread. Each of this callback    */
  /*   threads needs a memory zone (stack) where leave some data          */
  /*   temporally. With this parameter the user can set the stack size of */
  /*   each thread. The default value usually works well, but if you have */
  /*   problems instrumenting a threaded application using callbacks, try */
  /*   to adjust it to greater values. If you continue having problems    */
  /*   with infinite values perhaps you have a recursive interposition.   */
  int           cb_stack_size;

  /* no_check_on_config */
  /*   Boolean parameter. If enabled does not check the existence of    */
  /*   backends and other DSO's while building an interposition config. */
  /*   If some object doesn't exist will be discovered while installing */
  /*   interpositions.                                                  */
  int           no_check_on_config;
} PDI_CONFIG;

/*****************************************************************************
 * _pdi_configuration
 *
 *   Global structure with pDI-Tools configuration. Is recommend to use it
 *   throught its alias PDICFG to get a more compact and readable code.
 *
 *****************************************************************************/

extern PDI_CONFIG _pdi_configuration;
#define PDICFG  _pdi_configuration

int _pdi_loadConfiguration(void);
void _pdi_unloadConfiguration(void);

#ifdef __cplusplus
}
#endif

#endif
