/*****************************************************************************
 * $Id: types.h,v 1.3 2005/07/23 10:51:18 killabyte Exp $
 *
 * This file defines platform dependant structures and data types used by this
 * Elf Backend.
 *
 *       NOTE: This file must not be included directly from any program. You
 *             must use this file through header file '/include/ebeif.h'
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#ifndef __PDI_LINUX_TYPES_H__
#define __PDI_LINUX_TYPES_H__

#include<elf.h>
#include<link.h>

/*****************************************************************************
 * Check that this file have been included from '/include/ebeif.h'. If not we
 * will emit an error as big as a pine.
 *****************************************************************************/

#ifndef __PROCESSING_EBEIF_H__
#error "Don't use directly this file. Use instead <ebeif.h>"
#endif

/*****************************************************************************
 * MACROS
 *
 * ElfW(type)
 * ELFW(macro)
 *   These macros are used to simplify access to ELF structures and macros.
 *   Using this macros the programmer does not need to know if he is working in
 *   32 or 64 bits mode. For instance, if we write ElfW(Rela), the macro ElfW()
 *   will translate it to the type 'Elf32_Rela' or 'Elf64_Rela', according to
 *   the current platform.
 *
 *   NOTE that we check that these macros have not been declared previously. In
 *   Linux this macros are used widely.
 *
 * ELFW_R_SYM
 * ELFW_R_TYPE
 * ELFW_R_INFO
 *   Having 'ElfW()' and 'ELFW()' macros these macros does not seem too much
 *   necessary, but sometimes will make easier to read pDI-Tools code. For
 *   instance, take a look to this expression:
 *
 *     ELFW(R_SYM)(((ElfW(Rel) *) pos)->r_info)
 *
 *   If the programmer is a beginner probably will die when seeing that horror.
 *   With these macros we can write the same expression in a prettier way:
 *
 *     ELFW_R_SYM(((ElfW(Rel) *) pos)->r_info)
 *
 *   ...which is clearly clearer.
 *
 *****************************************************************************/

#if (PLATFORM_ASM == PDI_ASM_PPC)
# define PDI_LINUX_WORDSIZE  __WORDSIZE
#endif

#ifndef ElfW
#define ElfW(type)      _ElfW (Elf, __ELF_NATIVE_CLASS, type)
#define _ElfW(e,w,t)    _ElfW_1 (e, w, _##t)
#define _ElfW_1(e,w,t)  e##w##t
#endif

#ifndef ELFW
#define ELFW(macro)     _ELFW (ELF, __ELF_NATIVE_CLASS, macro)
#define _ELFW(e,w,t)    _ELFW_1 (e, w, _##t)
#define _ELFW_1(e,w,t)  e##w##t
#endif

#define ELFW_R_SYM      ELFW(R_SYM)
#define ELFW_R_TYPE     ELFW(R_TYPE)
#define ELFW_R_INFO     ELFW(R_INFO)

/*****************************************************************************
 * Platform configuration
 *
 *   Now is choosed what type of structures, software versions, etc. are used
 *   and supported in this system. For instance, in x86 (ia32) is used and
 *   ElfW(Rel) always, but in Itanium (ia64) is used a ElfW(Rela) structure.
 *
 *****************************************************************************/

/* ELF structures used */
#if (PLATFORM_ASM == PDI_ASM_IA32)
# define LINUX_REL                      ElfW(Rel)
# define LINUX_PLTREL                   ElfW(Rel)
# define LINUX_PLTREL_DT                DT_REL
# define LINUX_R_JMPREL_TYPE            R_386_JMP_SLOT
# define LINUX_R_JMPREL_TYPE_STR        "R_386_JMP_SLOT"
#elif (PLATFORM_ASM == PDI_ASM_IA64)
# define LINUX_REL                      ElfW(Rela)
# define LINUX_PLTREL                   ElfW(Rela)
# define LINUX_PLTREL_DT                DT_RELA
# define LINUX_R_JMPREL_TYPE            R_IA64_IPLTLSB
# define LINUX_R_JMPREL_TYPE_STR        "R_IA64_IPLTLSB"
#elif (PLATFORM_ASM == PDI_ASM_PPC)
# define LINUX_REL                      ElfW(Rela)
# define LINUX_PLTREL                   ElfW(Rela)
# define LINUX_PLTREL_DT                DT_RELA
# if PDI_LINUX_WORDSIZE == 32
#  define LINUX_R_JMPREL_TYPE           R_PPC_JMP_SLOT
#  define LINUX_R_JMPREL_TYPE_STR       "R_PPC_JMP_SLOT"
# else
#  define LINUX_R_JMPREL_TYPE           R_PPC64_JMP_SLOT
#  define LINUX_R_JMPREL_TYPE_STR       "R_PPC64_JMP_SLOT"
# endif
#else
# error No se como configurar esta plataforma (PLATFORM_ASM)
#endif

/* _r_debug minimum and maximum protocol version supported */
#define LINUX_R_DEBUG_VER_MIN   1
#define LINUX_R_DEBUG_VER_MAX   1

/* GNU libc minimum and maximum version supported */
#define LINUX_LIBC_VMAJ_MIN     2
#define LINUX_LIBC_VMAJ_MAX     2
#define LINUX_LIBC_VMIN_MIN     2
#define LINUX_LIBC_VMIN_MAX     3
#define LINUX_LIBC_VREV_MIN     0
#define LINUX_LIBC_VREV_MAX     999
#define LINUX_LIBC_VABS(x,y,z)  (((((x) * 1000) + (y)) * 1000) + (z))
#define LINUX_LIBC_MINVER       LINUX_LIBC_VABS(LINUX_LIBC_VMAJ_MIN, \
                                                LINUX_LIBC_VMIN_MIN, \
                                                LINUX_LIBC_VREV_MIN)
#define LINUX_LIBC_MAXVER       LINUX_LIBC_VABS(LINUX_LIBC_VMAJ_MAX, \
                                                LINUX_LIBC_VMIN_MAX, \
                                                LINUX_LIBC_VREV_MAX)

/*****************************************************************************
 * struct _tag_PDI_LINUX_ELFOBJ PDI_LINUX_ELFOBJ
 * struct _tag_PDI_LINUX_INTERCEPT PDI_LINUX_INTERCEPT
 *
 *   Architecture specific information for the structures PDI_ELFOBJ and
 *   PDI_INTERCEPT.
 *
 * PDI_ARCH_ELFOBJ
 * PDI_ARCH_INTERCEPT
 *   These two alias are for the structures PDI_LINUX_ELFOBJ and
 *   PDI_LINUX_INTERCEPT. Using this alias we can use this structures from
 *   platform independant code.
 *
 *****************************************************************************/

typedef struct _tag_PDI_LINUX_ELFOBJ {
  /* a pointer to the runtime linker data about this object */
  struct link_map  *lm;

  /* broken dso */
  /* From kernel 2.6.x appears a new element in the DSO list that has not   */
  /* name and smells bad: doesn't have important sections, does not follow  */
  /* correctly ELF standard, and it has no decency... and the worse: if we  */
  /* try to manipulate it, pDI-Tools will fail surely. To avoid this object */
  /* and similar cases we mark these DSO with this flag.                    */
  int              broken_dso;

  /* direct pointers to basic things and prefetched values */
  ElfW(Addr)       stringTable;
  ElfW(Word)       stringTableSize;
  ElfW(Addr)       symbolTable;
  ElfW(Word)       symbolSize;
  ElfW(Addr)       hashTable;
  ElfW(Addr)       gotTable;

  /* Relocation Tables */
  /* (Depending on platform group REL or RELA will be used) */
  ElfW(Addr)       relTable;
  ElfW(Word)       relTableSize;
  ElfW(Word)       relEntSize;
  ElfW(Addr)       relaTable;
  ElfW(Word)       relaTableSize;
  ElfW(Word)       relaEntSize;

  /* DT_JMPREL is special. It contains ONLY function relocations */
  /* (Relocation type used in this table (REL o RELA) is defined by pltRel */
  ElfW(Addr)       jmpRelTable;
  ElfW(Word)       jmpRelTableSize;
  ElfW(Word)       pltRel;          /* This can be DT_REL or DT_RELA */
  ElfW(Word)       pltRelEntSize;   /* Depends on 'pltRel'           */

  /* Info about hash table */
  Elf_Symndx       n_bucket, n_chain,
                   *l_bucket, *l_chain;

  /* Info about relocation table */
  int             jmpRelSortedBySymndx;
} PDI_LINUX_ELFOBJ;

typedef struct _tag_PDI_LINUX_PLT_UNDO_DATA {
#if PLATFORM_ASM == PDI_ASM_IA32
  void         **addr_addr;
  void          *undo_addr;
#elif PLATFORM_ASM == PDI_ASM_IA64
#error "I don't know how to undo changes in PLT on IA64"
#elif PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 32
  ElfW(Addr)   *addr_addr;
  ElfW(Addr)    undo_addr;
  void         *addr_data;
  char          undo_data[4*2];
#elif PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 64
  void         *addr_fd;
  ElfW(Addr)    func_desc[3];
#endif
} PDI_LINUX_PLT_UNDO_DATA;

typedef struct _tag_PDI_LINUX_INTERCEPT_RELINK {
  PDI_LINUX_PLT_UNDO_DATA undo_data;
} PDI_LINUX_INTERCEPT_RELINK;

typedef struct _tag_PDI_LINUX_INTERCEPT_REDEFINITION {
  ElfW(Addr)    symbol_st_value;
} PDI_LINUX_INTERCEPT_REDEFINITION;

typedef struct _tag_PDI_LINUX_INTERCEPT_CALLBACK {
  /* NOTE: To change the order of this things in this structure without      */
  /* knowing what are you doing is a Bad Idea (TM). The assembly code expect */
  /* to find these fields in this order. If you want to have a long time     */
  /* debugging and seeing Strange Things (TM), common... permute and play... */
  /* is funny, you'll see...                                                 */
  /* <DONT_TOUCH> */
  void          **undo_table;
  void          *beCallbackRequired;
  void          *bePreEventCallback;
  void          *bePostEventCallback;
  /* </DONT_TOUCH> */
} PDI_LINUX_INTERCEPT_CALLBACK;

typedef struct _tag_PDI_LINUX_INTERCEPT {
  /* <DONT_TOUCH> */
  /* Parts of this union are accesed from asembly programs in some archs. */
  /* Is highly recommended to mantain this union at the begining of this  */
  /* structure. Highly, very highly, sure.                                */
  union {
    PDI_LINUX_INTERCEPT_RELINK       relink;
    PDI_LINUX_INTERCEPT_REDEFINITION redefinition;
    PDI_LINUX_INTERCEPT_CALLBACK     callback;
  } u;
  /* </DONT_TOUCH> */
} PDI_LINUX_INTERCEPT;

typedef PDI_LINUX_ELFOBJ    PDI_ARCH_ELFOBJ;
typedef PDI_LINUX_INTERCEPT PDI_ARCH_INTERCEPT;

#endif
