/*****************************************************************************
 * $Id: beconfig.h,v 1.2 2005/07/14 19:58:37 killabyte Exp $
 *
 * This include file export types and functions that allow the programmer to
 * use and manage interposition commands files and configurations. This API
 * simplifies the creation, checking and use of new configurations during
 * runtime.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#ifndef __PDI_BECONFIG_H__
#define __PDI_BECONFIG_H__

#ifdef __cplusplus
extern "C" {
#endif

#include<stdio.h>
#include<ebeif.h>

/*****************************************************************************
 * ABOUT A INTERPOSITION COMMANDS FILE
 * ===================================
 *
 * NOTE: For happines of the reader sometimes we'll use Perl regexp's to
 *       specify syntax of these files.
 *
 * Structure and syntax
 * ====================
 *
 *   A file of interposition commands has this structure:
 *
 *     {module_list}
 *     # commands
 *     {command_list}
 *
 *   We can see two parts: 'module_list' and 'command_list'. Both are separed
 *   by the directive '#commands'.
 *
 *
 *   The first part has three objectives:
 * 
 *     - Declare all backends and objects (DSO's) that will be used or affected
 *       by this configuration file.
 *     - Assign a tag (alias) to each DSO, if desired
 *     - Mark which objects are backends and which are regular shared objects.
 *
 *   Each line of this part is a DSO declaration with this syntax:
 *
 *     [#backend|#define|#object] [alias] <path_to_object>
 *
 *   where 'alias' =~ /^[a-zA-Z0-9_-]+$/
 *
 *   If we want to declare a regular object we must use the directive '#define'
 *   or '#object'. This directive could be followed, if desired, by an alias to
 *   the object, and must be finished by a path to the object on disk. The
 *   alias could be used later by the interposition commmands.
 *
 *   To declare a backend we should use the directive '#backend' or begin
 *   directly with the path or alias to the backend, but we encourage to use
 *   this directive always. You can assign an alias to the backend if you want.
 *
 *   We strongly recommend to use alias always. It allows to make config files
 *   easier to mantain and read.
 *
 *   Note that the 'module_list' stablish the order in which we want backends
 *   to be loaded.
 *   
 *   
 *   The second part is a list of pirindolos (commands) which stablish the
 *   interpositions.
 *
 *   Each line of text of this part is a pirindolo with this syntax:
 *
 *     <type> <object|*> <function|*> <backend> [wrapper]
 *
 *     where: type =~ /^[CDFR]$/
 *            object is a path to the object or an alias or a wildcard.
 *            function is a function name or a wildcard.
 *            backend is a path to a object, an alias or a wildcard.
 *            wrapper can be nothing (in case of callbacks) or a backend's
 *              function.
 *
 *   If a wildcard is used, will affect
 *     - to all instrumentable objects if used as 'object'
 *     - to all functions of an object if used as 'function'
 *
 *   Some interpositions, as definition, are incompatible with wildcards on
 *   some fields.
 *
 *   Wildcards are dangerous because they are difficult to control. For
 *   instance, this is a Bad Idea (TM):
 *
 *     C * * BACKEND
 *
 *   This interposition uses a lot of memory an it is very difficult to
 *   control.
 *
 * About aliases
 * =============
 *
 *   We can assign an alias to each object. This allow us to write easier to
 *   read and mantain command files. For instance, if we include this line
 *   in a config file:
 *
 *     #backend IOBACKEND /usr/local/ditools/lib/backends/iobe.so
 *
 *   Later, in section 'commands', we can refer to this backend by its alias:
 *
 *     F MAIN read IOBACKEND read_wrapper
 *
 *   But there are some reserved and predefined alias:
 *
 *     LIBC
 *       Is an alias to the C Library DSO, usually '/lib/libc.so'.
 *     MAIN
 *       Is the DSO of the program being instrumented
 *     PDI
 *       This alias is assigned to 'libpdi.so', or in other words, to pDI-Tools
 *       DSO.
 *
 *****************************************************************************/

/*****************************************************************************
 * Definiciones
 *
 * BECFG_ANON_FILE
 *   This define is a string for naming config files readed directly from a
 *   file descriptor.
 *
 * BECFG_MAX_LINE
 *   It is the maximum legal length of a text line in a config file. If this
 *   limit is exceeded pDI-Tools will print an error and the megaline will be
 *   ignored.
 *
 * BECFG_IT_CALLBACK
 * BECFG_IT_RELINK
 * BECFG_IT_REDEFINITION
 *   These constants are the different types of interpositions that pDI-Tools
 *   recognizes.
 *
 *****************************************************************************/

#define BECFG_ANON_FILE         "<stream>"
#define BECFG_MAX_LINE          1024

#define BECFG_IT_RELINK         PDI_IT_RELINK
#define BECFG_IT_REDEFINITION   PDI_IT_REDEFINITION
#define BECFG_IT_CALLBACK       PDI_IT_CALLBACK

/*****************************************************************************
 * Data structures
 *
 * BECFG_OBJ_INFO (struct _tag_BECFG_OBJ_INFO)
 *   It contains information about a backend or a regular dynamic shared
 *   object. There are in fact only two important things: the path to the DSO
 *   and its alias (if it have assigned an alias).
 *
 * BECFG_INTERCEPT (struct _tag_BECFG_INTERCEPT)
 *   It groups all the data necessary to form a interposition, that is to say,
 *   the object that are instrumented (NULL in case of a wildcard) and the
 *   function that is instrumented (NULL in case of a wildcard). As well, we
 *   also kept a reference to the BACKEND in charge to interpose and the
 *   function of the same that will be interposed.
 *
 * BECFG_CONFIG (struct _tag_BECFG_CONFIG)
 *   Basically it contains two tables: one of structures 'BECFG_INTERCEPT' and
 *   another one of structures 'BECFG_OBJ_INFO', plus some information of
 *   control. It is a reflection of a interposition command file in memory.
 *
 * BECFG_GRAPH
 * BECFG_GRAPH_NODE
 *   This structures only are used in a interposition commands file when is
 *   combined with another config file.
 *
 *   The structures  BECFG_GRAPH and BECFG_GRAPH_NODE are used to build a graph
 *   where are shown all dependencies between the backends used in a
 *   interposition commands file. Dependencies are calculated in this way:
 *
 *     - if in a interposition commands file are used 2 o more backends, the
 *       order of appearence of them stablish the dependencies between them.
 *       So, if B appears after A, then B depends on A.
 *
 *     - between different interposition commands file doesn't exist
 *       dependencies. So if a interposition commands file uses backends A and
 *       B (in this order) and in the file uses C and D, B depends on A and D
 *       depends on C. But C nor D depends on A or B.
 *
 *****************************************************************************/

typedef struct _tag_BECFG_OBJ_INFO {
  /* Information about the object */
  /*   This info is the path to the object and, if it has, its alias. There  */
  /*   is also a flag that says if this object is a backend or a regular DSO */
  char                  *co_path;
  char                  *co_alias;
  int                   co_isBackend;

  /* the line number where this object was declared */
  int                   co_lineno;

  /* As soon as possible we cache here the true info about this object */
  PDI_ELFOBJ            *co_eo;

  /* A pointer to the next object declared in this config */
  struct _tag_BECFG_OBJ_INFO *co_next;
} BECFG_OBJ_INFO;

typedef struct _tag_BECFG_INTERCEPT {
  BECFG_OBJ_INFO        *ci_object;
  char                  *ci_symbol;
  BECFG_OBJ_INFO        *ci_backend;
  char                  *ci_wrapper;
  char                  ci_type;

  /* the line number where this interposition was declared */
  int                   ci_lineno;

  /* the next command in this config */
  struct _tag_BECFG_INTERCEPT *ci_next;
} BECFG_INTERCEPT;

typedef struct _tag_BECFG_GRAPH_NODE {
  BECFG_OBJ_INFO        *gn_object;

  int                   gn_weight;
  int                   gn_hasPar;

  int                   gn_nto;
  struct _tag_BECFG_GRAPH_NODE **gn_to;
} BECFG_GRAPH_NODE;

typedef struct _tag_BECFG_GRAPH {
  BECFG_GRAPH_NODE      **g_nodes;
  int                   g_nnodes;
} BECFG_GRAPH;

typedef struct _tag_BECFG_CONFIG {
  /* the filename of this config */
  char                  *bc_name;

  /* info for file reading */
  int                   bc_lineno;
  int                   bc_lastChar;
  char                  *bc_buffer;
  int                   bc_buffSize;
  FILE                  *bc_f;

  /* lists of things readed from the config */
  BECFG_OBJ_INFO        *bc_objlist;
  BECFG_OBJ_INFO        *bc_objlist_last;
  BECFG_INTERCEPT       *bc_rdeflist;
  BECFG_INTERCEPT       *bc_rdeflist_last;

  /* info about dependencies between backends (if available) */
  BECFG_GRAPH           *bc_deps;
} BECFG_CONFIG;

BECFG_CONFIG *_pdi_becfg_newConfig(char *name);
void _pdi_becfg_destroyConfig(BECFG_CONFIG *cfg);

int _pdi_becfg_initializeRead(BECFG_CONFIG *cfg, FILE *f);
void _pdi_becfg_finalizeRead(BECFG_CONFIG *cfg);

BECFG_OBJ_INFO *_pdi_becfg_getObjectByPath(BECFG_CONFIG *cfg, char *path);
BECFG_OBJ_INFO *_pdi_becfg_getObjectByAlias(BECFG_CONFIG *cfg, char *alias);
BECFG_OBJ_INFO *_pdi_becfg_getObject(BECFG_CONFIG *cfg, char *x);
char *_pdi_becfg_getObjectName(BECFG_OBJ_INFO *object);
int _pdi_becfg_setObjectAlias(BECFG_OBJ_INFO *object, char *new_alias);
BECFG_OBJ_INFO *_pdi_becfg_addObject(BECFG_CONFIG *cfg,
                                     char *o_path, char *o_alias,
                                     int is_backend);
int _pdi_becfg_addCommand(BECFG_CONFIG *cfg,
                          int type,
                          BECFG_OBJ_INFO *object, char *symbol,
                          BECFG_OBJ_INFO *backend, char *wrapper);


BECFG_CONFIG *_pdi_becfg_loadBackendConfigFromFile(FILE *f);
BECFG_CONFIG *_pdi_becfg_loadBackendConfig(char *fname);
int _pdi_becfg_applyBackendConfig(BECFG_CONFIG *cfg);
int _pdi_becfg_mergeBackendConfigFiles(BECFG_CONFIG *ctrg, BECFG_CONFIG *csrc);
void _pdi_becfg_showObjects(int level, BECFG_CONFIG *cfg);

#ifdef __cplusplus
}
#endif

#endif
