/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef PAPYRUSRGBA_H
#define PAPYRUSRGBA_H

#include <ostream>

#include <sigc++/sigc++.h>

namespace Papyrus {

/**
 * This class encapsulates an RGBA color with each element stored internally
 * as an IEEE double precision floating point value ranging from 0.0 to 1.0.
 *
 * Although this class provides a convenient mechanism for treating a group
 * of doubles as an RGBA value, additional methods are provided to allow
 * conversion between RGB and HLS (Hue, Light and Saturation) and HSV (Hue,
 * Saturation and Value).
 *
 * HLS and HSV calculations are implemented as methods and are calculated on
 * each call, since manipulating these values are probably the uncommon case
 * and storing them internally would require additional storage overhead for
 * an uncommon case.
 *
 * @author Rick L. Vinyard, Jr.
*/
class RGBA {
public:
    RGBA();
    RGBA(int r, int g, int b, int a=255);
    RGBA(double r, double g, double b, double a=1.0);
    RGBA(double rgba[4]);
    RGBA(const RGBA* other);

    void get_rgba(double rgba[4]) const;
    void get_rgba(float rgba[4]) const;
    void get_rgba(double& r, double& g, double& b, double& a) const;
    void get_rgba(float& r, float& g, float& b, float& a) const;

    void set_rgba(float rgba[4]);
    void set_rgba(double r, double g, double b, double a=1.0);

    void get_hls(double& h, double& l, double& s);
    void set_hlsa(double h, double l, double s, double a=1.0);

    void get_hsv(double& h, double& s, double& v);
    void set_hsva(double h, double s, double v, double a=1.0);

    double operator[](int index);
    bool operator==(const RGBA other) const;
    bool operator!=(const RGBA other) const;
    bool operator<(const RGBA other) const; // required for various STL containers, array is cast as guint32 and compared
    RGBA& operator=(const RGBA& other);

    virtual sigc::signal<void>& signal_changed() { return m_signal_changed; }

  protected:
    sigc::signal<void> m_signal_changed;


protected:
    double m_rgba[4];

protected:
    void rgb_to_hls(double r, double g, double b,
                    double& h, double& l, double& s);
    void hls_to_rgb(double h, double l, double s,
                    double& r, double& g, double& b);
    double Value(double n1, double n2, double hue);

    void rgb_to_hsv(double r, double g, double b,
                    double& h, double& s, double& v);
    void hsv_to_rgb(double h, double s, double v,
                    double& r, double& g, double& b);

};


std::ostream& operator<<(std::ostream& out, RGBA color);

}; // namespace Papyrus

#endif
