/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef PAPYRUSTRANSFORMABLE_H
#define PAPYRUSTRANSFORMABLE_H

#include <papyrus/renderable.h>
#include <papyrus/bbox.h>

namespace Papyrus {

/**
 * This is the base class for objects capable of being drawn onto a cairo.
 *
 * This class provides a common interface for getting and setting the
 * x and y translations, x and y scalings and rotation angle.
 *
 * Additionally, this class provides attributes for storing the affine
 * transformation matrix to be applied prior to rendering.
 *
 * This class implements the render method required of the parent
 * renderable by pushing the current cairo state onto the cairo
 * stack, applying any necessary transformation matrix, calling
 * the pure virtual draw method, and popping the cairo stack to
 * restore the state.
 *
 * Since this class handles the matrices directly, children only
 * need to override the draw method and draw themselves with a
 * local coordinate frame reference.
 *
 * @author Rick L Vinyard Jr
 */
class Drawable : public Renderable
{
public:
    /**
     * Constructor that sets initial transforms.
     *
     * If no parameters are specified, initial values set translation to (0,0) with a (1,1) scale
     * ratio and no rotation.
     *
     * Constructor that sets scale to 1.0 and 1.0 initially.
     *
     * Also creates propertymm references for member attributes. If either scale parameter < 0.0,
     * the parameter will be reset to 1.0.
     *
     * Properties:
     *    scale_x : scaling in the x dimension
     *    scale_y : scaling in the y dimension
     */
    Drawable(double x=0.0, double y=0.0, double sx=1.0, double sy=1.0, double r=0.0);

    virtual ~Drawable();

    double get_x();
    double get_y();
    void get_xy(double& x, double& y);

    int set_x(double x);
    int set_y(double y);
    int set_xy(double x, double y);

    int translate_x(double tx);
    int translate_y(double ty);

    /**
     * Translate current position by (tx, ty)
     *
     * If previous position was (px,py) new position is (px+tx, py+ty).
     */
    int translate(double tx, double ty);

    /** returns the current value of the x scaling */
    double get_scale_x();

    /** returns the current value of the y scaling */
    double get_scale_y();

    /** convenience method to get scaling in x and y dimensions with a single function call */
    void get_scale_xy(double& scale_x, double& scale_y);

    /**
     * sets scaling in the x dimension and emits changed signal with parameter SCALE_X
     *
     * @param scale_x The new x dimension scaling; must be >= 0.0 or no change will occur
     * @return SCALE_X if parameter is different and scale is actually changed or 0 if scale remains unchanged
     *
     * Although it adds a little more overhead than setting the value directly, this method checks to ensure
     * that a value change has actually taken place before calling the on_scale_changed method since needlessly
     * handling changes that haven't really occurred may add even more overhead.
     */
    int set_scale_x(double scale_x);

    /**
     * sets scaling in the y dimension and emits changed signal with parameter SCALE_Y
     *
     * @param scale_y The new y dimension scaling; must be >= 0.0 or no change will occur
     * @return SCALE_Y if parameter is different and scale is actually changed or 0 if scale remains unchanged
     *
     * Although it adds a little more overhead than setting the value directly, this method checks to ensure
     * that a value change has actually taken place before calling the on_scale_changed method since needlessly
     * handling changes that haven't really occurred may add even more overhead.
     */
    int set_scale_y(double scale_y);

    /**
     * sets scaling in the x and y dimensions and emits changed signal with one of parameters SCALE_X, SCALE_Y or SCALE_X|SCALE_Y
     *
     * @param scale_x The new x dimension scaling; must be >= 0.0 or no change to x scaling will occur
     * @param scale_y The new y dimension scaling; must be >= 0.0 or no change to y scaling will occur
     * @return SCALE_X, SCALE_Y or SCALE_X|SCALE_Y depending on which scale(s) actually changed or 0 if scales remain unchanged
     *
     * Although it adds a little more overhead than setting the value directly, this method checks to ensure
     * that a value change has actually taken place before calling the on_scale_changed method since needlessly
     * handling changes that haven't really occurred may add even more overhead.
     */
    int set_scale(double scale_x, double scale_y);

    /**
     * sets scaling in the x and y dimensions and emits changed signal with parameter SCALE_X|SCALE_Y
     *
     * @param s The amount to scale the x and y dimensions by; must be >= 0.0 or no change will occur
     * @return SCALE_X|SCALE_Y if scales actually changed or 0 if scales remain unchanged
     *
     * Although it adds a little more overhead than setting the value directly, this method checks to ensure
     * that a value change has actually taken place before calling the on_scale_changed method since needlessly
     * handling changes that haven't really occurred may add even more overhead.
     */
    int set_scale(double s);

    /**
     * Scales the current x dimension by s and emits changed signal with parameter SCALE_X
     *
     * @param s The amount to scale the x dimension by; must be >= 0.0 or no change will occur
     * @return SCALE_X if scaling occurs or 0 if scale remains unchanged
     */
    int scale_x(double s);

    /**
     * Scales the current y dimension by s and emits changed signal with parameter SCALE_Y
     *
     * @param s The amount to scale the y dimension by; must be >= 0.0 or no change will occur
     * @return SCALE_Y if scaling occurs or 0 if scale remains unchanged
     */
    int scale_y(double s);

    /**
     * Scales the current x and y dimensions by s and emits changed signal with parameter SCALE_X|SCALE_Y
     *
     * @param s The amount to scale the x and y dimensions by; must be >= 0.0 or no change will occur
     * @return SCALE_X, SCALE_Y or SCALE_X|SCALE_Y depending on which scale(s) actually changed or 0 if scales remain unchanged
     */
    int scale(double s);

    /**
     * Scales the current x dimension by sx and the y dimension by sy and emits changed signal with one of parameters SCALE_X, SCALE_Y or SCALE_X|SCALE_Y
     *
     * @param scale_x The amount to scale the x dimension by; must be >= 0.0 or no change will occur
     * @param scale_y The amount to scale the y dimension by; must be >= 0.0 or no change will occur
     * @return SCALE_X, SCALE_Y or SCALE_X|SCALE_Y depending on which scale(s) actually changed or 0 if scales remain unchanged
     */
    int scale(double scale_x, double scale_y);

    /**
     * Rotate to r radians
     */
    int set_rotate(double r);

    /**
     * Rotate by r radians
     *
     * If previous rotation was pr then new rotation is pr+r
     */
    int rotate(double r);

    /**
     * Reset position to default values of xy = (0,0), scale = (1,1)
     * and rotation = 0.
     */
    int reset_position();

    /**
     * Return the cairo matrix of this object with all transforms applied
     */
    const cairo_matrix_t& get_matrix();

    /**
     * Overridden render method from renderable.
     *
     * Generally, children should override the draw method rather than this
     * method.
     *
     * Performs the following actions:
     *   1. Pushes cairo state onto the stack
     *   2. Recalculates local matrix if necessary
     *   3. Applies the local matrix to the current matrix
     *   4. Calls the virtual draw method
     *   5. Restores the cairo state
     */
    virtual void render(cairo_t* cairo);

    /**
     * Pure virtual draw method to be overridden by children.
     *
     * This method is called by the supplied render method after the local
     * transformation matrix has been applied to the cairo stack.
     */
    virtual void draw(cairo_t* cairo) = 0;

//     virtual std::vector<Item*> pick(cairo_t* cairo, double x, double y, unsigned depth=1) = 0;

    /**
     * Returns the bounding box of this drawable without applying the matrix formed from the
     * affine transform methods translate*, scale* and rotate*.
     *
     * Children should modify the member attribute m_bbox as appropriate and generally shouldn't
     * need to provide a polymorphic version of this method.
     */
     virtual BBox& get_bbox();

     /**
      * Returns the bounding box of this drawable that has been transformed by the matrix formed
      * by the affine transform methods translate*, scale* and rotate*.
      *
      * Applies the local transformation matrix to the bounding box returned from the get_bbox()
      * method. The m_bbox member is not used so that the get_bbox() method may be overridden
      * by a child and this method will still work as intended without also overriding this
      * method.
      */
     virtual BBox get_bbox_transformed();

     /**
      * Returns the bounding box of this drawable that has been transformed by a combined local
      * and global matrix.
      *
      * The local matrix is applied to the supplied parameter matrix, and this transform is in
      * turn applied to the bbox returned from get_bbox().
      *
      * The m_bbox member is not used so that the get_bbox() method may be overridden
      * by a child and this method will still work as intended without also overriding this
      * method.
      */
     virtual BBox get_bbox_transformed(cairo_matrix_t& globalmatrix);

     virtual bool intersects(double x, double y);

     virtual Drawable* pick(double x, double y);

     sigc::signal<void, double, double, double, double>& signal_need_redraw() { return m_signal_need_redraw; }

     void set_pickable(bool pickable=true) { m_pickable = pickable; }

  protected:
    /**
   * Member attributes containing the current affine transforms
     */
    double m_x, m_y, m_scale_x, m_scale_y, m_rotate;
    bool m_pickable;

    /**
     * True if the cairo matrix needs to be regenerated before rendering or
     * returning from the get_matrix method.
     */
    bool m_regenerate_matrix;

    /**
     * The cairo matrix returned from get_matrix and used in rendering.
     */
    cairo_matrix_t m_matrix;

    /**
     */
    BBox m_bbox, m_bboxold;

    /**
     * Recalculates m_matrix if m_regenerate_matrix is true and sets
     * m_regenerate_matrix to false; does nothing if m_regenerate_matrix
     * is false.
     */
    void recalculate_matrix();

    sigc::signal<void, double, double, double, double> m_signal_need_redraw;

    friend class Group;

    virtual void on_changed(int i);

  private:

  /**
   * This serves as a stub to allow polymorphic behavior of the on_changed method.
   * The scale_x and scale_y properties are tied in the constructor to this callback
   * point, and this in turn calls the virtual on_scale_changed method that may be
   * reimplemented in children.
   */
    void on_changed_proxy(int i);

};

}

#endif
