/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "drawable.h"

#include <papyrus/bbox.h>

using namespace Papyrus;

Drawable::Drawable( double x, double y, double sx, double sy, double r )
        : Renderable(),
        m_x(x),
        m_y(y),
        m_scale_x(sx),
        m_scale_y(sy),
        m_rotate(r),
        m_pickable(true),
m_regenerate_matrix(true) {
    if (m_scale_x < 0.0)
        m_scale_x = 1.0;
    if (m_scale_y < 0.0)
        m_scale_y = 1.0;
}


Drawable::~Drawable() {}

const cairo_matrix_t& Drawable::get_matrix() {
    recalculate_matrix();
    return m_matrix;
}

double Drawable::get_x() {
    return m_x;
}

double Drawable::get_y() {
    return m_y;
}

void Drawable::get_xy(double& x, double& y) {
    x = m_x;
    y = m_y;
}

int Drawable::set_x(double x) {
    if (m_x != x) {
        m_x = x;
        on_changed(RENDER_X);
        return RENDER_X;
    }
    return 0;
}

int Drawable::set_y(double y) {
    if (m_y != y) {
        m_y = y;
        on_changed(RENDER_Y);
        return RENDER_Y;
    }
    return 0;
}

int Drawable::set_xy(double x, double y) {
    int which = 0;
    if (m_x != x) {
        m_x = x;
        which = RENDER_X;
    }
    if (m_y != y) {
        m_y = y;
        which |= RENDER_Y;
    }
    if (which)
        on_changed(which);
    return which;
}

int Drawable::translate_x(double tx) {
    if (tx != 0.0) {
        m_x += tx;
        on_changed(RENDER_X);
        return RENDER_X;
    }
    return 0;
}

int Drawable::translate_y(double ty) {
    if (ty != 0.0) {
        m_y += ty;
        on_changed(RENDER_Y);
        return RENDER_Y;
    }
    return 0;
}

int Drawable::translate(double x, double y) {
    int which = 0;
    if (x != 0.0) {
        m_x += x;
        which = RENDER_X;
    }
    if (y != 0.0) {
        m_y += y;
        which |= RENDER_Y;
    }
    if (which)
        on_changed(which);
    return which;
}

double Drawable::get_scale_x() {
    return m_scale_x;
}

double Drawable::get_scale_y() {
    return m_scale_y;
}

void Drawable::get_scale_xy(double& scale_x, double& scale_y) {
    scale_x = m_scale_x;
    scale_y = m_scale_y;
}

int Drawable::set_scale_x(double scale_x) {
    if (scale_x >= 0.0 && m_scale_x != scale_x) {
        m_scale_x = scale_x;
        on_changed(RENDER_SCALE_X);
        return RENDER_SCALE_X;
    }
    return 0;
}

int Drawable::set_scale_y(double scale_y) {
    if (scale_y >= 0.0 && m_scale_y != scale_y) {
        m_scale_y = scale_y;
        on_changed(RENDER_SCALE_Y);
        return RENDER_SCALE_Y;
    }
    return 0;
}

int Drawable::set_scale(double scale_x, double scale_y) {
    int which = 0;
    if (scale_x >= 0.0 && m_scale_x != scale_x) {
        m_scale_x = scale_x;
        which = RENDER_SCALE_X;
    }
    if (scale_y >= 0.0 && m_scale_y != scale_y) {
        m_scale_y = scale_y;
        which |= RENDER_SCALE_Y;
    }
    if (which)
        on_changed(which);
    return which;
}

int Drawable::set_scale(double s) {
    int which = 0;
    if (s >= 0.0) {
        if (m_scale_x != s) {
            m_scale_x = s;
            which = RENDER_SCALE_X;
        }
        if (m_scale_y != s) {
            m_scale_y = s;
            which |= RENDER_SCALE_Y;
        }
        if (which)
            on_changed(which);
    }
    return which;
}

int Drawable::scale_x(double s) {
    if (s >= 0.0) {
        m_scale_x *= s;
        on_changed(RENDER_SCALE_X);
        return RENDER_SCALE_X;
    }
    return 0;
}

int Drawable::scale_y(double s) {
    if (s >= 0.0) {
        m_scale_y *= s;
        on_changed(RENDER_SCALE_Y);
        return RENDER_SCALE_Y;
    }
    return 0;
}

int Drawable::scale(double s) {
    if (s >= 0.0) {
        m_scale_x *= s;
        m_scale_y *= s;
        on_changed(RENDER_SCALE_X|RENDER_SCALE_Y);
        return RENDER_SCALE_X|RENDER_SCALE_Y;
    }
    return 0;
}

int Drawable::scale(double scale_x, double scale_y) {
    int which = 0;
    if (scale_x >= 0.0 && m_scale_x != scale_x) {
        m_scale_x *= scale_x;
        which = RENDER_SCALE_X;
    }
    if (scale_y >= 0.0 && m_scale_y != scale_y) {
        m_scale_y *= scale_y;
        which |= RENDER_SCALE_Y;
    }
    if (which)
        on_changed(which);
    return which;
}

int Drawable::set_rotate( double r ) {
    if (m_rotate != r) {
        m_rotate = r;
        on_changed(RENDER_ROTATE);
        return RENDER_ROTATE;
    }
    return 0;
}

int Drawable::rotate( double r ) {
    m_rotate += r;
    on_changed(RENDER_ROTATE);
    return RENDER_ROTATE;
}

int Drawable::reset_position( ) {
    int which = 0;
    if (m_x != 0.0) {
        m_x = 0.0;
        which |= RENDER_X;
    }
    if (m_y != 0.0) {
        m_y = 0.0;
        which |= RENDER_Y;
    }
    if (m_scale_x != 1.0) {
        m_scale_x = 1.0;
        which |= RENDER_SCALE_X;
    }
    if (m_scale_y != 1.0) {
        m_scale_y = 1.0;
        which |= RENDER_SCALE_Y;
    }
    if (m_rotate != 0.0) {
        m_rotate = 0.0;
        which |= RENDER_ROTATE;
    }
    if (which)
        on_changed(RENDER_X|RENDER_Y|RENDER_SCALE_X|RENDER_SCALE_Y|RENDER_ROTATE);
    return which;
}

void Drawable::render( cairo_t * cairo ) {
    cairo_matrix_t oldmatrix, newmatrix;
    cairo_save(cairo);
    recalculate_matrix();
    cairo_transform(cairo, &m_matrix);
    draw(cairo);
    cairo_restore(cairo);
}

void Drawable::recalculate_matrix( ) {
    if (m_regenerate_matrix) {
        cairo_matrix_init_identity(&m_matrix);
        cairo_matrix_rotate(&m_matrix, m_rotate);
        cairo_matrix_translate(&m_matrix, m_x, m_y);
        cairo_matrix_scale(&m_matrix, m_scale_x, m_scale_y);
        m_regenerate_matrix = false;
    }
}

BBox& Drawable::get_bbox( ) {
    return m_bbox;
}

BBox Drawable::get_bbox_transformed( ) {
  cairo_matrix_t inverse_matrix;
    BBox xformed_bbox = get_bbox();

    recalculate_matrix();
    inverse_matrix = m_matrix;
    cairo_matrix_invert(&inverse_matrix);

    xformed_bbox.transform(m_matrix);

    return xformed_bbox;
}

BBox Drawable::get_bbox_transformed( cairo_matrix_t & globalmatrix ) {
    BBox xformed_bbox = get_bbox();
    cairo_matrix_t newmatrix;

    recalculate_matrix();
    cairo_matrix_multiply(&newmatrix, &m_matrix, &globalmatrix);

    xformed_bbox.transform(newmatrix);

    return xformed_bbox;
}

bool Drawable::intersects( double x, double y ) {
    return get_bbox_transformed().bounded(x, y);
}

Drawable* Drawable::pick( double x, double y ) {
  if (!m_pickable)
    return NULL;

    if (intersects(x, y))
        return this;
    else
        return NULL;
}

void Drawable::on_changed_proxy( int i ) {
    on_changed(i);
}

void Drawable::on_changed( int i ) {
    if (i & (RENDER_X | RENDER_Y | RENDER_SCALE_X | RENDER_SCALE_Y | RENDER_ROTATE | RENDER_WIDTH | RENDER_HEIGHT) ) {
        m_regenerate_matrix = true;

        double x, y, w, h;

        BBox tmpbox = get_bbox_transformed();
        BBox updatebox = m_bboxold;
        m_bboxold = tmpbox;
        updatebox.update(tmpbox);
        updatebox.get_xywh(x, y, w, h);

        m_signal_need_redraw.emit(x, y, w, h);
    }

    Renderable::on_changed(i);
}
