/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef PAPYRUSBBOX_H
#define PAPYRUSBBOX_H

#include <papyrus/object.h>

#include <cairo.h>

namespace Papyrus {

typedef enum BBoxProperties {
  BBOX_X      =1<<0,
  BBOX_Y      =1<<1,
  BBOX_WIDTH  =1<<2,
  BBOX_HEIGHT =1<<3,
} BBoxProperties;

/**
@author Rick L Vinyard Jr
*/
class BBox: public virtual Papyrus::Object {
public:

  BBox(double x=0.0, double y=0.0, double w=0.0, double h=0.0);

    ~BBox();

    double get_x();
    double get_y();
    double get_width();
    double get_height();

    double get_xywh(double& x, double& y, double& w, double& h);

    double get_left();
    double get_right();
    double get_top();
    double get_bottom();
    void get_lrtb(double& l, double& r, double& t, double& b);

    void set_x(double x);
    void set_y(double y);
    void set_width(double w);
    void set_height(double h);
    void set_xywh(double x, double y, double w, double h);


    int update(const BBox& other);
    int update(double x, double y);
    int update_x(double x);
    int update_y(double y);

    void get_left_bottom(double& l, double& b) const;
    void get_right_top(double& r, double& t) const;

    BBox& operator=(const BBox& other);

  /**
   * true if the bounding boxes are identical
   */
    bool operator==(const BBox& other) const;

    bool operator!=(const BBox& other) const { return !(*this == other); }

  /**
   * true if this bounding box is completely contained within the other bounding box
   */
    bool operator<(const BBox& other) const;
    bool operator<=(const BBox& other) const;

  /**
   * true of this bounding box is completely outside the other bounding box
   */
    bool operator>(const BBox& other) const;
    bool operator>=(const BBox& other) const;

    bool bounded(double x, double y);

    bool inside(double x, double y);

    bool boundary(double x, double y);

    void transform(const cairo_matrix_t& matrix);

protected:
  double m_x, m_y, m_w, m_h;

  void on_changed(int);

};

}

#endif
