; Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

; This file is part of Pandora, the Flexible Monitoring Platform.

; Pandora is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2, or (at your option)
; any later version.

; Pandora is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with Pandora; see the file COPYING.  If not, write to
; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
; Boston, MA 02111-1307, USA.

(use-modules (ice-9 popen))

(define pandora_debug #f)

(define scm-repl-prompt "pandora> ")

(define (string-tokenize str delim_str)
  (let ((len 		(string-length str))
	(delim_chars 	(string->list delim_str)))
    (define (_tokenize first last)
      (cond ((= last len) (if (> last first)
			      (list (make-shared-substring str first last))
			      '()))
	    ((member (string-ref str last) delim_chars) 
	     (if (>= last first)
		 (cons (make-shared-substring str first last)
		       (_tokenize (1+ last) (1+ last)))
		 (_tokenize (1+ first) (1+ last))))
	    (else (_tokenize first (1+ last)))))     
    (_tokenize 0 0)))

(define (scan-libconfig filename)
  (let ((f (open-input-file filename))
	(n '())
	(d '()))    
    (do ((buf "" (read-line f 'trim)))
	((eof-object? buf) (cons n d))
      (if (and (> (string-length buf) 8)
	       (string=? "[library]" (substring buf 0 9)))
	  (let* ((toks (cdr (string-tokenize buf " \t\r\n")))
		 (path (caddr toks))
		 (elts (string-tokenize path "/"))
		 (name (list-ref elts (1- (length elts))))
		 (core (car toks))
		 (dir  (substring path 0 (- (string-length path)
					       (string-length name)))))
	    (if core 
		(set! n (acons (string-append core ".la") dir n))))

	  (if (and (> (string-length buf) 11)
		   (string=? "[dependency]" (substring buf 0 12)))
	      (let* ((toks (cdr (string-tokenize buf " \t\r\n")))
		     (deps (map 
			    (lambda (l) 
			      (if (string=? "" l) l (string-append l ".la")))
			    (cdr toks)))
		     (core (car toks)))
		(if core 
		    (set! d (acons (string-append core ".la") deps d)))))))))

(define (search-lib name alst)
  (let* ((real (string-append "lib" name))
	 (len (string-length real)))    
    (define (_search al)
      (if (and (pair? al)
	       (pair? (car al)))
	  (let ((key (caar al)))
	    (if (and (>= (string-length key) len)
		     (string=? real (substring key 0 len)))
		(car al)
		(_search (cdr al))))
	  #f))
      (_search alst)))

(define (find-deps lib path deps)
  (let ((deplist '()))
    (define (_get_deps l)
      (or (assoc-ref deps l) '()))
    (define (_find l)
      (or (member l deplist)
	  (begin (map _find (_get_deps l))
		 (set! deplist (cons l deplist)))))
    (define (_expand lst)
      (if (pair? lst)
	  (let* ((elt (car lst))
		 (p (assoc-ref path elt)))
	    (if p
		(cons (string-append p elt) (_expand (cdr lst)))
		(_expand (cdr lst))))
	  lst))
    (and (_find lib) (reverse (_expand deplist)))))
	
(define (load-pguile)
  (define (_load file)
    (let* ((h (scan-libconfig file))
	   (n (car h))
	   (d (cdr h))
	   (p (search-lib "pguile" n))
	   (g (car p))
	   (r (cdr p)))
      (begin 
	;(display g) (newline)
	;(display n) (newline)
	;(display d) (newline)
	;(display (find-deps g n d)) (newline)
	(map (lambda(f)
	       (begin (and pandora_debug 
			   (display (string-append "loading: " f "\n")))
		      (dynamic-link f)))
	     (find-deps g n d))
	(set! %load-path (cons r %load-path)))))
  (_load (string-append
	  (or (getenv "PANDORA_HOME") ".")
	  "/"
	  (or (getenv "PANDORA_LIBCONFIG") "libconfig"))))

(load-pguile)

(define (read-file file)
  (let ((handle (false-if-exception (open-input-file file))))
    (and handle (read-delimited "" handle))))

(define (instring-eval str)
  (let ((tokens (string-tokenize str "#")))
    (define (_instring-eval toks flag)
      (if (pair? toks)
	  (cons (if flag
		    (let ((res (false-if-exception (eval-string (car toks)))))
		      (if (string? res) res ""))
		    (car toks))
		(_instring-eval (cdr toks) (not flag)))
	  toks))
    (apply string-append 
	   (map (lambda (str) (if (not (string? str))
				  ""
				  str))
		(_instring-eval tokens #f)))))

(define (infile-eval file)
  (instring-eval (read-file file)))

(define (system-string cmd)
  (let* ((port (open-input-pipe cmd))
	 (str  (read-delimited "" port))
	 (len  (string-length str))
	 (status (close-pipe port)))
    (and (= status 0) (begin (string-set! str (- len 1) #\nul) str))))

(define-module (pandora)
  :use-module (pguile)
  ;:no-backtrace
)

(if (not (feature? 'pguile))
    (scm-error 'misc-error
               #f
               "pandora module is not available in this Guile installation"
               '()
               '()))

; (use-modules (ice-9 readline))
; (and (provided? 'readline) (activate-readline))

;(display "** pandora module loaded **")
;(newline)

(export
 pandora-add-resource
 pandora-delete-resource
 pandora-set-resource-priority
 pandora-list-resources
 pandora-update-resources
 
 pandora-next-packet
 pandora-next-packet-ready?
 pandora-next-packet-nb
 pandora-connect
 pandora-close
 pandora-set-library
 pandora-get-library
 pandora-list-libraries
 pandora-set-binding
 pandora-get-binding
 pandora-list-symbols
 pandora-stop
 pandora-start
 pandora-suspend
 pandora-resume
 pandora-get-stack
 pandora-get-name
 pandora-query
 pandora-get-option
 pandora-get-option-default
 pandora-set-option
 pandora-list-options
 pandora-set-stack
 pandora-list-defined
 pandora-list-running
 pandora-alive-components
 pandora-alive-packets
 pandora-clean
 pandora-quit
 pandora-start-clock
 pandora-set-verbosity

 make-pandora-stack
 make-pandora-component
 make-pandora-macro
 make-pandora-option
 make-pandora-packet-pipe
 pandora-parse
 pandora-print
 stack:name
 set-stack:name
 stack:components
 set-stack:components
 component:name
 set-component:name
 component:type 
 set-component:type
 component:macro?
 component:options 
 set-component:options
 option:name
 set-option:name
 option:value
 set-option:value)

;; utilities

(define-public (pandora-stack-running? stk)
  (member stk (pandora-list-running)))

(define-public (pandora-restart stk)
  (let ((name (pandora-get-name stk)))
    (and (pandora-stop stk)
	 (pandora-start name))))

(define-public (pandora-start-script stack scr inter)
  (let ((stk (pandora-start stack #t)))
    (and stk (pandora-attach-script stk scr inter))))

(define-public (pandora-start-proc stack proc inter)
  (let ((stk (pandora-start stack #t)))
    (and stk (pandora-attach-proc stk proc inter))))

(define-public (pandora-attach-script stk scr inter)
  (let ((proc (eval (read (open-file scr "r")))))
    (and proc (pandora-attach-proc stk proc inter))))

(define-public (pandora-attach-proc stk proc inter)
  (while (pandora-stack-running? stk) 
	 (apply proc (list stk))
	 (usleep (* 1000 inter))))

(define-public (pandora-watch h)
  (while (member h (pandora-list-running))
	 (and (display "packets: ")
	      (display (pandora-alive-packets))
	      (display "\tcomponents: ")
	      (display (pandora-alive-components))
	      (display "\r")
	      (usleep 1000000))))

(define-public (pandora-wstart stk)
  (let ((h	(pandora-start stk #t)))
    (and h (pandora-watch h))))

(define-public (option-value->string val)
  (cond
   ((string? val)  (string-append "'" val "'"))
   ((number? val)  (number->string val))
   ((boolean? val) (if val "true" "false"))
   (else           #f)))  

(define-public (pandora-define stk-def)
  (pandora-set-stack (car (pandora-parse stk-def))))

(define-public (pandora-defstart stk-def)
  (let* ((stack (car (pandora-parse stk-def)))
	 (stk (stack:name stack)))
    (and stack
	 (pandora-set-stack stack)	 
	 (pandora-start stk #t))))

(define-public (pandora-expand-stack stk)
  (let ((complist (stack:components stk)))
    (define (_expand old)
      (if (pair? old)
	  (let ((comp (car old)))
	    (if (component:macro? comp)
		(let* ((mdef  (pandora-get-stack (component:name comp))))
		  (if mdef
		      (_expand (append (stack:components mdef) (cdr old)))
		      (cons comp (_expand (cdr old)))))
		(cons comp (_expand (cdr old)))))
	  '()))
    (set-stack:components stk (_expand complist))))

(define-public (pandora-insert-component stk comp pos)
  (let ((complist (stack:components stk)))
    (define (_insert i old)
	  (if (> i 0) 
	      (if (pair? old)
		  (cons (car old) (_insert (- i 1) (cdr old)))
		  (error "invalid position" pos))
	      (cons comp old)))
    (set-stack:components stk (_insert pos complist))))

(define-public (pandora-remove-component stk pos)
  (let ((complist (stack:components stk)))
    (define (_remove i old)
      (if (pair? old)
	  (if (> i 0) 
	      (cons (car old) (_remove (- i 1) (cdr old)))
	      (cdr old))
	  (error "invalid position" pos)))
    (set-stack:components stk (_remove pos complist))))

(define-public (pandora-split-stack stk pos)
   (let* ((clist (stack:components stk))
	  (sname (stack:name stk))
	  (chead (list-head clist pos))
	  (ctail (list-tail clist pos)))
     (list (make-pandora-stack (string-append sname "_0") chead)
	   (make-pandora-stack (string-append sname "_1") ctail))))

(define-public (pandora-add-component sname cdef pos)
  (let ((stk (pandora-get-stack sname))
	(comp  (car (pandora-parse cdef))))
    (and (pandora-insert-component stk comp pos)
	 (pandora-set-stack stk))))

(define-public (pandora-delete-component sname pos)
  (let ((stk (pandora-get-stack sname)))
    (and (pandora-remove-component stk pos)
	 (pandora-set-stack stk))))

(define-public (pandora-activate host . m)
  (false-if-exception
   (let* ((sock (socket AF_INET SOCK_DGRAM 0))
	  (addr (car (hostent:addr-list (gethost host))))
	  (port 23123)
	  (cnx  (connect sock AF_INET addr port))
	  (msg  (if (pair? m) (car m) "hello from pandora"))
	  (len  (string-length msg))
	  (buf  (make-string 1024 #\nul))
	  (sent (send sock msg))
	  (repl (if (= sent len) (and (recv! sock buf) #t) #f)))
     
     (begin (shutdown sock 2) repl))))

(define-public (pandora-launch host . cmd)
  (and (pandora-activate host)
       (pandora-connect host)
       (or (not (pair? cmd))
	   (apply (car cmd) (cdr cmd)))))

(define-public (pandora-list-components)
   (delq #f
	 (map (lambda (str)
		(let* ((len  (string-length str))
		       (cp   (pandora-get-component-prefix))
		       (clen (string-length cp)))

		  (if (or (< len clen)
			  (not (string=? cp (substring str 0 clen))))
		      #f
		      (string-downcase (substring str clen)))))
	      (pandora-list-symbols))))

(define-public (pandora-list-packets)
   (delq #f
	 (map (lambda (str)
		(let* ((len  (string-length str))
		       (cp   (pandora-get-packet-prefix))
		       (clen (string-length cp)))

		  (if (or (< len clen)
			  (not (string=? cp (substring str 0 clen))))
		      #f
		      (string-downcase (substring str clen)))))
	      (pandora-list-symbols))))

(define-public (pandora-make-packet-list p)
  (let ((pkt (pandora-next-packet-nb p)))
    (if pkt
	(cons pkt (pandora-make-packet-list p))
	'())))

(define-public (pandora-reload stk)
  (and (pandora-clean stk)
       (pandora-update-resources "stacks")))

(define-public (pandora-list-running-name)
  (map pandora-get-name (pandora-list-running)))

(define-public (pandora-stop-name stk)
  (let ((rl (pandora-list-running)))
    (define (_stop_name l)
      (if (pair? l) 
	  (let* ((h (car l))
		 (s (pandora-get-name h)))
	    (if (string=? stk s)
		(pandora-stop h)
		(_stop_name (cdr l))))
	  #f))
    (_stop_name rl)))
	

(define-public (pandora-refresh)
  (begin (pandora-update-resources "resources")
	 (pandora-update-resources "stacks")
	 (pandora-update-resources "libraries")))

(define-public (pandora-parse-prog file)
  (let ((str (infile-eval file)))
    (and str (map pandora-print (pandora-parse str)))))

; (define-public (pandora-load-stacks)
;   (and
;     (map (lambda (stk) 
; 	   (let* ((def  (pandora-get-stack stk))
; 		  (name (stack:name        def))
; 		  )
; 	     (and (display name) (display " -> ") (display def) (newline))))
; 	 (pandora-list-defined))
;     #t))

(define-public (pandora-profile stk) 
  (let* ((start (get-internal-run-time))
	 (tmp   (pandora-start stk #f))
	 (end   (get-internal-run-time)))
    (and (pandora-clean)
	 (* 10 (- end start)))))

;; testing tools

; (define-public (test-refresh stk-def1 stk-def2)
;   (let* ((stack1 (car (pandora-parse stk-def1)))
; 	 (stack2 (car (pandora-parse stk-def2)))
; 	 (stk (stack:name stack1)))
;     (and stack1
; 	 stack2
; 	 (string=? stk (stack:name stack2))
; 	 (pandora-set-stack stack1)	 
; 	 (pandora-start stk #t)
; 	 (sleep 1)
; 	 (pandora-set-stack stack2)
; 	 (sleep 1)
; 	 (pandora-stop stk))))

(define-public (test-update stk comp op val)
  (let ((my-option (car (pandora-parse opdef)))
	(sh        (pandora-start stk)))
    
    (and sh
	 (sleep 1)
	 (pandora-set-option sh comp op val) 
	 (sleep 1)
	 (pandora-stop sh))))

(define-public (test-suspend-stop stk)
  (let ((sh (pandora-start stk)))
  (and sh
       (pandora-suspend sh)
       (sleep 1)
       (pandora-resume sh)
       (pandora-stop sh))))

(define-public (test-restart stk)
  (let ((c 0))
    (while #t (let ((sh (pandora-start stk)))
		(and sh
		     (set! c (+ c 1))
		     (display c)
		     (newline)
		     (pandora-stop sh))))))


(define-public (test-loop-stop secs)
  (let ((sh (pandora-start "loop")))
    (and sh
	 (pandora-start "feed")
	 (sleep secs)
	 (pandora-stop sh))))

(define-public (test-loop-suspend secs)
  (let ((sh (pandora-start "loop")))
    (and sh
	 (pandora-start "feed")
	 (sleep secs)
	 (pandora-suspend sh)
	 (sleep secs)
	 (pandora-resume sh)
	 (pandora-start "feed")
	 (sleep secs)
	 (pandora-stop sh))))

(define-public (test-fork cmd . args)
  (let* ((comm1 (pipe))
	 (comm2 (pipe))
	 (in1   (car comm1))
	 (in2   (car comm2))
	 (out1  (cdr comm1))
	 (out2  (cdr comm2)))
    (begin
      (setvbuf out1 _IONBF)
      (setvbuf out2 _IONBF)      
      (if (= (primitive-fork) 0)
	  (begin
	    (close-input-port in2)
	    (close-output-port out1)
	    (dup->inport in1 0)
	    (dup->outport out2 1)
	    (dup->outport out2 2)
	    (apply cmd args)
	    (quit))
	  (begin
	    (close-input-port in1)
	    (close-output-port out2)
	    (cons in2 out1))))))
