/* Copyright (C) 1999, 2000, 2001, 2002 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <sys/times.h>
#include <sys/resource.h>
#include <libpandora/conf/string.h>
#include <libpandora/conf/unistd.h>
#include <libpandora/conf/snprintf.h>
#include <libpandora/conf/getopt.h>
#include <libpandora/conf/time.h>
#include <libpandora/conf/libgen.h>
}

#include <iostream>
#include <iomanip>

#include <libpandora/packet.h>
#include <libpandora/component.h>
#include <libpandora/pandora.h>
#include <libpandora/dynloader.h>
#include <libpandora/clock.h>
#include <libpandora/storage.h>
#include <libpandora/globalresourcemanager.h>
#include <libpandora/stackentry.h>
#include <libpandora/compcache.h>
#include <libpandora/error.h>
#include <libpandora/fileutil.h>
#include <libpandora/stackfactory.h>
#include <libpandora/setsignal.h>
#include <libpandora/timeval.h>
#include "libconfig.gen.h"
#include "stacks.gen.h"

static char *prog_name = NULL;

extern "C" int getopt_long (int argc, char *const *argv, const char *shortopts,
		        const struct option *longopts, int *longind);

/* Forwards */
static void usage(void);
static void version(void);
static bool set_global_option(const char *opt);
static bool display_comp_info(const text &comp);

extern "C" RETSIGTYPE pandora_cleanup(int);

#define MAX_STACKS 	512
#define MAX_COMPS 	64
#define MAX_OPTS 	64

#define STACK_ERROR	(1 << 0)
#define PACKET_ERROR	(1 << 1)
#define COMP_ERROR	(1 << 2)
#define USAGE_ERROR	(1 << 7)
#define SYSTEM_ERROR	(1 << 8)

#define USE_CLOCK 1

#if 0
#include <map>
template class __default_alloc_template<true,0>;
#endif

static int pandora_status = 0;

int main(int argc, char **argv, char *const envp[])
{
  prog_name = basename(argv[0]);
  if (strncmp(prog_name, "lt-", 3) == 0) prog_name += 3;

  //pandora_debug(prog_name);

  struct rlimit limit;
 
  // Increase addressable space
  getrlimit(RLIMIT_DATA,&limit);
  limit.rlim_cur = limit.rlim_max;
  if (setrlimit(RLIMIT_DATA,&limit)<0) {
    cerr << "setrlimit error\n";
    exit(SYSTEM_ERROR);
  }

#ifdef RLIMIT_RSS
  getrlimit(RLIMIT_RSS,&limit);
  limit.rlim_cur = limit.rlim_max;
  if (setrlimit(RLIMIT_RSS,&limit)<0) {
    cerr << "setrlimit error";
    exit(SYSTEM_ERROR);
  }
#endif

  int op;
  char 		*luri 			= NULL, 
    		*suri 			= NULL;
  char 		uri[BUFSIZ];

  char 		*opts[MAX_OPTS];
  char 		*sexp 			= NULL;
  bool 		threaded 		= true, 
                sigint			= true;
  int  		stackDump		= 0,
    		libDump 		= 0,
    		timed 			= 0;
  int 		timeout 		= 0;
  text	 	stacks[MAX_STACKS];
  stack_handle_t handles[MAX_STACKS];
  int 		nbOpts 			= 0;
  int 		nbComps			= 0;
  int 		toRun 			= 0;
  bool		printStats		= false,
    		wall			= false,
    		graph 			= false;
  int		verb   			= get_verbosity();
  text		comps[MAX_COMPS];
  bool		recover = false;
  char          *name = NULL;
  //pandora_debug(prog_name);

  char *ptr = strchr(prog_name, '-');
  if (ptr != NULL) {
    stacks[toRun].init(ptr+1);
    ++toRun;
  }

  struct option const long_options[] =
  {
    {"assign",		required_argument,	NULL,	'a'},
    {"no-sighandler",	no_argument, 		NULL, 	'c'},
    {"cache-size",	required_argument,	NULL, 	'C'},
    {"dump-stacks",	no_argument, 		NULL, 	'd'},
    {"dump-libraries",	no_argument, 		NULL, 	'D'},
    {"exec", 		required_argument, 	NULL, 	'e'},
    {"stack-uri", 	required_argument, 	NULL, 	'f'},
    {"graph", 		no_argument, 		NULL, 	'g'},
    {"help", 		no_argument, 		NULL, 	'h'},
    {"info", 		required_argument, 	NULL, 	'i'},
    {"library-uri", 	required_argument, 	NULL, 	'l'},
    {"name", 		required_argument,	NULL, 	'n'},
    {"recover", 	no_argument,		NULL, 	'r'},
    {"no-thread", 	no_argument, 		NULL, 	's'},
    {"print-stats", 	no_argument, 		NULL, 	'S'},
    {"profile", 	no_argument, 		NULL, 	't'},
    {"syslog", 	        no_argument, 		NULL, 	'u'},
    {"verbose", 	no_argument,		NULL, 	'v'},
    {"version", 	no_argument, 		NULL, 	'V'},
    {"wallclock", 	no_argument, 		NULL, 	'w'},
    {0, 		0, 			0, 	0}
  };

  while ((op = getopt_long(argc, argv, "a:cC:dDe:f:ghi:l:n:rsStuvVw", 
			   long_options, NULL)) != EOF)
    switch (op) {
    case 'a': if (nbOpts < MAX_OPTS) opts[nbOpts++] = optarg;	break;
    case 'c': sigint = false;	 	 	 	 	break;
    case 'C': CompCache::setDefaultCapacity(atoi(optarg));  	break;
    case 'd': ++stackDump;	 	 	 	 	break;
    case 'D': ++libDump;  	 	 	 	 	break;
    case 'e': sexp = optarg;					break;
    case 'f': suri = optarg; 					break;
    case 'g': graph = true;  					break;
    case 'h': usage();  	 	 	 	 	/*NOT REACHED*/
    case 'i': if (nbComps<MAX_COMPS) comps[nbComps++] = optarg;	break;
    case 'l': luri = optarg;  	 	 	 	 	break;
    case 'n': name = optarg; 	 				break;
    case 'r': recover = true;				 	break;
    case 's': threaded = false;  	 	 	 	break;
    case 'S': printStats = true;  	 	 	 	break;
    case 't': ++timed;		  	 	 	 	break;
    case 'u': use_syslog();	  	 	 	 	break;
    case 'v': ++verb;						break;
    case 'V': version(); 					/*NOT REACHED*/
    case 'w': wall = true;					break;
    default:
      cerr << "Usage: " << prog_name << " [OPTION]... [STACK]...\n"
	   << "Try '" << prog_name << " --help' for more information.\n";
      exit(USAGE_ERROR);
    }

  if (verb > 0) set_verbosity(verb);

  for (int i = optind; i < argc; ++i) {
    if (toRun >= MAX_STACKS) {
      pandora_warning("Too many stacks defined (maximum = " << MAX_STACKS 
		      << ") " << "skipping remainder");
      break;
    }
    stacks[toRun].init(argv[i]);
    ++toRun;
  }

  if (pandora_init() < 0) {
    pandora_error("cannot initialize Pandora");
  }

  if (wall) {
    wallclock->run();
  }

  if (recover) {
    storage->init(name);
  }
  
  if (luri != NULL) 
    grm->addResource(DynLoader::grm_id, luri, 1000);
  grm->addResource(DynLoader::grm_id, _libconfig_string_, 100);

  if (suri != NULL) 
    grm->addResource(Pandora::grm_id, suri, 1000);
  grm->addResource(Pandora::grm_id, _stacks_string_, 100);

  if (nbOpts > 0) {
    pandora->update();
    for (int i = 0; i < nbOpts; ++i) {
      if (!set_global_option(opts[i]))
	pandora_warning("failed setting global option: " << opts[i]);
    }
  }

  for (int i = 0; i < nbComps; ++i) display_comp_info(comps[i]);

  if (libDump) {
    dynloader->update();
    dynloader->dump();
  }

  if (stackDump) {
    if (toRun == 0) {
      pandora->dump();
    } else {
      for (int i = 0; i < toRun; ++i) {
	if (!pandora->expand(stacks[i], (stackDump > 1)))
	  pandora_status |= STACK_ERROR;
	StackEntry *se = pandora->getStackEntry(stacks[i]);
	if (se != NULL) { 
	  se->setup(); 
	  if (graph) {
	    se->print_graph();
	  } else {
	    se->print();
	  }
	} else {
	  pandora_warning(stacks[i] << ": no such stack");
	  pandora_status |= STACK_ERROR;
	}
      }
    }
    pandora_fini();
    return pandora_status;
  }

  if (sexp != NULL) {
    char stk[2048];
    snprintf(stk, 2048, "%%stack { @GuileInputComponent [$exec = '(and (display %s) (newline))'] @ControlComponent }", sexp);
    //pandora_debug(stk);
    pandora->init(stk);
    stacks[0].init("stack");
    toRun = 1;
  }

#ifdef _NOTHREADS
  threaded = false;
#endif

  if (sigint) (void) setsignal(SIGINT, &pandora_cleanup);

  clock_t start = 0, stop = 0, mid = 0;
  if (timed) start = clock();
  
  for (int i = 0; i < toRun; ++i) {
    if ((handles[i] = pandora->start(stacks[i], threaded, true)) 
	== NIL_STACK_HANDLE)
      pandora_status |= STACK_ERROR;
  }

#if 0
  for (int i = 0; i < toRun; ++i)
    pandora->poll(handles[i]);
#else
  pandora->poll(true);
#endif
  
  if (timed) stop = clock();

#ifndef NDEBUG
  long bpc = Packet::created, bcc = Component::created,
    bpa = Packet::alive, bca = Component::alive;
#endif

  pandora_fini();

#ifndef NDEBUG
  long apc = Packet::created, acc = Component::created,
    apa = Packet::alive, aca = Component::alive;


  if (apa != 0) {
    pandora_status |= PACKET_ERROR;
    pandora_warning("Packets have not been collected");
    printStats = true;
  }

  if (aca != 0) {
    pandora_status |= COMP_ERROR;
    pandora_warning("Components have not been collected");
    printStats = true;
  }

  if (printStats) {
    pandora_info("[packets]    "  << apa << "/" << apc
	 << " (" << bpa << "/" << bpc << ")");
    pandora_info("[components] " << aca << "/" << acc 
	 << " (" << bca << "/" << bcc << ")");
  }
#endif

  if (timed) {
#if USE_CLOCK
    cerr << (1000 * (stop - start)) / CLOCKS_PER_SEC << "\n";
#else
    cerr << (diffTimeStamp(stop.ru_utime, start.ru_utime) 
	     + diffTimeStamp(stop.ru_stime, start.ru_stime)) << "\n";
#endif
  }

  return pandora_status;
}

extern "C" RETSIGTYPE pandora_cleanup(int sig)
{
  static int called = 0;

  if (pandora != NULL) {
    ++called;
    pandora->stop_async();
  }

  return RETSIGVAL;
}

static void usage(void)
{
  cerr << "Usage: " << prog_name << " [OPTION]... [STACK]...\n"
       << "Runs all specified STACK(s)\n"
       << endl
       << "  -a, --assign=OPDEF     set option as specified in OPDEF\n"
       << "                         with OPDEF: stack.component.option=value\n"
       << "  -c, --no-sighandler    do not install SIGINT handler\n"
       << "  -d, --dump-stacks      dump stack definitions\n"
       << "  -D, --dump-libraries   dump dynamic library config\n"
       << "  -e, --exec=STRING      execute the command in Guile mode\n"
       << "  -f, --stack-uri=URI    read stack definitions from URI\n"
       << "  -h, --help             show this help message\n"
       << "  -i, --info=COMP        display information about component COMP\n"
       << "  -l, --library-uri=URI  read library config from URI\n"
       << "  -s, --no-thread        do not create additional threads\n"
       << "  -S, --print-stats      print statistics about objects used\n"
       << "  -t, --profile          print stack execution time\n"
       << "  -u, --syslog           log messages with syslog facility\n"
       << "  -v, --verbose          increase verbosity level\n"
       << "  -V, --version          print version information and exit\n"
       << "  -w, --wallclock        use wall clock time\n"
       << endl
       << "Report bugs to <simon.patarin@inria.fr>"                     
       << endl;
  exit(0);
}

static void version(void)
{
  cerr << "Pandora " << VERSION << "\n"
       << "Written by Simon Patarin <simon.patarin@inria.fr>\n\n"
       << "Copyright (C) 1999, 2000, 2001, 2002 Simon Patarin, INRIA\n"
       << "This is free software; see the source for copying conditions.  There is NO\n"
       << "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n";

  exit(0);
}

static bool set_global_option(const char *opt)
{
  if (opt == NULL) return false;
  if (pandora == NULL) return false;

  bool status = false;
  char *tmp = xstrdup(opt);
  char *stk, *comp, *op, *val;

  if ((stk =  strtok(tmp,  ":.")) == NULL) 	goto failed;
  if ((comp = strtok(NULL, ":.")) == NULL) 	goto failed;
  if ((op =   strtok(NULL, ":. =")) == NULL) 	goto failed;
  val =  strchr(opt, '=');
  if (val == NULL) goto failed;
  val = tmp + (val - opt) + 1;
  //pandora_debug("scanning: " << val);

  do {
    MultiValue mv;
    if (val != NULL) {
      StackFactory sf(val);
      if (sf.parseValue(&mv) <= 0) {
	pandora_warning("failed parsing value: " << val);
	goto failed;
      }
    }
    status = pandora->setOption(text(stk), text(comp), text(op), mv);
  } while(0);
    
 failed:
  __FREE(tmp);
  return status;
}

bool display_comp_info(const text &comp)
{
  text def_options[MAX_OPTS];
  MultiValue mv;
  text symbol, lib;
  text deps[16];
  symbol_id_t ids[16];
  int npkts = 0;  

  if (dynloader->lookup(symbol, comp))
    cout << "name:       " << symbol << "\n";
  else 
    return false;

  symbol_id_t id = DynLoader::make_id(symbol.data());
  cout << "id:         " << id << "\n";


  if (dynloader->getSymbol(symbol, lib)) 
    cout << "library:    " << lib << "\n";
  else
    return false;

  void *func = dynloader->load(id);
  int version = -1;
  text loc;
  int ndeps = dynloader->getLibrary(lib.data(), version, loc,
				    deps, sizeof(deps)/sizeof(text));
  cout << "version:    " << version << "\n";
  cout << "path:       " << loc << "\n";
  for (int j = 0; j < ndeps; ++j) {
    cout << "dependency: " << deps[j] << "\n";	
  }

  int nopts = Component::listOptions(symbol, def_options, 
				     sizeof(def_options)/sizeof(text));
  if (nopts < 0) {
    //pandora_warning("cannot get options for: " << comps[i]);
    goto out;
  }

  for (int j = 0; j < nopts; ++j) {
    mv.reset();
    cout << "option:     " << def_options[j] << " = ";
    if (!Component::getOptionDefault(symbol, def_options[j], &mv)) {
      cout << "<static>\n";
      continue;
    }
    if (mv.type == MultiValue::pointer) {
      cout << "<complex>\n";
      continue;
    }

    cout << mv << "\n";
  }

  npkts = Component::pkt_in.atOrNil(id, ids, 
				    sizeof(ids)/sizeof(symbol_id_t));
  for (int i = 0; i < npkts; ++i) {
    cout << "input:      " << Packet::get_name(ids[i]) << "\n";
  }
  
  npkts = Component::pkt_out.atOrNil(id, ids, 
				     sizeof(ids)/sizeof(symbol_id_t));
  for (int i = 0; i < npkts; ++i) {
    cout << "output:     " << Packet::get_name(ids[i]) << "\n";
  }

 out:
  dynloader->unload(func);
  return true;
}
