# Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA
#
# This file is part of Pandora, the Flexible Monitoring Platform.
#
# Pandora is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# Pandora is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Pandora; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

package Pandora;

#use strict;

BEGIN {
    use Exporter   ();
    #our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);
    # set the version for version checking
    $VERSION     = 1.00;
    @ISA         = qw(Exporter);
    @EXPORT      = qw(&findlib &split_file_name &lascan &finddeps, &core_name);
    %EXPORT_TAGS = ( );
    @EXPORT_OK   = qw($NM);
}
#our @EXPORT_OK;     # non-exported package globals go here

$Pandora::NM = '/usr/bin/nm -B';

sub findlib($)		{}
sub split_file_name($)	{}
sub lascan($)		{}  
sub finddeps(\%\%)	{}  
sub core_name($)	{}

END { }       # module clean-up code here (global destructor)     

use Cwd;

sub findlib ($) {
    my $file = shift(@_);
    my ($dir, $base, $ext) = split_file_name($file);
    my $lib;
    my @ver;
    my $version;
    my %h;

    if (! -f $file) {
	print STDERR "$0: $file: Not a plain file\n";
	return;
    }

    $lib = "$dir/";

    if ($ext eq "la") {
	%h = lascan($file);
	if ($h{"installed"} eq "no") {
	    $lib .= ".libs/";
	} 
	if ($h{"dlname"}) {
	    $lib .= $h{"dlname"};
	} elsif ($h{"old_library"}) {
	    $lib .= $h{"old_library"};
	} else {
	    $lib .= $base;
	}
	@ver = ($h{"current"}, $h{"age"}, $h{"revision"});
    } else {
	$lib .= $base;
	@ver = ($base =~ /\d+/g);
    }

    $version = 0;
    map { $version = ($version * 100) + $_; } @ver;

    if (! -e $lib) {
	print STDERR "$0: $file: Cannot find any library\n";
	return;
    }

    return ($lib, $version);
}

sub split_file_name ($) {
    my $file = shift(@_);
    my ($dir, $base, $ext);

    if ($file =~ /(.*\/)(.*)/) {
	$dir = Cwd::abs_path($1);
	$base = $2;
    } else {
	$dir = Cwd::abs_path('.');
	$base = $file;
    }

    if ($base =~ /\w+\.(\w+)((\.\d+)*)$/) {
	$ext = $1;
    } else {
	$ext = "??";
    }

    return ($dir, $base, $ext);

}

sub lascan ($) {
    my $file = shift(@_);
    my %hash;
    my $line;

    open(FILE, $file) or return %hash;
    while ($line = <FILE>) {
	if ($line =~ /(\S+)=[\'\"]?([^\'\n\"]*)/) {
	    $hash{$1} = $2;
	}
    }
    close(FILE);

    return %hash;
}

sub finddeps(\%\%) {
    my ($lib, $symbol, $dep, %deps, %dep2);
    my (%defined, %undefined);
    my ($d, $u) = (@_);
    
    %defined = %$d;
    %undefined = %$u;

    foreach $lib (keys %undefined) {
	foreach $symbol (keys %{ $undefined{$lib} }) {
	    if (($dep = $defined{$symbol})) {
		#print STDERR "$lib     \t-> $dep\t($symbol)\n";
		++$deps{$lib}{$dep};
	    }
	}
    }
    foreach $lib (keys %deps) {
	@{ $dep2{$lib} } = (keys %{ $deps{$lib} });
    }
    
    return %dep2;
}

sub core_name($) {
    my $file = shift(@_);
    my $core;
    if (!$file) {
	return "";
    }
    if ($file =~ /^(\/.*)*\/([^\.]+)\W+.*$/) {
	$core = $2;
    } else {
	$core = $file;
    }
    return $core;
}


1;
