/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "writercomponent.h"
#include <libpandora/fileutil.h>
#include <libpandora/common_options.h>
#include <libpandora/timersupervisor.h>
#include <libpandora/pktutil.h>

component_export(WriterComponent, Packet,);

WriterComponent::WriterComponent(void) 
  : f(stdout), prefix(NULL), fsize(INT_MAX), tinter(INT_MAX),
    offset(0), sizeCount(0), timeCount(0), lastPktTime(0) 
{
  registerStaticOption(write_file_option, "file", &f);
  registerOption("prefix", &prefix);
  registerOption("size", &fsize);
  registerOption("interval", &tinter);
}

bool WriterComponent::add(Packet *pkt) 
{
  int len = 0;
  if (f != NULL) 	len = packet_send(pkt, f, buf, MAX_PACKET_SIZE);
  if (prefix != NULL)	updateLimits(pkt->timeStamp.tv_sec, len);

  discard(pkt);
    
  return false;
}  

void WriterComponent::setup(void) 
{
  if (prefix != NULL) {
    time_t current_time = timerSupervisor->getTime();
    rotate_files(prefix, &f, current_time);
    lastPktTime = current_time;
  }
}

void WriterComponent::cleanup(void) 
{
  sizeCount = timeCount = 0;
  lastPktTime = 0;

  if (f != NULL) {
    if (prefix != NULL) {
      prefix = NULL;
      fclose(f);
      f = NULL;
    } else {
      fflush(f);
    }
  }
}

void WriterComponent::updateLimits(time_t curTime, int len)
{
  sizeCount += len;
  if (lastPktTime == 0) lastPktTime = curTime;
  if (curTime > lastPktTime) {
    timeCount += (curTime - lastPktTime);
    lastPktTime = curTime;
  }
  
  if ((f == NULL) | (sizeCount > fsize) | (timeCount > tinter)) {
    if (f != NULL) fclose(f);
    rotate_files(prefix, &f, lastPktTime);
    pandora_assert( f != NULL );
    if (fsize != INT_MAX) sizeCount = 0;
    if (tinter != INT_MAX) {
      offset = timeCount - tinter;
      timeCount = offset;
    }
  }
}
