/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <sys/socket.h>
}

#include <iostream>
#include "w2ninputcomponent.h"
#include <libpandora/server.h>
#include <libpandora/netutil.h>
#include <pandora_components/w2npacket.h>

component_export(W2NInputComponent,, Packet);

W2NInputComponent::W2NInputComponent(void)
  : listen(0), w2n_sl_name(NULL), w2n_sl_port(0), www_server_name(NULL),
    asnum(0), svr(NULL) 
{
  registerOption("listen", &listen);
  registerOption("addr", &w2n_sl_name);
  registerOption("port", &w2n_sl_port);
  registerOption("as", &asnum);
  registerOption("www", &www_server_name);
}

int W2NInputComponent::init(void)
{
  if (w2n_sl_name == NULL
      || w2n_sl_port == 0
      || www_server_name == NULL
      || listen <= 0) return ERROR_FILENO;
  

  pandora_info("will contact: " << w2n_sl_name << ":" << w2n_sl_port);
  int fd = openclient(w2n_sl_port, get_addr(w2n_sl_name), true, 2);
  if (fd < 0) return ERROR_FILENO;

  W2NPacket w2np(0);
  w2np.setOpCode(W2NPacket::MONITOR_REGISTER);
  w2np.asnum = asnum;
  w2np.server.init(www_server_name);
  w2np.port = listen;
  int len = w2np.write(buf, MAX_PACKET_SIZE, 0);

  if (net_writeblock(fd, buf, len) <= 0) 
    return ERROR_FILENO;

  close(fd);

  if (svr != NULL) delete svr;
  svr = new Server();
  if (!svr->init(&accept, (void *) this, listen, true, true)) 
    return ERROR_FILENO;
  return THREADED_FILENO /* svr->fileDesc() */;
}

bool W2NInputComponent::process(void)
{
  return svr->process();
}

void W2NInputComponent::finish(void)
{
  __DELETE(svr);
}


bool W2NInputComponent::read(int fd)
{
  W2NPacket *w2np = W2NPacket::recv_raw(fd, buf, MAX_PACKET_SIZE);
  if (w2np == NULL) return true;
  
  struct sockaddr_in peer;
  socklen_t len = sizeof(struct sockaddr);
  if (getpeername(fd, (struct sockaddr *)&peer, &len) < 0) {
    pandora_warning("cannot get peer address!!");
    cleanPacket(w2np);
    return true;
  }

  w2np->addr = peer.sin_addr.s_addr;

  push(w2np);

  return false;
}

bool W2NInputComponent::accept(int fd, void *udata) 
{
  W2NInputComponent *comp = (W2NInputComponent *)udata;
  pandora_assert(comp != NULL);

  return comp->read(fd);
}




