/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef VALUE_PACKET_H
#define VALUE_PACKET_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/time.h>
}

#include <libpandora/timeval.h>
#include <libpandora/serialize.h>
#include <libpandora/packet.h>
#include <libpandora/text.h>

template <class T>
class ValuePacket : public Packet {
public:
  text	tag;
  T	val;

public:
  inline ValuePacket(void) { }
  inline ValuePacket(const T &v, const timeval &ts);
  inline ValuePacket(const T &v);
  inline ValuePacket(const ValuePacket& x);
  inline ValuePacket& operator= (const ValuePacket& x);
  virtual ~ValuePacket(void) { }

  T *value(void) 				{ return &val; }
  
  void set(const T &v)				{ val = v; }

  void setTag(const char *s) 			{ tag.init(s); }
  void setTag(const char *s, int len) 		{ tag.init(s, len); }

  virtual void print(ostream *f) ;

  virtual size_t write(char *, size_t, int) ;
  virtual size_t read(const char *, int);
};

template <class T>
ValuePacket<T>::ValuePacket(const T &v, const timeval &ts)
{
  setTimeStamp(ts);
  set(v);
}

template <class T>
ValuePacket<T>::ValuePacket(const T &v)
{
  setTimeStamp();
  set(v);
}

template <class T>
ValuePacket<T>::ValuePacket(const ValuePacket& x) :
  Packet(x), tag(x.tag), val(x.val)
{
}

template <class T>
ValuePacket<T>& ValuePacket<T>::operator= (const ValuePacket& x) 
{
  Packet::operator=(x);
  tag = x.tag; val = x.val;
  return *this;
}

template <class T>
void ValuePacket<T>::print(ostream *f) 
{
  *f << timeStamp << '\t';
  if (!tag.isNull()) *f << "[" << tag << "] ";
  //else *f << "[value] ";
  *f << val << ' ';
  *f  << endl;
}

template <class T>
size_t ValuePacket<T>::write(char *str, size_t maxlen, int)
{
  size_t count = 0;

  serialVar(tag);
  serialVar(val);

  return count;
}

template <class T>
size_t ValuePacket<T>::read(const char *str, int)
{
  size_t count = 0;

  unserialVar(tag);
  unserialVar(val);

  return count;
}

#define VALPACKET_CLASS(name, type)					     \
  class name##ValuePacket : public ValuePacket<type>			     \
  {									     \
  public:								     \
    packet_init(name##ValuePacket, 1);					     \
    name##ValuePacket::name##ValuePacket(const type &v,	const timeval &ts)   \
      : ValuePacket<type> (v, ts) { } 					     \
    name##ValuePacket::name##ValuePacket(const type &v)   		     \
      : ValuePacket<type> (v) { } 					     \
    virtual ~name##ValuePacket(void) { }				     \
    name##ValuePacket(const name##ValuePacket& x) : ValuePacket<type>(x) { } \
    name##ValuePacket& operator= (const name##ValuePacket& x) {		     \
      ValuePacket<type>::operator=(x);					     \
      return *this;							     \
    }									     \
  }

VALPACKET_CLASS(Int, int);
VALPACKET_CLASS(Float, float);
VALPACKET_CLASS(Bool, bool);

class TextValuePacket : public ValuePacket<text>			     
{									     
public:								     
  packet_init(TextValuePacket, 1);					     
  TextValuePacket::TextValuePacket(const text &v, const timeval &ts)    
    : ValuePacket<text> (v, ts) { } 					     
  TextValuePacket::TextValuePacket(const char *str, const timeval &ts) {
    val.init(str);
    setTimeStamp(ts);
  }
  TextValuePacket::TextValuePacket(const char *str, size_t len,
				   const timeval &ts) {
    val.init(str, len);
    setTimeStamp(ts);
    
  }
  TextValuePacket::TextValuePacket(const text &v)    
    : ValuePacket<text> (v) { } 					     
  TextValuePacket::TextValuePacket(const char *str) {
    val.init(str);
    setTimeStamp();
  }
  TextValuePacket::TextValuePacket(const char *str, size_t len) {
    val.init(str, len);
    setTimeStamp();
    
  }
  virtual ~TextValuePacket(void) { }				     
  TextValuePacket(const TextValuePacket& x) : ValuePacket<text>(x) { } 
  TextValuePacket& operator= (const TextValuePacket& x) {		     
    ValuePacket<text>::operator=(x);					     
    return *this;							     
  }									     
};

#endif /* VALUE_PACKET_H */
