/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <arpa/inet.h>
#include <netinet/in.h>
	   }

#include <iostream>
#include <iomanip>
#include <libpandora/encoding.h>
#include <pandora_components/udppacket.h>
#include <pandora_components/ippacket.h>
#include <pandora_components/tcppacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/error.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>

packet_export(UDPPacket, TCPPacket|IPPacket);

UDPPacket::UDPPacket(IPPacket* ipp) 
  : length(0)
{
  if (ipp == NULL) return;

  u_int hlen = sizeof(udphdr);

  if(ipp->dlength() < hlen){
    pandora_warning("truncated UDP packet (header)");
    cleanPacket(ipp);
    return;
  }

  struct udphdr *udp_hdr = (struct udphdr *)(ipp->data());
  u_int ulen = ntohs(udp_hdr->uh_ulen);


  length = 	ipp->length - hlen;
  sport = 	udp_hdr->uh_sport;
  dport = 	udp_hdr->uh_dport;
  timeStamp = 	ipp->timeStamp;  
  (ipp->_data).move(hlen);

  packetSetUp(ipp);
}

UDPPacket::UDPPacket(TCPPacket *tcpp) 
  : length(0)
{
  if (tcpp == NULL) return;

  sport = tcpp->sport;
  dport = tcpp->dport;

  packetSetUpSkip(tcpp);
  timeStamp = tcpp->timeStamp;

  cleanPacket(tcpp);
}

UDPPacket::UDPPacket(const UDPPacket& x) 
  : Packet(x), sport(x.sport), dport(x.dport), length(0)
{
}

UDPPacket& UDPPacket::operator= (const UDPPacket& x) 
{
  Packet::operator=(x);
  sport = x.sport; dport = x.dport; length = 0; 
  return *this;
}


void UDPPacket::print(ostream *f)
{
  locatePacket(IPPacket, ipp, this);
  if (ipp == NULL) return;
  *f << timeStamp << '\t'
     << "[udp] "
     << ipp->src << ':' << ntohs(sport) << ' ' 
     << ipp->dst << ':' << ntohs(dport) << ' '
     << length << ' ';

  *f << endl;
}

size_t UDPPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(sport);
  serialVar(dport);

  beginLevel(0);
  serialVar(length);

  return count;
}

size_t UDPPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(sport);
  unserialVar(dport);

  beginLevel(0);
  unserialVar(length);

  return count;
}

extern_pandora(algo, bool, udpports, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(UDPPacket, udpp, pkt);
  if (udpp == NULL) return false;

  k->set(udpp->dport, udpp->sport);
 
  return true;
}
