/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/time.h>
	   }

#include "tracecomponent.h"
#include <libpandora/timeval.h>
#include <libpandora/timersupervisor.h>

component_export(TraceComponent, Packet, Packet);
component_export(CountComponent, Packet, Packet);
component_export(TimeComponent, Packet, Packet);
component_export(ClockComponent, Packet, Packet);
component_export(FlowRateComponent, Packet, Packet);

bool TraceComponent::add(Packet *pkt) 
{
  if (ts) cout << timerSupervisor->getTime() << ' ';
  if (tag != NULL) cout << tag;
#ifndef NDEBUG
  if (life) cout << rank << " [" << pkt->alive << " : " << alive << "]\t";
#endif
  pkt->print(&cout);
  push(pkt);
  return false;
}  

bool CountComponent::add(Packet *pkt) 
{
  ++count;
  push(pkt);
  return false;  
}

void CountComponent::cleanup(void) 
{
  pandora_info("component #" << rank << "@" << ind << ": " << count);  
  count = 0;
}

void TimeComponent::setup(void) 
{
  gettimeofday(&start, NULL);
}

bool TimeComponent::add(Packet *pkt)
{
  push(pkt);
  return false;
}

void TimeComponent::cleanup(void) 
{
  gettimeofday(&end, NULL);
  pandora_info("elapsed (s): " << ((float)diffTimeStamp(end, start))/1000);
  start.tv_sec = 0;
}

bool ClockComponent::add(Packet *pkt)
{
  push(pkt);
  return false;
}

void ClockComponent::setup(void) 
{
  start = clock();
}

void ClockComponent::cleanup(void) 
{
  end = clock();
  pandora_info("elapsed (s): " 
	       << ((float)(end - start))/CLOCKS_PER_SEC);
  start = 0;
}

FlowRateComponent::FlowRateComponent(void)
  : rate(0), interval(1), window(1), last(0), count(0)
{
  registerOption("rate", &rate);
  registerOption("interval", &interval);
  registerOption("window", &window);
}

bool FlowRateComponent::add(Packet *pkt) 
{
  time_t cur = pkt->timeStamp.tv_sec;
  if (last == 0) last = cur-1;
  while (cur > (last + interval)) {
    ring.put(count / interval);
    rate = ring.average();
    pandora_debug("component #" << rank << "@" << ind << ": " << rate);  
    count = 0;
    last += interval;
  }
  ++count;
  
  push(pkt);
  return false;  
}

void FlowRateComponent::setup(void) 
{
  if (window < 1) window = 1;
  if (interval < 1) interval = 1;
  rate = 0;
  ring.init(window);
}

void FlowRateComponent::cleanup(void) 
{
  rate = 0;
  last = 0;
}
