/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <ctype.h>
#include <libpandora/conf/string.h>
	   }
#include <libpandora/util.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>

#include "tcpmergecomponent.h"

component_export(TCPMergeComponent, TCPPacket, TCPPacket);

bool TCPMergeComponent::add(Packet *pkt) 
{
  TCPPacket *tcpp2 = static_cast<TCPPacket *>(pkt);

  bool eof = tcpp2->eof;

  if (flush) {
    //pandora_debug("[flushing]");
    if (all) push(tcpp2);
    else cleanPacket(tcpp2);
  } else {
    if (tcpp == NULL) {
      init(tcpp2);
    } else { 
      if (!merge(tcpp2)) cleanPacket(tcpp2);
    }
  }    

  return eof;
}


bool TCPMergeComponent::merge(TCPPacket *tcpp2)
{
  locatePacket(IPPacket, ipp2, tcpp2);
  pandora_assert(ipp2 != NULL);

  pandora_assert(ipp != NULL);
  pandora_assert(tcpp2 != tcpp);

  int len = (int) ipp2->dlength();
  if (len == 0) return false;

  int plen = (int) ipp->dlength();

  int bytes = pandora_min(len, size - plen);
  
  if (bytes != 0) {
    (ipp->_data).write(ipp2->data(), bytes);

    int mlen = match();
    if ((mlen >= 0) && (mlen > plen)) {
      bytes = (mlen - plen);
      (ipp->_data).shrink(bytes);
    }

    tcpp->eof |= 	tcpp2->eof;
    tcpp->gap |= 	tcpp2->gap;
    tcpp->ack = 	tcpp2->ack;
    tcpp->length += 	bytes;
  }

  offset = pandora_min(0, (ipp->dlength() - patlen));

  dispatch(true);

  if ((bytes < len) && (tcpp2->length > bytes)) {
    tcpp2->length -= bytes;
    (ipp2->_data).move(bytes);
    SEQ_ADD(tcpp2->seq, bytes);
    
    if (all) push(tcpp2);
    else cleanPacket(tcpp2);

    return true;
  } else {
    return false;
  }
}

void TCPMergeComponent::init(TCPPacket *tcpp2)
{
  locatePacket(IPPacket, ipp2, tcpp2);
  pandora_assert(ipp2 != NULL);

  tcpp = tcpp2;
  ipp = ipp2;

  if (!(tcpp->flags & TH_SYN)) tcpp->gap = true;
  if (size > 0) (ipp->_data).grow(size);  
  dispatch(true);
}

void TCPMergeComponent::setup(void)
{
  if (bstr != NULL) {
    escape_string(bstr);
    patlen = strlen(bstr);
    if (patlen > 0) bmh_setup(&hbmh, bstr, patlen);
  } 
}

void TCPMergeComponent::cleanup(void) 
{ 
  dispatch(false);
  if (patlen > 0) {
    bmh_done(&hbmh);
    bstr = NULL;
    patlen = 0;
  }
  flush = false;
}

void TCPMergeComponent::dispatch(bool fl)
{
  if (tcpp == NULL) return;
  
  if (fl 
      && (((int)ipp->dlength() != size)
	  && (!tcpp->eof)
	  && (!matched)))
    return;

#if 0  
  if (!tcpp->gap && ipp->dlength() > 0) {
    cout << ipp->data() 
	 << endl;
  }
#endif

  push(tcpp);
  tcpp = NULL;
  flush = true;
  matched = false;
}

int TCPMergeComponent::match(void)
{
  if (patlen == 0) return -1;
  if ((int)ipp->dlength() < patlen) return -1;

  //pandora_debug("looking for a match with pattern = '" << bstr << "'"
  //<< " (" << patlen << ")");
#if 1
  char *ptr = bmh_match(&hbmh, ipp->data() + offset, 
			ipp->dlength() - offset);
#else
  char *ptr = strstr(ipp->data() + offset, bstr);
#endif
  if (ptr == NULL) return -1;
  matched = true;
  return patlen + ptr - ipp->data();
}

void TCPMergeComponent::escape_string(char *str)
{
  const char *orig = str;
  char *esc = str;

  while(*orig != '\0') {
    if (*orig != '\\') {
      *esc = *orig;
      ++orig; ++esc;
    } else {
      ++orig;
      char c;
      switch(*orig) {
      case 't': c = '\t';  break;
      case 'r': c = '\r';  break;
      case 'n': c = '\n';  break;
      case 'b': c = '\b';  break;
      case '0': c = '\0';  break;
      default:  c = *orig;
      }
      *esc = c;
      ++esc; ++orig;
    }
  }
  *esc = '\0';
}

#if 0

#define SE	240
#define SB	250
#define WILL	251
#define WONT	252
#define DO	253
#define DONT	254
#define IAC	255

void TCPMergeComponent::merge(void) 
{
  cout << buf;

#if 0
  char buf2[TMERGE_BUFSIZ];
  int j=0;
  cout << "---------------------------------------"
       << "---------------------------------------" << endl
       << src << " -> " << dst
       << ": " << endl;
#if 1
  bool skip = false;
  for (int i = 0; i < count && buf[i] != '\0'; ++i) {
    switch((u_char) buf[i]) {
    case IAC:
      ++i;
      switch((u_char)buf[i]) {
      case WILL: case WONT: case DO: case DONT:
	++i;
	skip = false;
	break;
      case SB:
	skip = true;
	break;
      case SE:
	skip = false;
	break;
      default:
	break;
      }
      break;
    default:
      if (!skip) {
        buf2[j++] = buf[i];
	if (buf[i] == '\r' && buf[i+1] != '\n') buf2[j++] = '\n';
      }
      break;
    }
  }
#else
  char *off = strstr(buf, "DATA");
  char *endb = NULL;
  if (off != NULL) {
    off += 4;
    if (off - buf < TMERGE_BUFSIZ) {
      endb = strstr(off, "QUIT");
      if (endb != NULL) *endb = '\0';
      cout << off;
    }
  }
#endif
  buf2[j]='\0';
  cout << buf2;
  cout << endl << "---------------------------------------"
       << "---------------------------------------"
       << endl;
#endif
}

#endif
