/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <pandora_components/syscallpacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/timeval.h>
#include <libpandora/text.h>
#include <libpandora/serialize.h>

packet_export(SyscallPacket,);

SyscallPacket::SyscallPacket(syscallt_t t)
  : type(t), retval(0), pid(0), uid(0), args(0)
{
}

SyscallPacket::SyscallPacket(const syscall_hdr &hdr)
  : type(UNDEF), args(hdr.nargs)
{
  timeStamp =	hdr.stamp;
  type =	(syscallt_t) hdr.type;
  retval =	hdr.ret;
  pid = 	hdr.task.pid;
  uid =		hdr.task.uid;
  comm.init(hdr.task.comm, 16);
  cwd.init(hdr.task.cwd, 256);
}

SyscallPacket::SyscallPacket(const SyscallPacket& x) 
  : Packet(x), type(x.type), retval(x.retval),
    pid(x.pid), uid(x.uid), comm(x.comm), cwd(x.cwd), 
    args(x.args)
{
}

SyscallPacket& SyscallPacket::operator= (const SyscallPacket& x)
{
  Packet::operator=(x);
  type = x.type; retval = x.retval;
  pid = x.pid; uid = x.uid; comm = x.comm; cwd = x.cwd; 
  args = x.args;
 return *this;
}

void SyscallPacket::read_args(const char *str)
{
  // XXX lots of alignment problems....
  int offset = 0;
  int arg_len = 0;
  
  for (int i = 0; i < args.size(); ++i) {
    const syscall_arg *sarg = (const syscall_arg *)(str + offset);
    MultiValue *mv = &(args[i]);

    switch(sarg->type) {
    case SYSCALL_ARG_INT: mv->set(sarg->i); break;
    case SYSCALL_ARG_STR: mv->set(str + sizeof(syscall_arg) + offset, sarg->i);
      offset += sarg->i; break;
    default:
      pandora_warning("syscall arg: unknown type " << sarg->type);
      return;
    }
    offset += sizeof(syscall_arg);
  }
}

void SyscallPacket::print(ostream *f) 
{
  *f << timeStamp << '\t' 
     << "[syscall] #" << (int)type << " "
     << comm << "/" << pid << " (" << retval << ") ";
  for (int i = 0; i < args.size(); ++i) {
    args[i].print(f);
    *f << " ";
  }
    *f << "[" << cwd << "] ";
  *f  << endl;
}

size_t SyscallPacket::write(char *str, size_t maxlen, int)
{
  size_t count = 0;

  serialVar(type);
  serialVar(retval);
  serialVar(pid);
  serialVar(uid);
  serialVar(comm);
  serialVar(cwd);
  serialVar(args);

  return count;
}

size_t SyscallPacket::read(const char *str, int)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(retval);
  unserialVar(pid);
  unserialVar(uid);
  unserialVar(comm);
  unserialVar(cwd);
  unserialVar(args);

  return count;
}

extern_pandora(algo, bool, syscallarg, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(SyscallPacket, sp, pkt);
  if (sp == NULL) return false;

  int arg = -1;
  switch(sp->type) {
  case SyscallPacket::READ: case SyscallPacket::WRITE: 
  case SyscallPacket::CLOSE: 
    if ((sp->args).size() > 0) 
      arg = (sp->args[0]).value.d; 
    break;
  default:
    arg = sp->retval;
    break;
  }
  k->set(sp->pid, arg);
  
  return true;
}		 
