/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "synchrocomponent.h"
#include <libpandora/thread.h>

component_export(SynchroComponent, Packet, Packet);
component_export(SemaphoreComponent, Packet, Packet);

#define SYNCHRO_DEBUG 	0
#define SEMAPHORE_DEBUG 0

Mutex     SynchroComponent::mutex[NB_MUTEXES];
Semaphore SemaphoreComponent::sema[NB_SEMAPHORES];

bool SynchroComponent::add(Packet *p) 
{
#if SYNCHRO_DEBUG
  pandora_debug("["<< this << "] locking   mutex #" << mindex);
#endif
  mutex[mindex].lock(); 
  locked = true;

  push(p); 

  if (locked) {
#if SYNCHRO_DEBUG
    pandora_debug("["<< this << "] unlocking mutex #" << mindex);
#endif
    mutex[mindex].unlock(); 
    locked = false;
  }

  return false; 
}

void SynchroComponent::cleanup(void) 
{ 
  if (locked) { 
#if SYNCHRO_DEBUG
    pandora_debug("["<< this << "] unlocking mutex #" << mindex);
#endif
    mutex[mindex].unlock(); 
    locked = false; 
  }
}

void SynchroComponent::setup(void) 
{ 
  if (mindex < 0) {
    pandora_warning("invalid mutex index: " << mindex);
    pandora_warning("resetting to it 0");
    mindex = 0;
  }

  if (mindex >= NB_MUTEXES) {
    pandora_warning("invalid mutex index: " << mindex);
    pandora_warning("resetting it to " << (NB_MUTEXES - 1));
    mindex = NB_MUTEXES - 1;
  }
}


bool SemaphoreComponent::add(Packet *p) 
{
  push(p); 
  ++count;

  if (signal) {
#if SEMAPHORE_DEBUG
    pandora_debug("["<< this << "] signaling  semaphore #" << mindex);
#endif
    if (count == factor) {
      count = 0;
      sema[mindex].signal(); 
    }
  }

  if (wait) {
#if SEMAPHORE_DEBUG
    pandora_debug("["<< this << "] waiting on semaphore #" << mindex);
#endif
    if (count == factor) {
      count = 0;
      sema[mindex].wait(); 
    }
  }

  return false; 
}

void SemaphoreComponent::cleanup(void) 
{ 
#if 1
  if (signal) {
#if SEMAPHORE_DEBUG
    pandora_debug("["<< this << "] signaling  semaphore #" << mindex);
#endif
    sema[mindex].signal(); 
  }
#endif
}

void SemaphoreComponent::setup(void) 
{ 
  if (mindex < 0) {
    pandora_warning("invalid semaphore index: " << mindex);
    pandora_warning("resetting to it 0");
    mindex = 0;
  }

  if (mindex >= NB_SEMAPHORES) {
    pandora_warning("invalid semaphore index: " << mindex);
    pandora_warning("resetting it to " << (NB_SEMAPHORES - 1));
    mindex = NB_SEMAPHORES - 1;
  }
}

