/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */
 

#include <libpandora/global.h>

#include "snmpinputcomponent.h" 

#include <libpandora/conf/string.h>
#include <pandora_components/valuepacket.h> 

#ifdef USE_SBUF
#include <ucd-snmp/parse.h>
#endif

component_export(SNMPInputComponent,, IntValuePacket|TextValuePacket|BoolValuePacket);

SNMPInputComponent::SNMPInputComponent(void)
  : doWalk(false), host("localhost"), community("public"), variable(NULL)
{
  registerOption("walk", &doWalk);
  registerOption("host", &host); 
  registerOption("community", &community); 
  registerOption("var", &variable); 
}

int SNMPInputComponent::init(void) 
{
  init_snmp("pandora");

  if (variable == NULL) {
    pandora_warning("invalid null variable");
    return ERROR_FILENO;
  }

  snmp_sess_init(&session);
  session.peername = host;

  session.version = SNMP_VERSION_1;

  session.community = (u_char *)community;
  session.community_len = strlen(community);

  //ds_set_boolean(DS_LIBRARY_ID, DS_LIB_QUICK_PRINT, 1);

  return THREADED_FILENO;
}

bool SNMPInputComponent::process(void) 
{
  if (variable == NULL) return true;

  snmp_session *ss = snmp_open(&session);
  if (ss == NULL) {
    pandora_warning(snmp_api_errstring(snmp_errno));
    return true;
  }
    
  bool ret = (doWalk ? walk(ss) : get(ss));
  
  snmp_close(ss);
  
  return ret;
}

bool SNMPInputComponent::get(snmp_session *ss) 
{
  snmp_pdu *pdu = snmp_pdu_create(SNMP_MSG_GET);
  oid anOID[MAX_OID_LEN];
  size_t anOID_len = MAX_OID_LEN;
  
  if (!get_node(variable, anOID, &anOID_len)) {
    pandora_warning(variable << ": " << snmp_api_errstring(snmp_errno));
    return true;
  }
  snmp_add_null_var(pdu, anOID, anOID_len);
  
  snmp_pdu *response = NULL;
  int status = snmp_synch_response(ss, pdu, &response);
  bool ret = false;

  if (status == STAT_SUCCESS && response->errstat == SNMP_ERR_NOERROR) {
    for (variable_list *vars = response->variables; 
	 vars != NULL; 
	 vars = vars->next_variable) {
      ret = extract(vars) | ret;
    }
  } else {
    if (status == STAT_SUCCESS) {
      pandora_warning("error in packet: " <<
		      snmp_errstring(response->errstat));
    } else {
      snmp_sess_perror("snmpget", ss);
    }

    ret = true;
  }

  if (response != NULL)
    snmp_free_pdu(response);

  return ret;
}


bool SNMPInputComponent::walk(snmp_session *ss) 
{
  bool running = true;
  oid name[MAX_OID_LEN], root[MAX_OID_LEN];
  size_t name_len = MAX_OID_LEN, root_len = MAX_OID_LEN;
 
  if (!get_node(variable, root, &root_len)) {
    pandora_warning(variable << ": " << snmp_api_errstring(snmp_errno));
    return true;
  } 

  memmove(name, root, root_len * sizeof(oid));
  name_len = root_len; 

  while(running) {
    snmp_pdu *pdu = snmp_pdu_create(SNMP_MSG_GETNEXT);
    snmp_add_null_var(pdu, name, name_len);

    snmp_pdu *response = NULL;
    int status = snmp_synch_response(ss, pdu, &response);

    if (status == STAT_SUCCESS && response->errstat == SNMP_ERR_NOERROR) {
      for(variable_list *vars = response->variables; 
	  vars != NULL; 
	  vars = vars->next_variable) {

	//print_variable(vars->name, vars->name_length, vars);

	if ((vars->name_length < root_len) ||
	    (memcmp(root, vars->name, root_len * sizeof(oid)) != 0)) {
	  running = false;
	  continue;
	}
	
	if (!extract(vars)) {
	  memmove((char *)name, (char *)vars->name,
		 vars->name_length * sizeof(oid));
	  name_len = vars->name_length;
	} else {
	  running = false;
	}
      }
    } else {
    if (status == STAT_SUCCESS) {
      pandora_warning("error in packet: " <<
		      snmp_errstring(response->errstat));
    } else {
      snmp_sess_perror("snmpwalk", ss);
    }
      running = false;
    }

    if (response) snmp_free_pdu(response);
  } 

  return true;
}


bool SNMPInputComponent::extract(const variable_list *vars)
{
  if (vars == NULL) return true;

  Packet *vp = NULL;

  char _buf[BUFSIZ];

#ifdef USE_SBUF
  sbuf _sbuf;
  sbuf *buf = binit(&_sbuf, _buf, sizeof(_buf));
  if (buf == NULL) pandora_error("cannot init sbuf");
  char *base = buf->base;
#else
  char *buf = _buf;
  char *base = buf;
#endif

  switch(vars->type) {
  case ASN_OCTET_STR:
    if (vars->val.string == NULL) { vp = new TextValuePacket(""); break; }
    vp = new TextValuePacket((const char *)vars->val.string, vars->val_len);
    								break;
  case ASN_BOOLEAN:
    vp = new BoolValuePacket((bool)*(vars->val.integer));  	break;
  case ASN_INTEGER: case ASN_GAUGE: case ASN_UINTEGER: case ASN_COUNTER: 
  case ASN_TIMETICKS:
    vp = new IntValuePacket((int)*(vars->val.integer));	 	break;
  case ASN_BIT_STR:
    if (vars->val.bitstring == NULL) vp = new TextValuePacket(""); 	
    else vp = new TextValuePacket((const char *)vars->val.bitstring,
				  vars->val_len);
    break;
  case ASN_OBJECT_ID:
    sprint_objid(buf, (oid *)(vars->val.objid), vars->val_len / sizeof(oid));
    vp = new TextValuePacket(base);
    break;
  case ASN_IPADDRESS: {
    u_char *ip = vars->val.string;
    u_int32_t addr = ((ip[3] << 24) + (ip[2] << 16) + (ip[1] << 8) + ip[0]);
    vp = new IntValuePacket((int)addr);
  }								break;
  case ASN_NULL:
    break;
  case ASN_SEQUENCE:
  case ASN_SET:       case ASN_OPAQUE:     
  case ASN_COUNTER64:
#if 1
    pandora_warning("unsupported variable type: 0x" 
		    << hex << (int)vars->type << dec);
#endif
    sprint_value(buf, vars->name, vars->name_length, (variable_list *)vars);
    vp = new TextValuePacket(base);
  break;
  case SNMP_ENDOFMIBVIEW: case SNMP_NOSUCHOBJECT: case SNMP_NOSUCHINSTANCE:
    return true;
  default:
    pandora_warning("unkown variable type: 0x" 
		    << hex << (int)vars->type << dec);
    break;
  }

  if (vp == NULL) return true;

  sprint_objid(buf, vars->name, vars->name_length);
  ((IntValuePacket *)vp)->setTag(base);

  push(vp);
  return false;
}
