/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include <pandora_components/rpcpacket.h>
#include <pandora_components/udppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/serialize.h>
#include <libpandora/error.h>
#include <libpandora/timeval.h>

packet_export(RPCPacket, UDPPacket);

RPCPacket::RPCPacket(UDPPacket* udpp) 
  : type(undefined), length(0)
{
  locatePacket(IPPacket, ipp, udpp);
  if (ipp == NULL) return;

  u_int hlen = sizeof(struct rpc_msg);

  if (ipp->dlength() < hlen) {
    pandora_warning("truncated RPC packet");
    cleanPacket(udpp);
    return;
  }

  struct rpc_msg *rpc = (struct rpc_msg *)(ipp->data());

  length = udpp->length - hlen;

  type = ((rpc->rm_direction == CALL) ? request : response);
  xid = rpc->rm_xid;

  if (type == request) {
    prog = rpc->rm_call.cb_prog;
    vers = rpc->rm_call.cb_vers;
    proc = rpc->rm_call.cb_proc;
  } else { // type == REPLY
    accepted = (rpc->rm_reply.rp_stat == MSG_ACCEPTED);
    if (accepted) {
      status = rpc->rm_reply.rp_acpt.ar_stat;
    } else {
      status = rpc->rm_reply.rp_rjct.rj_stat;
    }
  }
  (ipp->_data).move(hlen);
  timeStamp = udpp->timeStamp; 

  packetSetUp(udpp);
}


RPCPacket::RPCPacket(const RPCPacket& x) 
  : Packet(x), type(x.type), xid(x.xid), status(x.status), 
    accepted(x.accepted), prog(x.prog), vers(x.vers), proc(x.proc),
    length(x.length)
{
}

RPCPacket& RPCPacket::operator= (const RPCPacket& x) 
{
  Packet::operator=(x);
  xid = x.xid; status = x.status; accepted = x.accepted;
  prog = x.prog; vers = x.vers; proc = x.proc;
  length = x.length;
  return *this;
}

void RPCPacket::print(ostream *f)
{
  locatePacket(UDPPacket,	udpp,	this);
  locatePacket(IPPacket,	ipp,	udpp);
  *f << timeStamp << '\t'
     << "[rpc] "
     << ipp->src << ':' << ntohs(udpp->sport) << ' ' 
     << ipp->dst << ':' << ntohs(udpp->dport) << ' '
     << length << ' '
     << ntohl(prog) << '.' << ntohl(vers)
     << "::" << ntohl(proc) << ' ';

  *f << (int) type << ' ' << xid << ' ';
  *f << endl;
}

size_t RPCPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(type);
  serialVar(xid);
  serialVar(status);
  serialVar(accepted);
  serialVar(prog);
  serialVar(vers);
  serialVar(proc);
  serialVar(length);
  
  return count;
}

size_t RPCPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(xid);
  unserialVar(status);
  unserialVar(accepted);
  unserialVar(prog);
  unserialVar(vers);
  unserialVar(proc);
  unserialVar(length);

  return count;
}

extern_pandora(algo, bool, rpcid, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(RPCPacket, rpcp, pkt);
  if (rpcp == NULL) return false;

  k->set(rpcp->xid);
  
  return true;
}		 
