/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "netwritercomponent.h"
#include <libpandora/netutil.h>
#include <libpandora/pktutil.h>
#include <libpandora/conf/poll.h>

component_export(NetWriterComponent, Packet,);

bool NetWriterComponent::add(Packet *pkt) 
{
  int retries = 0;
  bool ok = true;

  do {
    if ((fd < 0) && (!connect())) break;
    
    int n = packet_send(pkt, fd, buf, MAX_PACKET_SIZE);
    
    ok = ((fd > 0) & (n > 0));

    if (!ok) {
      cnx_close();
      retry_timeout *= (retries + 1);

      if (retries > 0) {
	pandora_notice("connection lost, will retry within " 
		     << retry_timeout << " seconds");
	sleep(retry_timeout);
      }
    }

  } while ((!ok) && (++retries < max_retries));

  if (retries == max_retries) {
    pandora_notice("maximum retries reached, giving up");
    return true;
  }

  int tout = wait;
  while (isReadable(tout)) {
    Packet *rpkt = NULL;
    packet_recv(rpkt, fd, buf, MAX_PACKET_SIZE);

    if (rpkt != NULL) {
      reply(rpkt);
#if 0
      CERR(LOG_DEBUG) << "received reply: "; rpkt->print(&cerr);
      discard(rpkt);
#endif
    } else {
      pandora_notice("lost connection from: " << host);
      cnx_close();
      //break;
    }
    tout = 0;
  }

  discard(pkt);
  
  return false;
}  

void NetWriterComponent::setup(void)
{
  cnx_close();
}

void NetWriterComponent::cleanup(void)
{
  cnx_close();
  host = NULL;
}

bool NetWriterComponent::connect(void) 
{
  if (!usock && (host == NULL)) {
    pandora_warning("invalid host name specification");
    return false;    
  }
  
  if (port == 0) {
    pandora_warning("invalid port number " << port);
    return false;
  }

  bool stream = (port > 0);

  fd = openclient((stream ? port : -port), 
		  (usock ? 0 : get_addr(host)), 
		  !usock, stream, 3);

  return (fd > 0);
}

bool NetWriterComponent::isReadable(int tout)
{
  struct pollfd ufd;
  if (fd < 0) return false;
  ufd.fd = fd;
  ufd.events = POLLIN;
  ufd.revents = 0;

  int p = poll(&ufd, 1, tout);
  if (p < 0) {
    pandora_pwarning("poll");
    cnx_close();
    return false;
  }

  if ((p == 1) && (ufd.revents != POLLIN)) {
    pandora_info("connection closed from: " 
		 << (usock ? "unix" : host)
		 << ":" << port); 
    cnx_close();
    return false;
  }

  //pandora_debug("readable: " << (ufd.revents == POLLIN));
  return (ufd.revents == POLLIN);
}

