/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include "membercomponent.h"
#include <libpandora/util.h>
#include <libpandora/netutil.h>
#include <libpandora/conf/poll.h>

component_export(MemberComponent,,);

MemberComponent::MemberComponent(void) 
  : round(0), leader(0), group(0), wait(3), timeout(1), fd(-1) 
{
  registerOption("leader",  &leader);
  registerOption("group",   &group);
  registerOption("wait",    &wait);
  registerOption("timeout", &timeout);
}

int MemberComponent::init(void)
{

  if (group <= 0) {
    pandora_warning("invalid port number " << listen);
    return ERROR_FILENO;
  }

  fd = get_broadcast(group);
  if (fd < 0) return ERROR_FILENO;

  return fd;
}

bool MemberComponent::process(void)
{
#if 0
  pandora_debug("round #" << round << ": " << members.size() << " members" 
		<< " (" << (leader ? intoa(leader) : "no leader") << ")");
#endif
  if (!do_send()) return true;
  if (!do_poll()) return true;
  ++round;
  if (!do_update()) return true;
  return false;
}

void MemberComponent::finish(void)
{
  close(fd);
  fd = -1;
}


int MemberComponent::get_broadcast(int port)
{
  int bcast_fd = socket(AF_INET, SOCK_DGRAM, 0); 
  if (bcast_fd < 0) {
    pandora_pwarning("socket");
    return -1;
  }

#ifdef SO_BROADCAST
  int one = 1;
  if (setsockopt(bcast_fd, SOL_SOCKET, SO_BROADCAST, 
		 (char *)&one, sizeof(int)) < 0) {
    pandora_pwarning("setsockopt");
    close(bcast_fd);
    return -1;
  }
#endif

  memset(&bcast_addr, 0, sizeof(struct sockaddr_in));
  bcast_addr.sin_family = AF_INET;
  bcast_addr.sin_port = htons(port);
  bcast_addr.sin_addr.s_addr = (in_addr_t) -1;  
  
  if (bind(bcast_fd, (struct sockaddr *)&bcast_addr, 
	   sizeof(struct sockaddr_in)) < 0) {
    pandora_pwarning("bind");
    close(bcast_fd);
    return -1;
  }

  return bcast_fd;
}

bool MemberComponent::do_read(void)
{
  sockaddr_in from_addr;
  socklen_t len = sizeof(sockaddr_in);
  u_int hm = 0;
  if (recvfrom(fd, &hm, 4, 0, (sockaddr *)&from_addr, &len) != 4) {
    pandora_pwarning("read");
    return false;
  }
  if (ntohl(hm) != magic) return true;
  if (ntohs(from_addr.sin_port) != group) return true;
  
  in_addr_t addr = from_addr.sin_addr.s_addr;

  if (!members.includesKey(addr)) 
    pandora_debug(intoa(addr) << " joined the group");
  members.atPut(addr, round);

  return true;
}

bool MemberComponent::do_poll(void)
{
  int tout = timeout * 1000;

  for (;;) {
    timeval start, end;
    pollfd ufd;
    memset((char *)&ufd, 0, sizeof(ufd));
    ufd.fd = fd;
    ufd.events = POLLIN;
    
    gettimeofday(&start, NULL);
    switch(poll(&ufd, 1, tout)) {
    case 0: 	
      return true;
      
    case 1: 	
      if (!do_read()) return false; 
      break;

    default:  	
      if (errno != EINTR) pandora_pwarning("poll"); 
      return false;
    }
    gettimeofday(&end, NULL);
    tout -= diffTimeStamp(end, start);
    if (tout <= 0) break;
  }

  return true;
}

bool MemberComponent::do_send(void)
{
  static u_int hm = htonl(magic);
  if (sendto(fd, &hm, 4, 0,
	     (sockaddr *) &bcast_addr, sizeof(sockaddr_in)) != 4) {
    pandora_pwarning("sendto");
    return false;
  }
  return true;
}

bool MemberComponent::do_update(void)
{
  in_addr_t addr, l = 0;
  int r;
  keysValuesDo(members, addr, r) {
    if (r < (round - wait)) {
      members.removeKey(addr);
      pandora_debug(intoa(addr) << " left the group");
    } else {
      if (addr > l) l = addr;
    }
  }
  if (l != leader) {
    leader = l;
    pandora_debug(intoa(leader) << " new leader");
  }
  return true;
}
