/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <libpandora/global.h>

#include <libpandora/conf/snprintf.h>

#include "mailercomponent.h" 
#include <pandora_components/valuepacket.h> 
#include <libpandora/netutil.h>

component_export(MailerComponent, TextValuePacket+,);


MailerComponent::MailerComponent(void)
  : smtp("localhost"), port(25), to(NULL), from(NULL), subject(NULL)
{ 
  registerOption("smtp", &smtp); 
  registerOption("port", &port); 
  registerOption("to", &to); 
  registerOption("from", &from);  
  registerOption("subject", &subject);  

  addr = inet_addr("127.0.0.1");
  if (gethostname(localhost, sizeof(localhost)) < 0) {
    *localhost = '\0';
    pandora_warning("cannot lookup localhost name");
  }
  snprintf(from_def, sizeof(from_def), "Pandora <pandora@%s>", localhost);
}

MailerComponent::~MailerComponent(void)
{
}

bool MailerComponent::add(Packet *pkt) 
{
  int fd;
  FILE *in = NULL, *out = (FILE *)0x23;

  locatePacket0(TextValuePacket, valuep, pkt);
  if (valuep == NULL) goto finished;

  if (to == NULL) {
    pandora_warning("recipient not set");
    goto finished;
  }

  if (*localhost == '\0') {
    pandora_warning("cannot send mail if local hostname not set");
    goto finished;
  }

  if (from == NULL) from = from_def;

  if (!format_message(from, to, subject, (valuep->val).data(), 
		      outbuf, sizeof(outbuf))) {
    pandora_warning("message too long");
    goto finished;
  }

  if ((fd = openclient(port, addr, true, 30)) < 0) {
    pandora_pwarning("connection to " << smtp << " failed");
    goto finished;
  }

  fdopen_socket(fd, &in, &out);
  if (in == NULL) {
    pandora_pwarning("fdopen_socket");
    goto finished;
  }
  setvbuf(in, NULL, _IONBF, 0);
  
  if (read_response(in) != 220) goto close_finished;

  fprintf(out, "HELO %s\r\n", localhost);
  if (read_response(in) != 250) goto quit_finished;

  fprintf(out, "MAIL FROM:%s\r\n", canon_addr(from));
  if (read_response(in) != 250) goto quit_finished;

  fprintf(out, "RCPT TO:%s\r\n", canon_addr(to));
  if (read_response(in) != 250) goto quit_finished;

  fprintf(out, "DATA\r\n");
  if (read_response(in) != 354) goto quit_finished;

  fprintf(out, "%s\r\n.\r\n", outbuf);
  if (read_response(in) != 250) goto quit_finished;

 quit_finished:
  fprintf(out, "QUIT\r\n");
  if (read_response(in) != 221) ;

 close_finished:
  //close(fd);
  fclose(in);
  fclose(out);

 finished:
  discard(pkt);
  return false;
}
 
void MailerComponent::setup(void) 
{
  addr = get_addr(smtp);
}

void MailerComponent::cleanup(void) 
{
}

const char *MailerComponent::canon_addr(const char *a) 
{
  int pos = strlen(a)-1;
  if (pos <= 0) return a;
  while(isspace(a[pos])) --pos;
  if (pos <= 0) return a;
  
  const char *ptr;
  for (ptr = a + pos; ptr >= a; --ptr)
    if (isspace(*ptr)) break;

  return ptr+1;
}

int MailerComponent::read_response(FILE *in) 
{
  if (fgets(inbuf, sizeof(inbuf), in) == NULL) {
    pandora_pwarning("fgets");
    return -1;
  }

#if 0
  char *ptr = strchr(inbuf, '\n');
  if (ptr != NULL) { while (isspace(*--ptr)) ;  *(ptr+1) = '\0'; }
  pandora_debug("read: " << inbuf);
#endif

  return atoi(inbuf);
}

bool MailerComponent::format_message(const char *from, const char *to, 
				     const char *subject, const char *msg,
				     char *buf, int size)
{
  int n = 0;
  n = snprintf(buf, size, 
	       "From: %s\r\n"
	       "To: %s\r\n"
	       "Subject: %s\r\n"
	       "MIME-Version: 1.0\r\n"
	       "Content-Type: text/plain\r\n\r\n",
	       from, to, subject);
  
  if (n < 0) return false;
  size -= n;
  buf += n;

  const char *start = msg, *end = NULL;
  while ((end = strchr(start, '\n')) != NULL) {
    int offset = 1;
    if (*(end-1) == '\r') { --end; ++offset; } 
    n = (end - start);
    if ((n+3) > size) return false;
    if ((n == 1) && (*start == '.')) {
      n = 2;
      memcpy(buf, "..", 2);
    } else { 
      memcpy(buf, start, (end - start));
    }
    memcpy(buf + n, "\r\n", 2);
    buf += (n + 2);
    size -= (n + 2);
    start = end + offset;
  }
  
  strncpy(buf, start, size);

  return true;
}


#if 0
127.0.0.1:25 -> 127.0.0.1:32789
220 localhost ESMTP Debian Sendmail 8.12.0.Beta7/8.12.0.Beta7/Debian 8.
12.0.Beta7-1; Mon, 21 Jan 2002 19:46:50 +0100..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
EHLO localhost..
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
250-pandora.dnsalias.org Hello localhost [127.0.0.1], pleased to meet y
ou..250-ENHANCEDSTATUSCODES..250-PIPELINING..250-EXPN..250-VERB..250-8BI
TMIME..250-SIZE..250-DSN..250-ETRN..250-DELIVERBY..250 HELP..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
RSET..
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
250 2.0.0 Reset state..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
MAIL FROM:<simon@pandora.dnsalias.org>..
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
250 2.1.0 <simon@pandora.dnsalias.org>... Sender ok..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
RCPT TO:<simon@pandora.dnsalias.org>..
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
250 2.1.5 <simon@pandora.dnsalias.org>... Recipient ok..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
DATA..
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
354 Enter mail, end with "." on a line by itself..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
Date: Mon, 21 Jan 2002 19:46:50 +0100 (CET)..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
From: Simon Patarin <simon@pandora.dnsalias.org>..To: Simon Patarin <si
mon@pandora.dnsalias.org>..Subject: test..Message-ID: <Pine.LNX.4.21.020
1211946340.655-100000@pandora.dnsalias.org>..MIME-Version: 1.0..Content-
Type: TEXT/PLAIN; charset=US-ASCII....test......test.............
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
250 2.0.0 g0LIkoR5000873 Message accepted for delivery..
----------------------------------------------------------------------
127.0.0.1:32789 -> 127.0.0.1:25
QUIT..
----------------------------------------------------------------------
127.0.0.1:25 -> 127.0.0.1:32789
221 2.0.0 pandora.dnsalias.org closing connection..
----------------------------------------------------------------------
#endif
