/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#if (defined __linux__) || (defined __FreeBSD__) || (defined __svr4__)

#include <iostream>

extern "C" {
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/syscall.h>
#include <dirent.h> 
#include <dlfcn.h>
#include <unistd.h>
#include <sched.h>
#include <assert.h>
#include "systrace.h"

struct syscall_task task = { 0, 0, "", "" };
#define curpid 	task.pid
#define curuid 	task.uid
#define curcomm task.comm
#define curwd 	task.cwd

/* 
   Note that LIBCPATH isn't actually used on Linux or Solaris, as RTLD_NEXT
   is defined and we use that to get the `next_*' functions

   Linux:
*/
#ifdef __alpha__
#define LIBCPATH "/lib/libc.so.6.1"
#else
#define LIBCPATH "/lib/libc.so.6"
#endif

/* OSF1 :*/
/*#define LIBCPATH "/usr/shlib/libc.so"*/

/*
#ifdef __USE_FILE_OFFSET64
#error " Can't compile like this"
#endif
*/

static void *get_libc(void)
{ 
#ifndef RTLD_NEXT
 static void *lib = 0;
 if (!lib) { 
   lib= dlopen(LIBCPATH, RTLD_LAZY);
 }
 return lib;
#else
  return RTLD_NEXT;
#endif
}

/** end of snippet from libfakeroot.c **/


#define _wrap_decl(ret, name, proto, body...)		\
  typedef ret (* name##_t) proto;			\
  ret name##_xxx proto {				\
    static name##_t name##_real = NULL;			\
    body						\
  } 

#define wrap_decl(ret, name, proto, body...)				 \
  typedef ret (* name##_t) proto;					 \
  ret name proto {							 \
    static name##_t name##_real = (name##_t) dlsym((void *) -1l, #name); \
    body								 \
  } 

#define syscall_log(sc, ret, fmt, args...) \
  sc_log(#sc, SYS_##sc, ret, fmt, args)

static void sc_log(const char *name, int nsys, int ret, const char *fmt, ...);

wrap_decl(pid_t, fork, (void),
{
  pid_t result = (*fork_real)();
  if (!result) curpid = getpid();
  return result;
} );
  
/* Quoting from vfork(2) man page on GNU/Linux:

--   
It is rather unfortunate that Linux revived  this  spectre
from  the past.  The BSD manpage states: "This system call
will be eliminated when proper system  sharing  mechanisms
are  implemented.  Users  should  not depend on the memory
sharing semantics of vfork as it will, in  that  case,  be
made synonymous to fork."

Formally  speaking,  the  standard description given above
does not allow one to use vfork() since a  following  exec
might fail, and then what happens is undefined.
--

As such libintercept is not able to follow vfork calls. :(
So, let's make it be a fork!
*/

wrap_decl(pid_t, vfork, (void),
{
  return fork();
} );
  
wrap_decl(int, chdir, (const char *pathname),
{
  int result = (*chdir_real)(pathname);
  getcwd(curwd, sizeof(curwd));
  return result;    
} );

wrap_decl(int, fchdir, (int fd),
{
  int result = (*fchdir_real)(fd);
  getcwd(curwd, sizeof(curwd));
  return result;    
} );

wrap_decl(int, open, (const char *file, int oflag, mode_t mode),
{
  int result = (*open_real)(file, oflag, mode);
  syscall_log(open, result, "sdd", file, oflag, mode);
  return result;
} );

wrap_decl(int, open64, (const char *file, int oflag, mode_t mode),
{
  int result = (*open64_real)(file, oflag, mode);
  syscall_log(open, result, "sdd", file, oflag, mode);
  return result;
} );

wrap_decl(int, close, (int fd),
{
  int result = (*close_real)(fd);
  syscall_log(close, result, "d", fd);
  return result;  
} );

wrap_decl(int, __open, (const char *file, int oflag, mode_t mode),
{
  int result = (*__open_real)(file, oflag, mode);
  syscall_log(open, result, "sdd", file, oflag, mode);
  return result;
} );

wrap_decl(int, __close, (int fd),
{
  int result = (*__close_real)(fd);
  syscall_log(close, result, "d", fd);
  return result;  
} );

wrap_decl(int, unlink, (const char *pathname),
{
  int result = (*unlink_real)(pathname);
  syscall_log(unlink, result, "s", pathname);
  return result;
} );

wrap_decl(ssize_t, write, (int fd, const void *buf, size_t nbytes),
{
  ssize_t result = (*write_real)(fd, buf, nbytes);
  syscall_log(write, result, "dpd", fd, buf, nbytes);
  return result;    
} );

wrap_decl(ssize_t, read, (int fd, void *buf, size_t nbytes),
{
  ssize_t result = (*read_real)(fd, buf, nbytes);
  syscall_log(read, result, "dpd", fd, buf, nbytes);
  return result;    
} );

wrap_decl(ssize_t, __write, (int fd, const void *buf, size_t nbytes),
{
  ssize_t result = (*__write_real)(fd, buf, nbytes);
  syscall_log(write, result, "dpd", fd, buf, nbytes);
  return result;    
} );

wrap_decl(ssize_t, __read, (int fd, void *buf, size_t nbytes),
{
  ssize_t result = (*__read_real)(fd, buf, nbytes);
  syscall_log(read, result, "dpd", fd, buf, nbytes);
  return result;    
} );

wrap_decl(int, mkdir, (const char *pathname, mode_t mode),
{
  int result = (*mkdir_real)(pathname, mode);
  syscall_log(mkdir, result, "sd", pathname, mode);
  return result;    
} );

wrap_decl(int, rmdir, (const char *pathname),
{
  int result = (*rmdir_real)(pathname);
  syscall_log(rmdir, result, "s", pathname);
  return result;    
} );

wrap_decl(int, dup, (int oldfd),
{
  int result = (*dup_real)(oldfd);
  syscall_log(dup, result, "d", oldfd);
  return result;  
} );

wrap_decl(int, dup2, (int oldfd, int newfd),
{
  int result = (*dup2_real)(oldfd, newfd);
  syscall_log(dup2, result, "dd", oldfd, newfd);
  return result;  
} );


/** communication / logging **/

static int create_socket(void)
{
  static const char unix_path[] = "/tmp/.pandora/2300";

  int fd = socket(AF_LOCAL, SOCK_DGRAM, 0);
  if (fd < 0) 
    goto fail;

  struct sockaddr_un unix_addr;
  memset(&unix_addr, 0, sizeof(sockaddr_un));
  unix_addr.sun_family = AF_LOCAL;
  strncpy(unix_addr.sun_path, unix_path, sizeof(unix_addr.sun_path));

  if (connect(fd, (sockaddr *)&unix_addr, SUN_LEN(&unix_addr)) < 0)
    goto fail;

  return fd;

 fail:
  _exit(23);
}

static void sc_log(const char *name, int nsys, int ret, const char *fmt, ...)
{
#if 1
  static int fd = create_socket();
  static char buf[BUFSIZ];
  static const size_t header_len = sizeof(struct syscall_hdr);
  static const size_t arg_len = sizeof(struct syscall_arg);
  
  int n = 0, p = header_len;
  int nargs = 0;

  if (!curpid) curpid = getpid();
  if (!*curwd) getcwd(curwd, sizeof(curwd));

  struct syscall_hdr header;
  header.type = nsys;
  header.ret = ret;

  n = sizeof(buf);

  va_list ap;
  va_start(ap, fmt);
  while (*fmt) {
    struct syscall_arg arg;
    switch(*fmt++) {
    case 's':           /* string */
      arg.type = SYSCALL_ARG_STR;
      arg.p = va_arg(ap, char *);
      arg.i = strlen(arg.p);
      break;
    case 'd':           /* int */
      arg.type = SYSCALL_ARG_INT;
      arg.i = va_arg(ap, int);
      break;
    case 'p':           /* void* */
      arg.type = SYSCALL_ARG_INT;
      arg.i = va_arg(ap, int);
      break;
    }

    if (n > (int)(p + arg_len)) {
      ++nargs;
      memcpy(buf + p, &arg, arg_len);
      p += arg_len;
      n -= arg_len;

      if (arg.type == SYSCALL_ARG_STR) {
	if (n > arg.i) {
	  memcpy(buf + p, arg.p, arg.i);
	  p += arg.i;
	  n -= arg.i;
	} else {
	  memset(buf + p - arg_len + sizeof(int), 0, sizeof(int));
	}
      }
    } 
  }
  va_end(ap);

  header.nargs = nargs;
  header.len = p - header_len;
  memcpy(buf, &header, header_len);
  memcpy(&(((syscall_hdr *)buf)->task), &task, sizeof(syscall_task));
  send(fd, buf, p, 0);
#else
  cerr << "** " << name << "  \t#" << nsys << " [" << curpid << "] ";
  va_list ap;
  va_start(ap, fmt);
  while (*fmt)
    switch(*fmt++) {
    case 's':           /* string */
      cerr << va_arg(ap, char *) << ' ';
      break;
    case 'd':           /* int */
      cerr << va_arg(ap, int) << ' ';
      break;
    case 'p':           /* char */
      cerr << va_arg(ap, void *) << ' ';
      break;
    }
  va_end(ap);
  cerr << "-> " << ret << '\n';
#endif
}

}
#endif
