/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <netinet/in.h>
#include <libpandora/conf/string.h>
	   }

#include <iostream>
#include <iomanip>
#include <libpandora/encoding.h>
#include <pandora_components/icppacket.h>
#include <pandora_components/udppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>
#include <libpandora/error.h>

packet_export(ICPPacket, UDPPacket);

ICPPacket::ICPPacket(UDPPacket* udpp) 
  : type(ICP_OP_INVALID)
{
  locatePacket(IPPacket, ipp, udpp);
  if (ipp == NULL) return;

  u_int hlen = sizeof(icphdr);

  if (ipp->dlength() < hlen) {
    pandora_warning("truncated ICP packet");
    cleanPacket(udpp);
    return;
  }

  struct icphdr *icp = (struct icphdr *)(ipp->data());

  u_int ilen = ntohs(icp->ih_len);
  if (ilen != udpp->length) {
    pandora_warning("ICP and UDP length mismatch: " 
		    << ilen << " / " << udpp->length);
    cleanPacket(udpp);
    return;
  }
  ilen -= hlen; 

  type = icp->ih_opcode;
  length = ilen - hlen;
  flags = ntohl(icp->ih_opt);
  req = icp->ih_req;
  (ipp->_data).move(hlen);
  
  if (type == ICP_OP_QUERY) {
    client = *((in_addr_t*) ipp->data());
    (ipp->_data).move(sizeof(in_addr_t));
    ilen -= sizeof(in_addr_t);
  }

  if (type != ICP_OP_INVALID 
      && ilen > 0 
      && ipp->_data[ilen-1] == '\0') {
     
    url.init((char *)ipp->data());
    (ipp->_data).move(url.length()+1);
  }
  
  if (type == ICP_OP_HIT_OBJ) {
    if (url.length() < 0 && ilen > 0) {
      url.init((char *)ipp->data());
      (ipp->_data).move(url.length()+1);
    }
    size = *((u_short *) ipp->data());
  }

  if (flags & ICP_FLAG_SRC_RTT) {
    mrtt = ntohs(icp->ih_optdata & 0x0000FFFF);
  }

  timeStamp = udpp->timeStamp;  
  packetSetUp(udpp);
}

ICPPacket::ICPPacket(const ICPPacket& x) 
  : Packet(x), type(x.type), length(x.length), flags(x.flags), req(x.req), 
    mrtt(x.mrtt), size(x.size), client(x.client), url(x.url)
{
}

ICPPacket& ICPPacket::operator= (const ICPPacket& x) 
{
  Packet::operator=(x);
  type = x.type; length = x.length; flags = x.flags; req = x.req; 
  mrtt = x.mrtt; size = x.size; client = x.client; url = x.url;
  return *this;
}

void ICPPacket::print(ostream *f)
{
  locatePacket(UDPPacket,	udpp,	this);
  locatePacket(IPPacket,	ipp,	udpp);
  
  *f << timeStamp << '\t'
     << "[icp] "
     << ipp->src << ':' << ntohs(udpp->sport) << ' ' 
     << ipp->dst << ':' << ntohs(udpp->dport) << ' ';

  *f << (int) type << ' ' 
     << (ICP_IS_QUERY(type) ? '>' : '<') << ' '
     << url << ' ';
  *f << endl;
}

size_t ICPPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(type);
  serialVar(length);
  serialVar(flags);
  serialVar(req);
  serialVar(mrtt);
  serialVar(size);
  serialVar(client);
  serialVar(url);

  return count;
}

size_t ICPPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(length);
  unserialVar(flags);
  unserialVar(req);
  unserialVar(mrtt);
  unserialVar(size);
  unserialVar(client);
  unserialVar(url);

  return count;
}

extern_pandora(algo, bool, icpid, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(ICPPacket, icpp, pkt);
  if (icpp == NULL) return false;

  k->set(icpp->req);
  
  return true;
}		 

extern_pandora(algo, bool, icpreq, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(ICPPacket, icpp, pkt);
  locatePacket(IPPacket,   ipp,  icpp);
  if (ipp == NULL) return false;

  if (ipp->src.s_addr < ipp->dst.s_addr) {
    k->set(ipp->src.s_addr, ipp->dst.s_addr, icpp->req, 0);
  } else {
    k->set(ipp->dst.s_addr, ipp->src.s_addr, icpp->req, 0);
  }

  return true;
}		 
