/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include <pandora_components/icmppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/extract.h>
#include <pandora_components/udp.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>
#include <libpandora/error.h>

packet_export(ICMPPacket, IPPacket);

ICMPPacket::ICMPPacket(IPPacket *ipp) 
  : type(ICMP_UNDEF), id((u_int32_t) -1), length(0)
{
  if (ipp == NULL) return;

  if (ipp->dlength() < ICMP_MINLEN) {
    pandora_warning("truncated ICMP packet");
    cleanPacket(ipp);
    return;
  }

  struct icmp *icmp_hdr = (struct icmp *)(ipp->data());
  
  length = ipp->length - sizeof(struct icmp);

  type = icmp_hdr->icmp_type;
  code = icmp_hdr->icmp_code;
  (ipp->_data).move(4);
  memcpy((u_char *)&data, ipp->data(), 
	 pandora_min(ipp->dlength(), sizeof(icmp_t)));
  id = ((((u_int32_t )data.icmp_id) << 16) + (u_int32_t)data.icmp_seq);
  //id = data.icmp_seq;

  timeStamp = ipp->timeStamp;
  packetSetUp(ipp);
}

ICMPPacket::ICMPPacket(const ICMPPacket& x) 
  : Packet(x), type(x.type), code(x.code), data(x.data), id(x.id),
    length(0)
{
}

ICMPPacket& ICMPPacket::operator= (const ICMPPacket& x) 
{
  Packet::operator=(x);
  type = x.type; code = x.code; data = x.data; id = x.id;
  length = 0;
  return *this;
}

/* rfc1700 */
#ifndef ICMP_UNREACH_NET_UNKNOWN
#define ICMP_UNREACH_NET_UNKNOWN	6	/* destination net unknown */
#endif
#ifndef ICMP_UNREACH_HOST_UNKNOWN
#define ICMP_UNREACH_HOST_UNKNOWN	7	/* destination host unknown */
#endif
#ifndef ICMP_UNREACH_ISOLATED
#define ICMP_UNREACH_ISOLATED		8	/* source host isolated */
#endif
#ifndef ICMP_UNREACH_NET_PROHIB
#define ICMP_UNREACH_NET_PROHIB		9	/* admin prohibited net */
#endif
#ifndef ICMP_UNREACH_HOST_PROHIB
#define ICMP_UNREACH_HOST_PROHIB	10	/* admin prohibited host */
#endif
#ifndef ICMP_UNREACH_TOSNET
#define ICMP_UNREACH_TOSNET		11	/* tos prohibited net */
#endif
#ifndef ICMP_UNREACH_TOSHOST
#define ICMP_UNREACH_TOSHOST		12	/* tos prohibited host */
#endif

/* rfc1716 */
#ifndef ICMP_UNREACH_FILTER_PROHIB
#define ICMP_UNREACH_FILTER_PROHIB	13	/* admin prohibited component*/
#endif
#ifndef ICMP_UNREACH_HOST_PRECEDENCE
#define ICMP_UNREACH_HOST_PRECEDENCE	14	/* host precedence violation */
#endif
#ifndef ICMP_UNREACH_PRECEDENCE_CUTOFF
#define ICMP_UNREACH_PRECEDENCE_CUTOFF	15	/* precedence cutoff */
#endif

/* rfc1256 */
#ifndef ICMP_ROUTERADVERT
#define ICMP_ROUTERADVERT		9	/* router advertisement */
#endif
#ifndef ICMP_ROUTERSOLICIT
#define ICMP_ROUTERSOLICIT		10	/* router solicitation */
#endif

/* Most of the icmp types */
static struct tok icmp2str[] = {
	{ ICMP_ECHOREPLY,		"echo reply" },
	{ ICMP_SOURCEQUENCH,		"source quench" },
	{ ICMP_ECHO,			"echo request" },
	{ ICMP_ROUTERSOLICIT,		"router solicitation" },
	{ ICMP_TSTAMP,			"time stamp request" },
	{ ICMP_TSTAMPREPLY,		"time stamp reply" },
	{ ICMP_IREQ,			"information request" },
	{ ICMP_IREQREPLY,		"information reply" },
	{ ICMP_MASKREQ,			"address mask request" },
	{ 0,				NULL }
};

/* Formats for most of the ICMP_UNREACH codes */
static struct tok unreach2str[] = {
	{ ICMP_UNREACH_NET,		"net %s unreachable" },
	{ ICMP_UNREACH_HOST,		"host %s unreachable" },
	{ ICMP_UNREACH_SRCFAIL,
	    "%s unreachable - source route failed" },
	{ ICMP_UNREACH_NET_UNKNOWN,	"net %s unreachable - unknown" },
	{ ICMP_UNREACH_HOST_UNKNOWN,	"host %s unreachable - unknown" },
	{ ICMP_UNREACH_ISOLATED,
	    "%s unreachable - source host isolated" },
	{ ICMP_UNREACH_NET_PROHIB,
	    "net %s unreachable - admin prohibited" },
	{ ICMP_UNREACH_HOST_PROHIB,
	    "host %s unreachable - admin prohibited" },
	{ ICMP_UNREACH_TOSNET,
	    "net %s unreachable - tos prohibited" },
	{ ICMP_UNREACH_TOSHOST,
	    "host %s unreachable - tos prohibited" },
	{ ICMP_UNREACH_FILTER_PROHIB,
	    "host %s unreachable - admin prohibited component" },
	{ ICMP_UNREACH_HOST_PRECEDENCE,
	    "host %s unreachable - host precedence violation" },
	{ ICMP_UNREACH_PRECEDENCE_CUTOFF,
	    "host %s unreachable - precedence cutoff" },
	{ 0,				NULL }
};

/* Formats for the ICMP_REDIRECT codes */
static struct tok type2str[] = {
	{ ICMP_REDIRECT_NET,		"redirect %s to net %s" },
	{ ICMP_REDIRECT_HOST,		"redirect %s to host %s" },
	{ ICMP_REDIRECT_TOSNET,		"redirect-tos %s to net %s" },
	{ ICMP_REDIRECT_TOSHOST,	"redirect-tos %s to net %s" },
	{ 0,				NULL }
};

/* rfc1191 */
struct mtu_discovery {
	short unused;
	short nexthopmtu;
};

/* rfc1256 */
struct ih_rdiscovery {
	u_char ird_addrnum;
	u_char ird_addrsiz;
	u_short ird_lifetime;
};

struct id_rdiscovery {
	u_int32_t ird_addr;
	u_int32_t ird_pref;
};



void ICMPPacket::print(ostream *f)
{
  locatePacket(IPPacket, ipp, this);
  
  *f << timeStamp << '\t'
     << "[icmp] " 
     << ipp->src << ' ' << ipp->dst << ' '
     << length
     << " [" << (int)type << ':' << (int)code << "] "
     << id << ' ';


  switch (type) {
  register const struct icmp *dp;
  register const char *fmt;
  register const struct ip *oip;
  register const struct udphdr *ouh;
  register u_int hlen, dport, mtu;

  case ICMP_UNREACH:

    switch (code) {

    case ICMP_UNREACH_PROTOCOL:
      *f << data.icmp_ip.ip_dst
	 << " protocol " << data.icmp_ip.ip_p
	 << " unreachable ";
      break;

    case ICMP_UNREACH_PORT:
      oip = &data.icmp_ip;
      hlen = IP_HL(oip) * 4;
      ouh = (const struct udphdr *)(((u_char *)oip) + hlen);
      dport = ntohs(ouh->uh_dport);
      *f <<  oip->ip_dst
         << " protocol " << oip->ip_p
         << " port " << dport
         <<" unreachable ";
      break;

    case ICMP_UNREACH_NEEDFRAG:
      {
	register const struct mtu_discovery *mp;

	mp = (const struct mtu_discovery *)&data.icmp_void;
	mtu = EXTRACT_16BITS(&mp->nexthopmtu);
	if (mtu)
	  *f << data.icmp_ip.ip_dst
	     << " unreachable - need to frag (mtu " << mtu << ") ";
	else
	  *f << data.icmp_ip.ip_dst
	     << " unreachable - need to frag ";
      }
      break;

    default:
      fmt = tok2str(unreach2str, "#%d %%s unreachable ",
		    (int)code);
#if 0
#ifdef __GNUC__
      f->form(fmt, intoa(data.icmp_ip.ip_dst.s_addr));
#endif
#endif
      break;
    }
    break;

  case ICMP_REDIRECT:

    fmt = tok2str(type2str, "redirect-#%d %%s to net %%s ",
		  (int)code);
#if 0
#ifdef __GNUC__
    f->form(fmt, intoa(data.icmp_ip.ip_dst.s_addr), 
	    intoa(data.icmp_gwaddr.s_addr));
#endif
#endif
    break;

  case ICMP_ROUTERADVERT:
    {
      register const struct ih_rdiscovery *ihp;
      register const struct id_rdiscovery *idp;
      u_int lifetime, num, size;

      *f <<  "router advertisement ";
      ihp = (const struct ih_rdiscovery *)&data.icmp_void;
      *f << " lifetime ";
      lifetime = EXTRACT_16BITS(&ihp->ird_lifetime);
      if (lifetime < 60)
	*f <<  lifetime << ' ';
      else if (lifetime < 60 * 60)
	*f <<  lifetime / 60 << ':' << lifetime % 60 << ' ';
      else
	*f << lifetime / 3600 << ':' << (lifetime % 3600) / 60 << ':'
	   << lifetime % 60 << ':' << endl;

      num = ihp->ird_addrnum;
      *f <<  num << ": ";

      size = ihp->ird_addrsiz;
      if (size != 2) {
	*f << " [size " <<  size << "] ";
	break;
      }
      idp = (const struct id_rdiscovery *)&data.icmp_data;
      while (num-- > 0) {

	*f << '{' << intoa(idp->ird_addr) << ' ' 
	   << EXTRACT_32BITS(&idp->ird_pref) << "} ";
      }
    }
    break;

  case ICMP_TIMXCEED:

    switch (code) {

    case ICMP_TIMXCEED_INTRANS:
      *f << "time exceeded in-transit ";
      break;

    case ICMP_TIMXCEED_REASS:
      *f << "ip reassembly time exceeded ";
      break;

    default:
      *f << "time exceeded-#" << (int)code << ' ';
      break;
    }
    break;

  case ICMP_PARAMPROB:
    if (code)
      *f << "parameter problem - code " << (int)code << ' ';
    else {
      *f << "parameter problem - octet " << data.icmp_pptr << ' ';
    }
    break;

  case ICMP_MASKREPLY:
    *f << "address mask is 0x" << hex << (u_int32_t)ntohl(data.icmp_mask);
    break;

  case ICMP_ECHO:
    *f << tok2str(icmp2str, "type-#%d", (int)type)
       << " id = " << data.icmp_id
       << " seq = " << data.icmp_seq;
    break;

  case ICMP_ECHOREPLY:
    *f << tok2str(icmp2str, "type-#%d", (int)type)
       << " id = " << data.icmp_id
       << " seq = " << data.icmp_seq;
    break;
    
  default:
    *f << tok2str(icmp2str, "type-#%d", (int)type);
    break;
  } 
 
  *f << endl;
}

size_t ICMPPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(type);
  serialVar(code);
  //serialVar(data);
  serialVar(id);
  serialVar(length);

  return count;
}

size_t ICMPPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(code);
  //unserialVar(data);
  unserialVar(id);
  unserialVar(length);
  
  return count;
}

extern_pandora(algo, bool, icmpid, (Packet *pkt, PandoraKey *k))
{
  locatePacket0(ICMPPacket, icmpp, pkt);
  if (icmpp == NULL) return false;

  k->set(icmpp->id);
  
  return true;
}		 
