/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <iomanip>
#include <pandora_components/httptranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/pandorakey.h>
#include <libpandora/algo_funcs.h>
#include <libpandora/serialize.h>
#include <libpandora/timeval.h>

packet_export(HTTPTransPacket, HTTPPacket);

HTTPTransPacket::HTTPTransPacket(HTTPPacket *request, HTTPPacket *response) 
  : type(undefined), pos(alone)
{
  if (request == NULL || response == NULL) {
    cleanPacket(request); cleanPacket(response);
    return;
  }

  type = match;

  request->code =     response->code;
  request->cl =       response->cl;
  request->ce =       response->ce;
  request->ct =       response->ct;
  request->lm =       response->lm;
  request->expires =  response->expires;
  request->gap |=     response->gap;
  request->dynamic |= response->dynamic;

  rversion =          response->version;
  date_s =            response->date;

  rtt = (int)((response->last.tv_sec - request->timeStamp.tv_sec) * 1000
	      + (response->last.tv_usec - request->timeStamp.tv_usec) /1000);
  lat = (int)((response->timeStamp.tv_sec - request->last.tv_sec) * 1000
	      + (response->timeStamp.tv_usec - request->last.tv_usec) /1000);

  //request->last =     response->last;

#if 0
  pandora_assert((long)request->endh >=  (long)request->beg);  
  pandora_assert((long)request->endm >=  (long)request->endh); 
  pandora_assert((long)response->endh>= (long)response->beg); 
  pandora_assert((long)response->endm>= (long)response->endh);
#endif

  reqhlen = (short)  SEQ_SUB(request->endh,  request->beg);  
  reqmlen = (int)    SEQ_SUB(request->endm,  request->endh); 
  resphlen = (short) SEQ_SUB(response->endh, response->beg); 
  respmlen = (int)   SEQ_SUB(response->endm, response->endh);

  timeStamp = request->timeStamp;

  packetSetUp(request);
  cleanPacket(response);
}

HTTPTransPacket::HTTPTransPacket(HTTPPacket *httpp)
  : type(undefined), pos(alone), rtt(0), lat(0), reqhlen(0), reqmlen(0),
    resphlen(0), respmlen(0), date_s(0), rversion(0)
{
  if (httpp == NULL) return;
  type = (httpp->type == HTTPPacket::request) ? request : response;
  timeStamp = httpp->timeStamp;
  packetSetUp(httpp);
}

HTTPTransPacket::HTTPTransPacket(const HTTPTransPacket& x) 
  : Packet(x), type(x.type), pos(x.pos), rtt(x.rtt), lat(x.lat), 
    reqhlen(x.reqhlen), reqmlen(x.reqmlen), 
    resphlen(x.resphlen), respmlen(x.respmlen),
    date_s(x.date_s), rversion(x.rversion)
{
}

HTTPTransPacket& HTTPTransPacket::operator=(const HTTPTransPacket& x) 
{
  Packet::operator=(x);
  type = x.type; pos = x.pos; rtt = x.rtt; lat = x.lat;
  reqhlen = x.reqhlen; reqmlen = x.reqmlen; 
  resphlen = x.resphlen; respmlen = x.respmlen;
  date_s = x.date_s; rversion = x.rversion;
  return *this;
}

void HTTPTransPacket::print(ostream *f)
{
  locatePacket(HTTPPacket, httpp, this);
  locatePacket(TCPPacket, tcpp, httpp);
  locatePacket(IPPacket, ipp, tcpp);
  if (ipp == NULL) return;

  *f << timeStamp << ' '
     << setw(6) << setfill(' ')
     << rtt << ' ' << lat << ' '
     << ipp->src << ':' << ntohs(tcpp->sport) << ' ' 
     << ipp->dst << ':' << ntohs(tcpp->dport) << ' '
     << httpp->code << ' '
     << (resphlen+respmlen) 
     << " (" << respmlen << ") ";

  switch (httpp->method) {
  case HTTPPacket::GET:     *f << "GET " ;     break;
  case HTTPPacket::POST:    *f << "POST " ;    break;
  case HTTPPacket::HEAD:    *f << "HEAD " ;    break;
  case HTTPPacket::PUT:     *f << "PUT " ;     break;
  case HTTPPacket::OPTIONS: *f << "OPTIONS " ; break;
  case HTTPPacket::TRACE:   *f << "TRACE " ;   break;
  case HTTPPacket::DELETE:  *f << "DELETE " ;  break;
  case HTTPPacket::UNDEF: default:             break;
  }
#if 1
  *f << httpp->url << ' ';
#else
  *f << "(hashed)" << ' ';
#endif

  *f << httpp->version << ' '
     << httpp->ct << ' ' << httpp->ce << ' '
     << "[" << httpp->ref << "] "
     << (httpp->absolute ? 'A' : '-' )
     << (httpp->dynamic ? 'D' : '-' )
     << (httpp->gap ? 'G' : '-' ) 
     << (httpp->pragma == HTTPPacket::no_cache ? "N" : "-")
     << (httpp->coll > 0 ? 'P' : '-');
  *f << endl;
}
  

size_t HTTPTransPacket::write(char *str, size_t maxlen, int level)
{
  size_t count = 0;

  serialVar(type);
  serialVar(pos);
  serialVar(rtt);
  serialVar(lat);
  serialVar(reqhlen);
  serialVar(reqmlen);
  serialVar(resphlen);
  serialVar(respmlen);
  serialVar(date_s);
  serialVar(rversion);
  
  return count;
}

size_t HTTPTransPacket::read(const char *str, int level)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(pos);
  unserialVar(rtt);
  unserialVar(lat);
  unserialVar(reqhlen);
  unserialVar(reqmlen);
  unserialVar(resphlen);
  unserialVar(respmlen);
  unserialVar(date_s);
  unserialVar(rversion);

  return count;
}
