/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef HTTP_SCAN_COMPONENT_H
#define HTTP_SCAN_COMPONENT_H

#include <libpandora/global.h>

#include <pandora_components/tcppacket.h>
#include <pandora_components/httppacket.h>
#include <libpandora/component.h>
#include <libpandora/encoding.h>
#include <libpandora/common_options.h>
#include <libpandora/timeval.h>

#define NO_FLEX 1
#define PERSIST

class HTTPScanComponent;
class packetData;
extern int headerlex(HTTPScanComponent *c);
extern void init_header_buffer(char *str, int len);
extern void release_header_buffer(void);

struct stringBuf {
private:
  char *buf;
  char *tmp;
  int len;
  
public:
  inline stringBuf() : buf(NULL), tmp(NULL), len(0) {}
  inline ~stringBuf() { __FREE(buf); __FREE(tmp); }

  inline void save(const char *str, const size_t l) {
    __FREE(buf);
    buf = (char *)xmalloc(l*sizeof(char));
    memcpy(buf, str, l*sizeof(char));
    len = l;
  }

  inline size_t restore(char **dst, const char *str, const size_t l) {
    pandora_assert( dst != NULL);
    size_t ret = (size_t) len;
    clean();
    tmp = (*dst) = (char *)xmalloc((len+l+2)*sizeof(char));
    if ( len > 0) {
      memcpy(*dst, buf, len*sizeof(char));
      __FREE(buf);
      len = 0;
    }
    memcpy(*dst + ret, str, l*sizeof(char));
    (*dst)[ret+l] = (*dst)[ret+l+1] = '\0';
    return ret;
  }

  inline bool isEmpty(void) { return (len == 0); }
  inline void clean(void) { __FREE(tmp); }
  inline void reset(void) { __FREE(tmp); len = 0;}
};

  
#define HDR_ERROR	-2
#define ACT_MSG_LEN	(SEQ_SUB(actSeq, msg->endh))
#define SEQ_GO(x)	{					\
  data = (x);							\
  if (data > 0) actSeq = SEQ_ADD(tcpp->seq, data);		\
  else actSeq = SEQ_SUB(tcpp->seq, -data);			\
}  
#define SEQ_MOVE(x)	{					\
  data += (x);							\
  if (data > 0) actSeq = SEQ_ADD(tcpp->seq, data);		\
  else actSeq = SEQ_SUB(tcpp->seq, -data);			\
}  
#define SEQ_UPDATE	{					\
  if (data > 0) actSeq = SEQ_ADD(tcpp->seq, data);		\
  else actSeq = SEQ_SUB(tcpp->seq, -data);			\
}  

class HTTPScanComponent : public Component {
public:
  HTTPPacket *msg;
  encoding *MIME, *ENC;

private:
  u_int32_t actSeq;
  long actLength;
  bool fast;
  bool dump;
  
  TCPPacket *tcpp;
  packetData *ipdata;

  int data, unmatched;
  HTTPPacket::http_t flowType;
  stringBuf hdrbuf, reqbuf, respbuf;
  bool gap;

  static Mutex mx;

public:
  component_init(HTTPScanComponent, 1);
  inline  HTTPScanComponent(void);
  virtual ~HTTPScanComponent(void)  { }
    
  virtual bool add(Packet *);
  virtual void cleanup(void);

  static void openEncoding(const char *fname, void *e);
  
private:
  inline void dispatch(void);
  inline void newHTTPpkt(void);
  inline void endHeader(void) { if (msg != NULL) msg->endh = actSeq; }
  inline void endHTTPpkt(bool mmsg);
  inline void adjustHTTPpkt(void);
  inline int match(const char *, size_t, const char *, size_t, stringBuf *);  
  int matchRequest(void);
  int matchResponse(void);
  void parseHeaders(void);

  inline bool keepGoing(void);
  inline int parseVersion(const char *str);
  inline int matchFlow(void);

  friend int headerlex(HTTPScanComponent *);
#if NO_FLEX
  int doParse(char *str, int len);
#endif
};


HTTPScanComponent::HTTPScanComponent(void) 
  : msg(NULL), MIME(NULL), ENC(NULL), actSeq(0), actLength(0),
    fast(false), /*dump(false),*/ flowType(HTTPPacket::unknown), 
    gap(false)
{
  registerStaticOption(template_option<encoding>, "mime", &MIME);
  registerStaticOption(template_option<encoding>, "enc", &ENC);
  registerOption("fast", &fast);
  //registerOption("dump", &dump);
}

void HTTPScanComponent::dispatch(void) 
{
  if (msg != NULL) {
    msg->urlCanonicalize();    
    msg->checkDynamicity();
    push(msg);
    msg = NULL;
  }
}


void HTTPScanComponent::newHTTPpkt(void)
{
  //pandora_debug("new");
  if (msg != NULL) endHTTPpkt(true);
  pandora_assert(tcpp != NULL);
  msg = new HTTPPacket(tcpp);
  msg->beg = actSeq;
}


void HTTPScanComponent::endHTTPpkt(bool mmsg) 
{
  //pandora_debug("end " << mmsg);
  if (msg != NULL) {
    if (!msg->hdrOK) endHeader();
    if (!msg->timeStamp.tv_sec) adjustHTTPpkt();
    msg->gap = gap;
    msg->endm = actSeq;
    msg->eof = mmsg;
    dispatch();
  }
}


void HTTPScanComponent::adjustHTTPpkt(void)
{
  pandora_assert(tcpp != NULL);
  if (tcpp->length == 0) return;

  if (msg->timeStamp.tv_sec == (time_t) 0) {
    msg->timeStamp = tcpp->timeStamp;
    msg->last = tcpp->timeStamp;
    return;
  }

  if (tcpp->timeStamp < msg->timeStamp) {
    msg->timeStamp = tcpp->timeStamp;
  }

  if (compareTimeStamp(tcpp->timeStamp, msg->last) > 0) {
    msg->last = tcpp->timeStamp;
  }
}

int HTTPScanComponent::match(const char *ptr, size_t len, 
			     const char *str, size_t s, 
			     stringBuf *sbuf) 
{
#ifdef PERSIST
  if (len < s) {
    sbuf->save(ptr, len);
    return -1;
  }
#endif    
  if (strncmp(ptr, str, s) == 0) return s;
  else return -1;
}

bool HTTPScanComponent::keepGoing(void) 
{ 
  if ((msg != NULL) && (msg->hdrOK) && (msg->cl > 0)) {
    int actMsgLen = ACT_MSG_LEN;
    if ((actMsgLen + (tcpp->length - data)) > msg->cl) {
      size_t data_off = (msg->cl < actMsgLen) ? 0 : (msg->cl - actMsgLen);
      SEQ_MOVE(pandora_min((int)data_off, (int)tcpp->length));
      adjustHTTPpkt();	
      return true;
    } else {
      return false;
    }
  } else {
    return (data < tcpp->length);
  }
}

int HTTPScanComponent::parseVersion(const char *str) 
{ // only for HTTP/x.y
  if (str == NULL) return 0;
  return (1000 * (str[0] - '0')) + (str[2] - '0');
}

int HTTPScanComponent::matchFlow(void) 
{
  int off = -1;
    
  switch(flowType) {
  case HTTPPacket::response:	return matchResponse();
  case HTTPPacket::request:	return matchRequest();
  case HTTPPacket::unknown:
    if ((off = matchResponse()) > 0) {
      flowType = HTTPPacket::response;
      return off;
    }
    if ((off = matchRequest()) > 0) { 
      flowType = HTTPPacket::request;
      return off;
    }
    break;
  default:	pandora_warning("invalid flow type"); break;
  }    
  return -1;
}

#endif /* HTTP_SCAN_COMPONENT_H */
