/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <strstream>

#include <libpandora/stackfactory.h>
#include <libpandora/pandoraentry.h>
#include <libpandora/packet.h>
#include "guile_pandora_types.h"
#include "guile_pandora_proc.h"

#define PIPE_DEBUG	0

long option_tag = 0, comp_tag = 0, stack_tag = 0;
long packet_pipe_tag = 0;

Map<text, SCM> packet_pipes;
static void register_pipe(SCM);
static void release_pipe(SCM);

static SCM mark_option(SCM s_poption)
{
  return SCM_BOOL_F;
}

static scm_sizet free_option(SCM s_poption)
{
  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);

  __DELETE(poption);

  return 0;
}

static int print_option(SCM s_poption, SCM port, scm_print_state *pstate)
{
  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);

  scm_puts("#<option ", port);
  scm_puts((poption->id).data(), port); 
  if (!(poption->alias).isNull() && (poption->alias != poption->id)) {
    scm_puts(":", port);
    scm_puts((poption->alias).data(), port);     
  }
  if ((poption->mv).type != MultiValue::undefined) {
    scm_puts("=", port);
    scm_display(mv2scm(poption->mv), port);
  }
  scm_puts(">", port);

  return 1;
}

static SCM mark_comp(SCM s_pcomp)
{
  return SCM_BOOL_F;
}

static scm_sizet free_comp(SCM s_pcomp)
{
  CompEntry *pcomp = (CompEntry *) SCM_CDR(s_pcomp);

  __DELETE(pcomp);

  return 0;
}

static int print_comp(SCM s_pcomp, SCM port, scm_print_state *pstate)
{
  CompEntry *pcomp = (CompEntry *) SCM_CDR(s_pcomp);

  if (pcomp->isMacro()) {
    scm_puts("#<macro ", port);
  } else {
    scm_puts("#<component ", port);
  }

  scm_puts((pcomp->id).data(), port);   
  if (!(pcomp->alias).isNull() && (pcomp->alias != pcomp->id)) {
    scm_puts(":", port);
    scm_puts((pcomp->alias).data(), port);     
  }
  scm_puts("(", port);
  scm_display(SCM_MAKINUM(pcomp->getNbOptions()), port);
  scm_puts(")>", port);

  return 1;
}

static SCM mark_stack(SCM s_pstack)
{
  return SCM_BOOL_F;
}

static scm_sizet free_stack(SCM s_pstack)
{
  StackEntry *pstack = (StackEntry *) SCM_CDR(s_pstack);

  __DELETE(pstack);

  return 0;
}

static int print_stack(SCM s_pstack, SCM port, scm_print_state *pstate)
{
  StackEntry *pstack = (StackEntry *) SCM_CDR(s_pstack);

  scm_puts("#<stack ", port);
  scm_puts((pstack->id).data(), port); scm_puts("(", port);
  scm_display(SCM_MAKINUM(pstack->getNbComps()), port);
  scm_puts(")>", port);

  return 1;
}

static SCM mark_packet_pipe(SCM s_ppipe)
{
  pandora_packet_pipe *ppipe = (pandora_packet_pipe *) SCM_CDR(s_ppipe);
  return SCM_BOOL_F;
}

static scm_sizet free_packet_pipe(SCM s_ppipe)
{
  pandora_packet_pipe *ppipe = (pandora_packet_pipe *) SCM_CDR(s_ppipe);
  release_pipe(s_ppipe);

  scm_sizet size = sizeof(pandora_packet_pipe);
  packet_pipe_t *pktp = ppipe->pkt_pipe;
  ppipe->pkt_pipe = NULL;
  if (pktp != NULL) {
    while(!pktp->isEmpty()) {
      Packet *pkt = pktp->get();
      cleanPacket(pkt);
    }
    delete pktp;
  }
  __FREE(ppipe);
  return size;
}

static int print_packet_pipe(SCM s_ppipe, SCM port, scm_print_state *pstate)
{
  pandora_packet_pipe *ppipe = (pandora_packet_pipe *) SCM_CDR(s_ppipe);

  scm_puts("#<packet-pipe ", port);
  scm_display(text2scm(ppipe->id), port);	scm_puts(" ", port);
  scm_display((ppipe->pkt_pipe != NULL 
	       ? SCM_MAKINUM(ppipe->pkt_pipe->size())
	       : SCM_BOOL_F), 
	      port);
  scm_puts(">", port);

  return 1;
}

void pandora_init_types(void) {
  option_tag = scm_make_smob_type_mfpe("pandora-option", 
				       sizeof(OptionEntry),
				       mark_option, free_option, 
				       print_option, NULL);
  comp_tag = scm_make_smob_type_mfpe("pandora-comp", sizeof(CompEntry),
				     mark_comp, free_comp, 
				     print_comp, NULL);
  stack_tag = scm_make_smob_type_mfpe("pandora-stack", sizeof(StackEntry),
				      mark_stack, free_stack, 
				      print_stack, NULL);
  packet_pipe_tag = scm_make_smob_type_mfpe("pandora-packet-pipe", 
					    sizeof(pandora_packet_pipe),
					    mark_packet_pipe, 
					    free_packet_pipe, 
					    print_packet_pipe, 
					    NULL);
}

SCM mv2scm(const MultiValue &mv)
{
  SCM value;

  switch (mv.type) {
  case MultiValue::integer: 	value = gh_long2scm(mv.value.d);
    break;			
  case MultiValue::boolean: 	value = SCM_BOOL(mv.value.b);
    break;			
  case MultiValue::floating: 	value = gh_double2scm(mv.value.f);
    break;			
  case MultiValue::textual: 	value = gh_str02scm(mv.value.s);
    break;
  case MultiValue::pointer:	/* fall through */
  default:			value = SCM_UNDEFINED; 
    break;
  }
  
  return value;
}

MultiValue::multi_value_t pandora_get_value_type(SCM s_value)
{
  if (SCM_INUMP(s_value)) 	return MultiValue::integer;
  else if(SCM_STRINGP(s_value)) return MultiValue::textual;
  else if(SCM_BOOLP(s_value)) 	return MultiValue::boolean;
  else if(SCM_REALP(s_value)) 	return MultiValue::floating;
  else 				return MultiValue::undefined;
}

void scm2mv(SCM s_value, MultiValue &mv)
{
  mv.type = pandora_get_value_type(s_value);
  switch (mv.type) {
  case MultiValue::integer: 	mv.value.d= SCM_INUM(s_value); 		  break;
  case MultiValue::boolean: 	mv.value.b= SCM_NFALSEP(s_value); 	  break;
  case MultiValue::floating: 	mv.value.f=gh_scm2double(s_value);  	  break; 
  case MultiValue::textual: 	mv.value.s= gh_scm2newstr(s_value, NULL); break;
  case MultiValue::pointer:	/* fall through */
  default:			mv.value.v = NULL;     			  break;
  }
}

SCM oe2scm(const OptionEntry *oe)
{
  OptionEntry *poption = new OptionEntry(*oe);
  SCM_RETURN_NEWSMOB(option_tag, poption);
}

SCM take_oe(OptionEntry *oe)
{
  OptionEntry *poption = oe;
  oe = NULL;
  SCM_RETURN_NEWSMOB(option_tag, poption);  
}

void scm2oe(SCM s_poption, OptionEntry *oe)
{
  OptionEntry *poption = (OptionEntry *) SCM_CDR(s_poption);
  *oe = *poption;
}

OptionEntry *scm2oe(SCM s_poption)
{
  return (OptionEntry *) SCM_CDR(s_poption);
}

SCM pandora_make_option(void)
{
  OptionEntry *poption = new OptionEntry();

  SCM_RETURN_NEWSMOB (option_tag, poption);
}

SCM ce2scm(const CompEntry *ce)
{
  CompEntry *pcomp = new CompEntry(*ce);
  SCM_RETURN_NEWSMOB (comp_tag, pcomp);
}

SCM take_ce(CompEntry *ce)
{
  CompEntry *pcomp = ce;
  ce = NULL;
  SCM_RETURN_NEWSMOB (comp_tag, pcomp);
}

void scm2ce(SCM s_pcomp, CompEntry *ce)
{
  CompEntry *pcomp = (CompEntry *) SCM_CDR(s_pcomp);
  *ce = *pcomp;
}

CompEntry *scm2ce(SCM s_pcomp)
{
  return (CompEntry *) SCM_CDR(s_pcomp);
}

SCM pandora_make_component(void)
{
  CompEntry *pcomp = new CompEntry(CompEntry::component, NULL);
  SCM_RETURN_NEWSMOB (comp_tag, pcomp);
}

SCM pandora_make_macro(void)
{
  CompEntry *pcomp = new CompEntry(CompEntry::macro, NULL);
  SCM_RETURN_NEWSMOB (comp_tag, pcomp);
}


SCM se2scm(const StackEntry *se)
{
  StackEntry *pstack =  new StackEntry(*se);
  SCM_RETURN_NEWSMOB (stack_tag, pstack);
}

SCM take_se(StackEntry *se)
{
  StackEntry *pstack = se;
  se = NULL;
  SCM_RETURN_NEWSMOB (stack_tag, pstack);
}

void scm2se(SCM s_pstack, StackEntry *se)
{
  StackEntry *pstack = (StackEntry *) SCM_CDR(s_pstack);
  *se = *pstack;
}

StackEntry *scm2se(SCM s_pstack)
{
  return (StackEntry *) SCM_CDR(s_pstack);
}

SCM pandora_make_stack(void)
{
  StackEntry *pstack =  new StackEntry();

  SCM_RETURN_NEWSMOB (stack_tag, pstack);
}


SCM pandora_make_packet_pipe(SCM s_name)
{
  SCM_ASSERT(SCM_NIMP(s_name) && SCM_STRINGP(s_name), s_name, SCM_ARG1, 
	     "make-pandora-packet-pipe");
  
  text name(SCM_ROCHARS(s_name));
  
  SCM s_ppipe = pandora_get_pipe(name);
  if (s_ppipe != SCM_BOOL_F) return s_ppipe;

  pandora_packet_pipe *ppipe =  (pandora_packet_pipe *)
    scm_must_malloc(sizeof(pandora_packet_pipe), "pandora-packet-pipe");

  (ppipe->id).reset();
  ppipe->id = name;
  ppipe->pkt_pipe = new packet_pipe_t();

  SCM_NEWSMOB(s_ppipe, packet_pipe_tag, ppipe);
  register_pipe(s_ppipe);
  return s_ppipe;
}

static bool _pandora_parse(StackFactory *sf, SCM *ret)
{
  PandoraEntry pe;
  if (!sf->parse(&pe)) return false;

  if (pe.stack() != NULL) {
    *ret = se2scm(pe.stack());
  } else if (pe.component() != NULL) {
    *ret = ce2scm(pe.component());
  } else if (pe.option() != NULL) {
    *ret = oe2scm(pe.option());
  } else {
    return false;
  }
  return true;
}

SCM pandora_parse(SCM s_str)
{
  SCM_ASSERT(SCM_NIMP(s_str) && SCM_STRINGP(s_str), s_str, SCM_ARG1, 
	     "pandora-parse");
  char *str = SCM_ROCHARS(s_str);
  StackFactory sf(str);

  SCM ret = SCM_BOOL_F;
  SCM s_list = SCM_LIST0;

  while (_pandora_parse(&sf, &ret))
    s_list = gh_cons(ret, s_list);
  
  return gh_reverse(s_list);
}

SCM pandora_print(SCM s_obj)
{
  char buf[1024];
  bzero(buf, sizeof(buf));
  ostrstream ostr(buf, sizeof(buf));
  int len = 0;
  
  if (SCM_POPTIONP(s_obj)) {
    scm2oe(s_obj)->print(&ostr);
  } else if (SCM_PCOMPP(s_obj)) {
    scm2ce(s_obj)->print(&ostr);
  } else if (SCM_PSTACKP(s_obj)) {
    scm2se(s_obj)->print(&ostr);
  } else {
    scm_wrong_type_arg("pandora-print", SCM_ARG1, s_obj);
  }

  len = strlen(buf);
  if (len == 0) return SCM_BOOL_F;
  while(isspace(buf[len-1])) buf[--len] = '\0';

  return gh_str2scm(buf, len);
}

void register_pipe(SCM s_ppipe)
{
#if PIPE_DEBUG
  SCM_ASSERT(SCM_PPIPEP(s_ppipe), s_ppipe, SCM_ARG1, "register_pipe");
#endif
  pandora_packet_pipe *ppipe = (pandora_packet_pipe *) SCM_CDR(s_ppipe);

#if PIPE_DEBUG
  pandora_debug("inserting pipe: \"" << ppipe->id << "\" [" 
	       << packet_pipes.size() << "]");
#endif
  packet_pipes.atPut(ppipe->id, s_ppipe);
}

void release_pipe(SCM s_ppipe)
{
#if PIPE_DEBUG
  SCM_ASSERT(SCM_PPIPEP(s_ppipe), s_ppipe, SCM_ARG1, "release_pipe");
#endif
  pandora_packet_pipe *ppipe = (pandora_packet_pipe *) SCM_CDR(s_ppipe);

#if PIPE_DEBUG
  pandora_debug("removing  pipe: \"" << ppipe->id << "\" [" 
	       << packet_pipes.size() << "]");
#endif
  packet_pipes.removeKey(ppipe->id);
}

SCM pandora_get_pipe(const text &name)
{
#if PIPE_DEBUG
  pandora_debug("getting   pipe: \"" << name << "\" [" 
	       << packet_pipes.size() << "]");
#endif
  SCM s_ppipe = packet_pipes.atOrNil(name);
  if (s_ppipe == 0) return SCM_BOOL_F;
  return s_ppipe;
}

packet_pipe_t *pandora_get_pipe_ptr(SCM s_ppipe)
{
  //SCM_ASSERT(SCM_PPIPEP(s_ppipe), s_ppipe, SCM_ARG1, "pandora_get_pipe_ptr");
  if (s_ppipe == SCM_BOOL_F) return NULL;
  pandora_packet_pipe *ppipe = (pandora_packet_pipe *) SCM_CDR(s_ppipe);
  return ppipe->pkt_pipe;
}
