/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <stdlib.h>
#include <libpandora/conf/string.h>
#include <limits.h>
}

#include <pandora_components/fseventpacket.h>
#include <pandora_components/syscallpacket.h>
#include <pandora_components/valuepacket.h>
#include <libpandora/timeval.h>
#include <libpandora/serialize.h>

packet_export(FSEventPacket, SyscallPacket|TextValuePacket);

FSEventPacket::FSEventPacket(SyscallPacket *sp)
  : type(UNDEF), success(true)
{
  update(sp);
  cleanPacket(sp);
}

void FSEventPacket::update(SyscallPacket *sp)
{
  switch(sp->type) {
  case SyscallPacket::OPEN:	case SyscallPacket::MKDIR:  
    setType(CREATED);	
    break;
  case SyscallPacket::UNLINK:	case SyscallPacket::RMDIR: 
    setType(DELETED);	
    break;
  case SyscallPacket::READ:  
    setType(ACCESSED);
    break;
  case SyscallPacket::WRITE: 
    setType(MODIFIED);
    break;
  default: 
    break;
  }
  
  if (((sp->args).size() > 0) 
      && (sp->args[0].type == MultiValue::textual)) {
    setPath(&path, (sp->cwd).data(), (sp->args[0]).value.s);
  }
  
  if ((timeStamp.tv_sec == 0) || (sp->timeStamp < timeStamp))
    timeStamp = sp->timeStamp;
}

FSEventPacket::FSEventPacket(TextValuePacket *vp)
  : type(UNDEF), success(true)
{
  if (vp == NULL) return;  
  if (*((vp->val).data()) != '/') {
    cleanPacket(vp);
    return;
  }
  path.init(vp->val);
  pathCanonicalize(&path, path.data());
  type = MODIFIED;
  timeStamp = vp->timeStamp;
  cleanPacket(vp);
}

FSEventPacket::FSEventPacket(char *line, fseventt_t t)
  : type(t), success(true)
{
  init(line);
  setTimeStamp();
}
  
void FSEventPacket::init(char *line)
{
  if (line == NULL) return;

  const char *delims = " \t\r\n";
  char *p = strtok(line, delims);
  if (p == NULL) return;
  setPath(&path, NULL, p);

  if (type != UNDEF)  return;

  char *op = strtok(NULL, delims);
  if (op != NULL) {
    type = (fseventt_t)atoi(op);
  } else {
    type = MODIFIED;
  }
}

FSEventPacket::FSEventPacket(const FSEventPacket& x) 
  : Packet(x), type(x.type), success(x.success), path(x.path)
{
}

FSEventPacket& FSEventPacket::operator= (const FSEventPacket& x)
{
  Packet::operator=(x);
  type = x.type; success = x.success; path = x.path;
  return *this;
}

void FSEventPacket::print(ostream *f) 
{
  *f << timeStamp << '\t' 
     << "[fs] #" << (int)type << "/" << success
     << "\t" << path;
  *f  << endl;
}


void FSEventPacket::setPath(text *p, const char *cwd, const char *pwd)
{
  if (pwd == NULL) return;

  if (pwd[0] == '/') {
    strncpy(path_buf, pwd, PATH_MAX);
  } else {
    if (cwd != NULL) {
      strncpy(path_buf, cwd, PATH_MAX-1);
      strncat(path_buf, "/", 1);
      strncat(path_buf, pwd, PATH_MAX - strlen(cwd));
    } else {
      strncpy(path_buf, pwd, PATH_MAX);
    }
  }

#if 0
  p->init(PATH_MAX);
  if (realpath(path_buf, p->data()) == NULL) {
    //pandora_warning(path_buf << ": " << strerror(errno));
    p->init(path_buf);
  } else {
    p->update();
  }
#else
  pathCanonicalize(p, path_buf);
#endif
}


void FSEventPacket::pathCanonicalize(text *_path, char *opwd)
{
  if (_path == NULL || opwd == NULL) return;
  char *pwd = opwd;
  char *p = NULL;

  while (*pwd == '/' && *(pwd+1)=='/') {  /* Some URLs start //<foo> */
    pwd += 1;
  } 

  if ((p = pwd)) {
    char *end = pwd + strlen(pwd);

    /* Parse string second time to simplify */
    p = pwd;
    while(p < end) {
      if (*p == '/') {
	if (p > pwd && *(p+1) == '.' && (*(p+2) == '/' || !*(p+2))) {
	  char *orig = p+1;
	  char *dest = (*(p+2)!='/') ? p+2 : p+3;
	  while ((*orig++ = *dest++)); 	      /* Remove a slash and a dot */
	  end = orig-1;
	} else if (*(p+1)=='.' && *(p+2)=='.' && (*(p+3)=='/' || !*(p+3))) {
	  char *q = p;
	  while (q > pwd && *--q != '/');     /* prev slash */
	  if (strncmp(q, "/../", 4)) {
	    char *orig = q+1;
	    char *dest = (*(p+3)!='/') ? p+3 : p+4;
	    while ((*orig++ = *dest++));      /* Remove /xxx/.. */
	    end = orig-1;
	    p = q;		      	      /* Start again with prev slash */
	  } else
	    p++;
	} else if (*(p+1)=='/') {
	  while (*(p+1)=='/') {
	    char *orig=p, *dest=p+1;
	    while ((*orig++ = *dest++));      /* Remove multiple /'s */
	    end = orig-1;
	  }
	} else
	  p++;
      } else
	p++;
    }
  }

  /*
  **  Check for /../.. kind of things
  */
  while (*pwd=='/' && *(pwd+1)=='.' && *(pwd+2)=='.' &&
	 (!*(pwd+3) || *(pwd+3)=='/')) {
    char * orig = pwd;
    char * dest = pwd+3;
    while ((*orig++ = *dest++));
  }
  pandora_assert(*pwd == '/');
  _path->init(pwd);
}

size_t FSEventPacket::write(char *str, size_t maxlen, int)
{
  size_t count = 0;

  serialVar(type);
  serialVar(success);
  serialVar(path);

  return count;
}

size_t FSEventPacket::read(const char *str, int)
{
  size_t count = 0;

  unserialVar(type);
  unserialVar(success);
  unserialVar(path);

  return count;
}
