/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <iostream>
#include <libpandora/error.h>

#include "controlcomponent.h"
#include <pandora_components/controlpacket.h>
#include <libpandora/stackentry.h>
#include <libpandora/resource_entry.h>
#include <libpandora/pandora.h>
#include <libpandora/dynloader.h>
#include <libpandora/globalresourcemanager.h>
#include <libpandora/netutil.h>
#include <libpandora/util.h>

component_export(ControlComponent, ControlPacket, ControlPacket);

void ControlComponent::cleanup(void)
{
  cleanPacket(resp);
}


bool ControlComponent::add(Packet *pkt)
{
  ControlPacket *cp = static_cast<ControlPacket *>(pkt);

  bool ret = false;

  ControlPacket::controlt_t opcode = cp->getOpcode();

  switch (opcode) {
  case ControlPacket::start:		ret = start_stack(cp); 		break;
  case ControlPacket::stop:		ret = stop_stack(cp);  		break;
  case ControlPacket::suspend:		ret = suspend_stack(cp);  	break;
  case ControlPacket::resume:		ret = resume_stack(cp);  	break;
  case ControlPacket::clean:		ret = clean(cp);  		break;
  case ControlPacket::quit:		ret = quit(cp);  		break;
  case ControlPacket::get_stack:	ret = get_stack(cp);  		break;
  case ControlPacket::get_stack_name:	ret = get_stack_name(cp);	break;
  case ControlPacket::set_stack:	ret = set_stack(cp);  		break;
  case ControlPacket::list_defined:	ret = list_defined(cp); 	break;
  case ControlPacket::list_running:	ret = list_running(cp);  	break;
  case ControlPacket::comp_query:	ret = comp_query(cp);  		break;
  case ControlPacket::get_option:	ret = get_option(cp);  		break;
  case ControlPacket::get_option_def:	ret = get_option_def(cp);	break;
  case ControlPacket::set_option:	ret = set_option(cp);  		break;
  case ControlPacket::set_option_live:	ret = set_option_live(cp);	break;
  case ControlPacket::list_options:	ret = list_options(cp);		break;
  case ControlPacket::set_lib:		ret = set_lib(cp);  		break;
  case ControlPacket::get_lib:		ret = get_lib(cp);  		break;
  case ControlPacket::list_libs:	ret = list_libs(cp);  		break;
  case ControlPacket::set_binding:	ret = set_binding(cp);  	break;
  case ControlPacket::get_binding:	ret = get_binding(cp);  	break;
  case ControlPacket::list_symbols:	ret = list_symbols(cp);  	break;
  case ControlPacket::comp_alive:	ret = comp_alive(cp);  		break;
  case ControlPacket::packet_alive:	ret = packet_alive(cp);  	break;
  case ControlPacket::add_res:		ret = add_res(cp); 	 	break;
  case ControlPacket::del_res:		ret = del_res(cp); 	 	break;
  case ControlPacket::set_res_pri:	ret = set_res_pri(cp); 	 	break;
  case ControlPacket::list_res:		ret = list_res(cp);  	 	break;
  case ControlPacket::update_res:	ret = update_res(cp);  	 	break;
  case ControlPacket::undef:	        /* fall through */
  default:				ret = false;  		 	break;
  }

  discard(cp);
  
  response()->writeParam(ret);
  sendResponse();

  if (opcode == ControlPacket::quit) exit(0);

  return false;
}

bool ControlComponent::set_lib(ControlPacket *cp)
{
  text lib, loc;
  int length;
  long version;
  text *deps = NULL;

  cp->readParam(lib);
  cp->readParam(version);
  cp->readParam(loc);  
  cp->readParam(length);
  
  if (length > 0) {
    deps = new text[length];
    for(int i = 0; i < length; ++i) {
      cp->readParam(deps[i]);
    }
  }

  bool ret = (dynloader->registerLibrary(lib, version, loc)
	      && dynloader->registerDependencies(lib, deps, length));
  __DELETE_ARRAY(deps);

  return ret;
}

bool ControlComponent::get_lib(ControlPacket *cp)
{
  text lib;
  cp->readParam(lib);

  int version;
  text loc;
  text deps[16];
  int n = dynloader->getLibrary(lib, version, loc, 
				deps, sizeof(deps)/sizeof(text));
  response()->writeParam(version);
  response()->writeParam(loc);  
  response()->writeParam(n);

  for (int i = 0; i < n; ++i) response()->writeParam(deps[i]);
  return true;
}

bool ControlComponent::list_libs(ControlPacket *cp)
{
  text libs[512];
  int n = dynloader->listLibraries(libs, sizeof(libs)/sizeof(text));
  response()->writeParam(n);
  for (int i = 0; i < n; ++i) response()->writeParam(libs[i]);
  return true;
}

bool ControlComponent::set_binding(ControlPacket *cp)
{
  text compclass, libname, prefix;
  cp->readParam(compclass);
  cp->readParam(libname);
  cp->readParam(prefix);

  return dynloader->registerSymbol(compclass, libname, prefix);
}

bool ControlComponent::get_binding(ControlPacket *cp)
{
  text compclass;
  cp->readParam(compclass);

  text lib;
  bool ret = dynloader->getSymbol(compclass, lib);
  response()->writeParam(lib);

  return ret;
}

bool ControlComponent::list_symbols(ControlPacket *cp)
{
  text syms[512];
  int n = dynloader->listSymbols(syms, sizeof(syms)/sizeof(text));
  response()->writeParam(n);
  for (int i = 0; i < n; ++i) response()->writeParam(syms[i]);
  return true;
}

bool ControlComponent::start_stack(ControlPacket *cp)
{
  text id;
  bool threaded;
  cp->readParam(id);
  cp->readParam(threaded);
  stack_handle_t h = pandora->start(id, threaded, true);
  response()->writeParam(h);  
  return (h != NIL_STACK_HANDLE);
}

bool ControlComponent::stop_stack(ControlPacket *cp)
{
  stack_handle_t h;
  cp->readParam(h);
  return pandora->stop(h);
}

bool ControlComponent::suspend_stack(ControlPacket *cp)
{
  stack_handle_t h;
  cp->readParam(h);
  return pandora->suspend(h);
}

bool ControlComponent::resume_stack(ControlPacket *cp)
{
  stack_handle_t h;
  cp->readParam(h);
  return pandora->resume(h);
}

bool ControlComponent::get_stack(ControlPacket *cp)
{
  text id;
  cp->readParam(id);
  StackEntry *se = pandora->getStackEntry(id);
  if (se != NULL) {
    response()->writeParam(*se);
  } else {
    StackEntry empty_se;
    response()->writeParam(empty_se);
  }
  return (se != NULL);
}

bool ControlComponent::get_stack_name(ControlPacket *cp)
{
  stack_handle_t h;
  cp->readParam(h);

  text id;
  bool ret = pandora->getName(h, id);
  response()->writeParam(id);
  return ret;
}

bool ControlComponent::comp_query(ControlPacket *cp)
{
  stack_handle_t h;
  text comp_name;
  text option_name;

  cp->readParam(h);
  cp->readParam(comp_name);
  cp->readParam(option_name);

  MultiValue mv;

  bool ret = pandora->query(h, comp_name, option_name, &mv);

  response()->writeParam(mv);

  return ret;
}

bool ControlComponent::get_option(ControlPacket *cp)
{
  stack_handle_t h;
  text comp;
  text op;

  cp->readParam(h);
  cp->readParam(comp);
  cp->readParam(op);

  MultiValue mv;

  bool ret = pandora->getOption(h, comp, op, &mv);

  response()->writeParam(mv);

  return ret;
}

bool ControlComponent::get_option_def(ControlPacket *cp)
{
  text comp;
  text op;

  cp->readParam(comp);
  cp->readParam(op);

  MultiValue mv;

  bool ret = Component::getOptionDefault(comp, op, &mv);

  response()->writeParam(mv);

  return ret;
}

bool ControlComponent::set_option(ControlPacket *cp)
{
  text stk;
  text comp;
  text op;
  MultiValue val;
  
  cp->readParam(stk);
  cp->readParam(comp);
  cp->readParam(op);
  cp->readParam(val);

  return pandora->setOption(stk, comp, op, val);
}

bool ControlComponent::set_option_live(ControlPacket *cp)
{
  stack_handle_t h;
  text comp;
  text op;
  MultiValue val;
  
  cp->readParam(h);
  cp->readParam(comp);
  cp->readParam(op);
  cp->readParam(val);

  return pandora->setOption(h, comp, op, val);
}

bool ControlComponent::list_options(ControlPacket *cp)
{
  text comp;
  cp->readParam(comp);
  
  text ids[512];

  int n = Component::listOptions(comp, ids, sizeof(ids)/sizeof(text));

  response()->writeParam(n);
  for (int i = 0; i < n; ++i) response()->writeParam(ids[i]);
 
  return (n >= 0);
}

bool ControlComponent::set_stack(ControlPacket *cp)
{
  StackEntry se;
  cp->readParam(se);
  return pandora->setStack(se, true);
}

bool ControlComponent::list_defined(ControlPacket *cp)
{
  text ids[512];
  int n = pandora->listDefined(ids, sizeof(ids)/sizeof(text));
  response()->writeParam(n);
  for (int i = 0; i < n; ++i) response()->writeParam(ids[i]);
  return true;
}

bool ControlComponent::list_running(ControlPacket *)
{
  stack_handle_t ids[512];
  int n = pandora->listRunning(ids, sizeof(ids)/sizeof(text));
  response()->writeParam(n);
  for (int i = 0; i < n; ++i) response()->writeParam(ids[i]);
  return true;
}

bool ControlComponent::comp_alive(ControlPacket *)
{
#ifndef NDEBUG
  response()->writeParam(Component::alive);
#else
  response()->writeParam(-1);
#endif
  return true;
}

bool ControlComponent::packet_alive(ControlPacket *)
{
#ifndef NDEBUG
  response()->writeParam(Packet::alive);  
#else
  response()->writeParam(-1);
#endif
  return true;
}

bool ControlComponent::add_res(ControlPacket *cp)
{
  long id;
  text uri;
  int pri;
  cp->readParam(id);
  cp->readParam(uri);
  cp->readParam(pri);
  return grm->addResource(id, uri, pri);
}

bool ControlComponent::del_res(ControlPacket *cp)
{
  long id;
  text uri;
  cp->readParam(id);
  cp->readParam(uri);

  return grm->deleteResource(id, uri);
}

bool ControlComponent::set_res_pri(ControlPacket *cp)
{
  long id;
  text uri;
  int pri;
  cp->readParam(id);
  cp->readParam(uri);
  cp->readParam(pri);

  return grm->setPriority(id, uri, pri);
}

bool ControlComponent::list_res(ControlPacket *cp)
{
  long id;
  cp->readParam(id);

  resource_entry_t re[64];
  int n = grm->listResources(id, re, sizeof(re)/sizeof(resource_entry_t));
  response()->writeParam(n);
  for (int i = 0; i < n; ++i) response()->writeParam(re[i]);
  return true;  
}

bool ControlComponent::update_res(ControlPacket *cp)
{
  long id;
  cp->readParam(id);

  return grm->update(id);
}


bool ControlComponent::clean(ControlPacket *cp)
{
  stack_handle_t h;
  cp->readParam(h);
  if (h != NIL_STACK_HANDLE) {
    return pandora->clean(h);
  } else {
    pandora->dclean();
  }
  return true;
}


bool ControlComponent::quit(ControlPacket *cp)
{
  return true;
}


ControlPacket *ControlComponent::response(void)
{
  if (resp == NULL)
    resp = new ControlPacket(ControlPacket::reply);
  return resp;
}

void ControlComponent::sendResponse(void)
{
  if (resp != NULL) {
    reply(resp);
    resp = NULL;
  }
}

