/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef CACHE_MATCH2_COMPONENT_H
#define CACHE_MATCH2_COMPONENT_H

#include <libpandora/global.h>

extern "C" {
#include <sys/socket.h>
}

#include <map>
#include <pandora_components/cachetranspacket.h>
#include <pandora_components/httptranspacket.h>
#include <libpandora/component.h>
#include <libpandora/util.h>
#include <libpandora/memstats.h>

#define CACHE_MATCH_DEBUG 0

class CacheMatch2Component : public Component {
private:
  class chain_t {
  public:
    in_addr_t 		src, dst;

  private:
    timeval 		ts;
  
    const static int 	max_sibs = 8;
    int 		nsibs;
    HTTPTransPacket	*before;
    HTTPTransPacket	*after;
    HTTPTransPacket 	*siblings[max_sibs];
  
  public:
    chain_t(HTTPTransPacket *htp);
    virtual ~chain_t(void);

    void reset(void);
    bool check(HTTPTransPacket *);
    bool check(chain_t *);
  
    bool add(HTTPTransPacket *htp);
    bool add(chain_t *l);
    inline bool isComplete(void);

    CacheTransPacket *makeRecord(HTTPTransPacket *cache);
    inline HTTPTransPacket *getBefore(void) 	{ return before; }
    
  private:
    bool store(HTTPTransPacket *htp);
  };

  class CacheMatcher {
  private:
    typedef multimap<in_addr_t, chain_t *>  	map_t;
    typedef pair<in_addr_t, chain_t *>  	pair_t;
    typedef map_t::iterator 			map_ptr_t;
    map_t			  		srcs, dsts;

  public:
    CacheMatcher(void) { }
    virtual ~CacheMatcher(void) { cleanup(); }

    chain_t *add(HTTPTransPacket *);
    chain_t *flush(void);
    void cleanup(void);

    HTTPTransPacket *getBefore(void);

  private:
    chain_t *getChain(HTTPTransPacket *);
    chain_t *findChain(chain_t *);
  
    inline void store(chain_t *);
    inline void storeSrc(chain_t *);
    inline void storeDst(chain_t *);
    inline void remove(chain_t *);
    inline void removeSrc(chain_t *);
    inline void removeDst(chain_t *);
  };


private:
  HTTPTransPacket	*cache;
  CacheMatcher		matcher;
  int 			cto;
  bool 			check;

public:
  component_init(CacheMatch2Component, 1);
  CacheMatch2Component(void);
  virtual ~CacheMatch2Component(void) { }

  virtual bool add(Packet*);
  virtual void cleanup(void);
  virtual bool prepare(void);

private:
  void flush(void);
  void setCacheRequest(HTTPTransPacket *);
  void flushCacheRequest(void);
  void dispatch(chain_t *);

};


bool CacheMatch2Component::chain_t::isComplete(void)  
{ 
  return ((before != NULL) && (after != NULL)); 
}

void CacheMatch2Component::CacheMatcher::store(chain_t *c)
{
  storeSrc(c);
  storeDst(c);
}

void CacheMatch2Component::CacheMatcher::storeSrc(chain_t *c)
{
  srcs.insert(pair_t(c->src, c));
}

void CacheMatch2Component::CacheMatcher::storeDst(chain_t *c)
{
  dsts.insert(pair_t(c->dst, c));
}

void CacheMatch2Component::CacheMatcher::remove(chain_t *c)
{
  removeDst(c);
  removeSrc(c);
}

void CacheMatch2Component::CacheMatcher::removeSrc(chain_t *c)
{
  multimap_erase(srcs, c->src, c);
}

void CacheMatch2Component::CacheMatcher::removeDst(chain_t *c)
{
  multimap_erase(dsts, c->dst, c);
}

#endif /* CACHE_MATCH2_COMPONENT_H */
