/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include "cachematch2component.h"
#include <pandora_components/httptranspacket.h>
#include <pandora_components/httppacket.h>
#include <pandora_components/tcppacket.h>
#include <pandora_components/ippacket.h>
#include <libpandora/timeval.h>

component_export(CacheMatch2Component, HTTPTransPacket+, HTTPPacket|CacheTransPacket);

CacheMatch2Component::CacheMatch2Component(void) 
  : cache(NULL), cto(0), check(false)
{
  registerOption("cache", &cto);
  registerOption("check", &check);
}

bool CacheMatch2Component::add(Packet* obj)
{
  HTTPTransPacket *msg = static_cast<HTTPTransPacket *>(obj);

  HTTPTransPacket::httpt_pos_t pos = msg->pos;

  if (pos == HTTPTransPacket::alone) {
    push(new CacheTransPacket(msg));
    return false;
  }

  locatePacket(HTTPPacket,	httpp,	msg);

  if (httpp->coll > 0) {
    //pandora_debug("[reemitted request]");
    if (cache == NULL) {
      setCacheRequest(msg);
      flush();
    } else {
      cleanPacket(msg);
    }
    return false;
  }

  if ((cache == NULL) && (pos == HTTPTransPacket::after)) {
    setCacheRequest(msg);
  }      

  chain_t *chain = matcher.add(msg);
  if (chain != NULL) dispatch(chain);

  return false;
}


bool CacheMatch2Component::prepare(void)
{
  if (!check 
      || cache != NULL)
    return true;

  HTTPTransPacket *htp = matcher.getBefore();
  //pandora_debug("[get before: " << htp << "]");
  locatePacket(HTTPPacket, cachebuf, htp);

  if (cachebuf == NULL) return true;

  HTTPPacket *tmp = new HTTPPacket(*cachebuf);
  locatePacket(IPPacket, ipp, tmp);
  if (ipp == NULL) return true;
  (ipp->dst).s_addr = 0;

  push(tmp, 1);
  if (cto > 0) timeout = cto;
  check = false;

  return false;
}

void CacheMatch2Component::cleanup(void)
{
  locatePacket(HTTPPacket, httpp, cache);
  if (httpp != NULL) httpp->coll = 1;

  flush();
  matcher.cleanup();
  flushCacheRequest();
}

void CacheMatch2Component::flush(void)
{
#if CACHE_MATCH_DEBUG > 1
  pandora_debug("[flushing]");
#endif
  chain_t *chain = NULL;
  while ((chain = matcher.flush()) != NULL) {
    dispatch(chain);
  }
}

void CacheMatch2Component::dispatch(chain_t *chain)
{
#if CACHE_MATCH_DEBUG
  pandora_debug("[dispatching chain " << chain << "]");
#endif
  if (chain == NULL) return;
  CacheTransPacket *ctp = chain->makeRecord(cache);
  if (ctp != NULL) push(ctp);

  __DELETE(chain);
} 

void CacheMatch2Component::setCacheRequest(HTTPTransPacket *httptp)
{
  if (cache == NULL) {
    cache = httptp;
    cache->refer();
#if 0
    locatePacket(HTTPPacket, httpp, cache);
    pandora_debug(lastTime
		  << " [caching request for:  " 
		  << httpp->url << "]");
#endif
  }
}

void CacheMatch2Component::flushCacheRequest(void)
{
  if (cache != NULL) {
#if 0
    locatePacket(HTTPPacket, httpp, cache);
    pandora_debug(lastTime
		  << " [flushing request for: " 
		  << httpp->url << "]");
#endif
    cache->release();
    cleanPacket(cache);
  }
}

CacheTransPacket *
CacheMatch2Component::chain_t::makeRecord(HTTPTransPacket *cache)
{
  if (after == NULL) after = cache;

  CacheTransPacket *ctp = 
    ((before != NULL) 
     ? ((after != NULL) 
	? new CacheTransPacket(before, after)
	: new CacheTransPacket(before))
     : ((after != NULL) 
	? new CacheTransPacket(after)
	: ((nsibs > 0) 
	   ? new CacheTransPacket(siblings[--nsibs])
	   : NULL)));
  
  before = after = NULL;
  
  if (ctp != NULL) {
    while (nsibs > 0) {
      --nsibs;
      ctp->sibling_update(siblings[nsibs]);
    }      
  }

  return ctp;
}

CacheMatch2Component::chain_t::chain_t(HTTPTransPacket *htp) 
  : src(0), dst(0), nsibs(0), before(NULL), after(NULL)
{ 
  ts.tv_sec = 0;

  locatePacket(IPPacket, ipp, htp);
  if (ipp == NULL) return;
  src = (ipp->src).s_addr;
  dst = (ipp->dst).s_addr;
  ts =  htp->timeStamp;
  if (!store(htp)) {
    pandora_warning("internal error");
    cleanPacket(htp);
  }
}

CacheMatch2Component::chain_t::~chain_t(void)
{
  if (before != NULL) {
    pandora_warning("orphaned requests [before: "<< before << "] (" 
		    << this << ")");
    cleanPacket(before);
  }
  
  if (after != NULL) {
    pandora_warning("orphaned requests [after: "<< after << "] (" 
		    << this << ")");
    cleanPacket(after);
  }

  for (int i = 0; i < nsibs; ++i) {
    pandora_warning("orphaned requests [sibling #" << i
		    << ": "<< siblings[i] << "] (" 
		    << this << ")");
    cleanPacket(siblings[i]);
  }
}

void CacheMatch2Component::chain_t::reset(void) 
{
  src = 	0; 
  dst = 	0; 
  ts.tv_sec = 	0;
  nsibs = 	0;
  before = 	NULL;
  after = 	NULL;
}

bool CacheMatch2Component::chain_t::check(HTTPTransPacket *htp)
{
  locatePacket(IPPacket, ipp, htp);

  if (ipp == NULL)  return false;
  if (src != (ipp->dst).s_addr) return false;
  if (ts < htp->timeStamp) return false;

  switch (htp->pos) {
  case HTTPTransPacket::before:  return (before == NULL);
  case HTTPTransPacket::after:	 return (after == NULL);
  case HTTPTransPacket::sibling: return (nsibs < (max_sibs-1));
  case HTTPTransPacket::alone: 		
  case HTTPTransPacket::matched: pandora_warning("[wrong type]"); return false;
  }

  return true;
}

bool CacheMatch2Component::chain_t::check(chain_t *l)
{
  if (src != l->dst)				return false;
  if (ts < l->ts) 				return false;

  if ((before != NULL) && (l->before != NULL))	return false;
  if (l->after != NULL)				return false;
  if ((nsibs + l->nsibs) > max_sibs)		return false;

  return true;
}

bool CacheMatch2Component::chain_t::add(HTTPTransPacket *htp)
{
  locatePacket(IPPacket, ipp, htp);
  if (ipp == NULL) return false;

  src = (ipp->src).s_addr;
  ts = htp->timeStamp;

  if (!store(htp)) {
    pandora_warning("internal error");
    cleanPacket(htp);
    return false;
  }

  return true;
}

bool CacheMatch2Component::chain_t::add(chain_t *l)
{
  if (l == NULL) return false;
  
  src = 	l->src;
  ts = 		l->ts;
  before = 	l->before;

  if (l->after != NULL) {
    pandora_warning("invalid merge");
    cleanPacket(l->after);
  }

  memcpy((char *)(siblings + nsibs), (char *)(l->siblings),
	 (l->nsibs) * sizeof(HTTPTransPacket *));
  
  nsibs += l->nsibs;

  l->reset();
  return true;
}


bool CacheMatch2Component::chain_t::store(HTTPTransPacket *htp)
{
  if (htp == NULL) 			return false;

#if CACHE_MATCH_DEBUG > 1
  pandora_debug("[storing packet " << htp << " into chain " << this << "]");
#endif
  switch (htp->pos) {
  case HTTPTransPacket::before: 	
    before = htp;
    break;

  case HTTPTransPacket::after:		
    after = htp;
    break;

  case HTTPTransPacket::sibling:        
    siblings[nsibs] = htp;
    ++nsibs;
    break;

  case HTTPTransPacket::alone: 
  case HTTPTransPacket::matched:
    pandora_warning("invalid packet: cannot add");
    return false;
  }

  return true;
}

void CacheMatch2Component::CacheMatcher::cleanup(void)
{
#if CACHE_MATCH_DEBUG > 1
  pandora_debug("[cache matcher cleanup]");
#endif

  for (map_ptr_t ptr = srcs.begin(); ptr != srcs.end(); ++ptr) {
    chain_t *c = ptr->second;
    pandora_warning("remaining chain from " << c->src << " to " << c->dst);
    __DELETE(c);
  }

  srcs.clear();
  dsts.clear();
}

CacheMatch2Component::chain_t *
CacheMatch2Component::CacheMatcher::add(HTTPTransPacket *htp)
{
  if (htp == NULL) return NULL;

  chain_t *chain = getChain(htp);

  if (chain == NULL) {
    chain = new chain_t(htp);
    storeDst(chain);
  } else {
    if (!chain->add(htp)) {
      pandora_warning("internal error");
      cleanPacket(htp);
    }
  }

  if (chain->isComplete()) {
    removeDst(chain);
    return chain;
  }

  chain_t *chain2 = findChain(chain);
  if (chain2 != NULL) {
    if (!chain->add(chain2)) {
      pandora_warning("internal error");
    }
    __DELETE(chain2);
  }

  if (chain->isComplete()) {
    removeDst(chain);
    return chain;
  }

  storeSrc(chain);
  return NULL;
}

CacheMatch2Component::chain_t *
CacheMatch2Component::CacheMatcher::flush(void)
{
  chain_t *c = NULL;
  map_ptr_t ptr = srcs.begin();

  if (ptr != srcs.end()) c = ptr->second;
  if (c != NULL) remove(c);
  return c;
}

HTTPTransPacket *CacheMatch2Component::CacheMatcher::getBefore(void)
{
  HTTPTransPacket *htp = NULL;
  map_ptr_t ptr = srcs.begin();

  if (ptr != srcs.end()) htp = ptr->second->getBefore();
  return htp;
}

CacheMatch2Component::chain_t *
CacheMatch2Component::CacheMatcher::getChain(HTTPTransPacket *htp)
{
  locatePacket(IPPacket, ipp, htp);
  if (ipp == NULL) return NULL;
  
  if (htp->pos == HTTPTransPacket::after) return NULL;

#if CACHE_MATCH_DEBUG
  pandora_debug("[matcher: " << this 
		<< " searching chain for src: " << (ipp->dst).s_addr << "]");

#endif
  pair<map_ptr_t, map_ptr_t> p = srcs.equal_range((ipp->dst).s_addr);
  map_ptr_t first = p.first;
  map_ptr_t last = p.second;

  for (map_ptr_t ptr = first; ptr != last; ++ptr) {
    chain_t *chain = ptr->second;
#if CACHE_MATCH_DEBUG
    pandora_info("[matcher: " << this 
		 << " checking chain" << chain << "]");
#endif
    if (chain->check(htp)) {
      removeSrc(chain);
      return chain;
    }
  }

  return NULL;
}

CacheMatch2Component::chain_t *
CacheMatch2Component::CacheMatcher::findChain(chain_t *c)
{
  pair<map_ptr_t, map_ptr_t> p = dsts.equal_range(c->src);
  map_ptr_t first = p.first;
  map_ptr_t last = p.second;

  for (map_ptr_t ptr = first; ptr != last; ++ptr) {
    chain_t *chain = ptr->second;
    if (c->check(chain)) {
      remove(chain);
      return chain;
    }
  }

  return NULL;
}

