/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <ctype.h>
#include <netinet/in.h>
#include <errno.h>
#include <stdlib.h>
#include <libpandora/conf/string.h>
#include <libpandora/conf/time.h>
#include <libpandora/conf/unistd.h>
#include <libpandora/conf/snprintf.h>
#include <libpandora/conf/usleep.h>
   }

#include <iomanip>

#include <libpandora/util.h>
#include <libpandora/error.h>

/// Convert a token value to a string; use "fmt" if not found.
char *tok2str(register const struct tok *lp, register const char *fmt,
	register int v)
{
  static char buf[128];

  while (lp->s != NULL) {
    if (lp->v == v)
      return (lp->s);
    ++lp;
  }
  if (fmt == NULL)
    fmt = "#%d";
  (void)sprintf(buf, fmt, v);
  return (buf);
}


/// A faster replacement for inet_ntoa().
char *intoa(u_int32_t addr)
{

  register char *cp;
  register u_int byte;
  register int n;
  static char buf[sizeof(".xxx.xxx.xxx.xxx")];

  addr=ntohl(addr);
  cp = &buf[sizeof buf];
  *--cp = '\0';

  n = 4;
  do {
    byte = addr & 0xff;
    *--cp = byte % 10 + '0';
    byte /= 10;
    if (byte > 0) {
      *--cp = byte % 10 + '0';
      byte /= 10;
      if (byte > 0)
	*--cp = byte + '0';
    }
    *--cp = '.';
    addr >>= 8;
  } while (--n > 0);
  return cp + 1;
}


char *skip_ws(register char *str)
{
  for(/*empty*/; (str != NULL) && isspace(*str); ++str) /*empty*/;
  return str;
}

ostream& operator<<(ostream& f, const struct in_addr &a)
{
  register u_int32_t addr = a.s_addr;

  register char *cp;
  register u_int byte;
  register int n;
  static char buf[sizeof(".xxx.xxx.xxx.xxx")];

  addr=ntohl(addr);
  cp = &buf[sizeof buf];
  *--cp = '\0';

  n = 4;
  do {
    byte = addr & 0xff;
    *--cp = byte % 10 + '0';
    byte /= 10;
    if (byte > 0) {
      *--cp = byte % 10 + '0';
      byte /= 10;
      if (byte > 0)
	*--cp = byte + '0';
    }
    *--cp = '.';
    addr >>= 8;
  } while (--n > 0);

  return f << (cp + 1);
}

u_short check_sum(u_char *addr, size_t count) {
  /* Compute Internet Checksum for "count" bytes
   *         beginning at location "addr".
   */
  register int32_t sum = 0;

  while( count > 1 )  {
    /*  This is the inner loop */
    sum += *(unsigned short *)addr++;
    count -= 2;
  }

  /*  Add left-over byte, if any */
  if( count > 0 )
    sum += *(unsigned char *)addr;

  /*  Fold 32-bit sum to 16 bits */
  while (sum>>16)
    sum = (sum & 0xffff) + (sum >> 16);

  return (u_short) ~sum;
}

void sleep_until(const struct timeval &target)
{
  struct timeval now;

  if (gettimeofday(&now, NULL) < 0) {
    pandora_pwarning("gettimeofday");
    return;
  }

  long sleeping = (target.tv_sec - now.tv_sec) * 1000000
      + (target.tv_usec - now.tv_usec);    
  if (sleeping > 0) usleep(sleeping);
}
