/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <stdlib.h>
#include <stdio.h>
#include <libpandora/conf/string.h>
#include <ctype.h>
#include <libpandora/conf/snprintf.h>
#include <libpandora/conf/time.h>
	   }


#include <libpandora/url.h>
#include <libpandora/http_date.h>
#include <libpandora/netutil.h>
#include <libpandora/error.h>

#ifndef TOASCII
#define TOASCII(c) (c)
#define FROMASCII(c) (c)
#endif
#define TOLOWER(c) tolower((int) (c)) 

#define MASK	0x8

#define ACCEPTABLE(a)	( a>=32 && a<128 && ((isAcceptable[a-32]) & MASK))

static char isAcceptable[96] =
{/* 0x0 0x1 0x2 0x3 0x4 0x5 0x6 0x7 0x8 0x9 0xA 0xB 0xC 0xD 0xE 0xF */
    0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xF,0xE,0x0,0xF,0xF,0xC, 
/* 2x  !"#$%&'()*+,-./   */
    0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0x8,0x0,0x0,0x0,0x0,0x0, 
/* 3x 0123456789:;<=>?   */
    0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,
/* 4x @ABCDEFGHIJKLMNO   */
    0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0x0,0x0,0x0,0x0,0xF,
/* 5X PQRSTUVWXYZ[\]^_   */
    0x0,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,
/* 6x `abcdefghijklmno   */
    0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0xF,0x0,0x0,0x0,0x0,0x0 
/* 7x pqrstuvwxyz{\}~DEL */
};
static char *hexa = "0123456789ABCDEF";

#define HEX_ESCAPE '%'

void URL::escape(void)
{
  if (absolute.isNull()) return;
  const char *str = absolute.data();
  const char *p;
  char *q;
  char *result;
  int unacceptable = 0;

  for(p = str; *p; p++)
    if (!ACCEPTABLE((unsigned char)TOASCII(*p)))
      unacceptable++;
  result = (char *) xmalloc(p-str + unacceptable + unacceptable + 1);
  for(q = result, p = str; *p; p++) {
    unsigned char a = TOASCII(*p);
    if (!ACCEPTABLE(a)) {
      *q++ = HEX_ESCAPE;	/* Means hex commming */
      *q++ = hexa[a >> 4];
      *q++ = hexa[a & 15];
    }
    else *q++ = *p;
  }
  *q++ = 0;			/* Terminate */
  absolute.take(result);
}

char URL::ascii2char(char c)
{
  return  c >= '0' && c <= '9' ?  c - '0' 
    : c >= 'A' && c <= 'F'? c - 'A' + 10
    : c - 'a' + 10;	/* accept small letters just in case */
}

void URL::unescape(void)
{
  if (absolute.isNull()) return;
  char *p, *q;
  p = q = absolute.data();

  while(*p) {
    if (*p == HEX_ESCAPE) {
      p++;
      if (*p) *q = ascii2char(*p++) * 16;
      /* Suggestion from Markku Savela */
      if (*p) *q = FROMASCII(*q + ascii2char(*p)), ++p;
      q++;
    } else {
      *q++ = *p++; 
    }
  }
    
  *q++ = 0;
  absolute.update();
}

void URL::scan(const char *url)
{
  if (url == NULL) return;

  char *name = xstrdup(url);
  char *p;
  char *after_access = name;
  char *host_str, *port_str = NULL;
 
  for(p = name; *p; p++) {
 
    /*
    ** Look for any whitespace. This is very bad for pipelining as it
    ** makes the request invalid
    */
    if (isspace((int) *p)) {
      char *orig = p, *dest = p+1;
      while ((*orig++ = *dest++));
      p = p-1;
    }
    if (*p == '/' || *p == '#' || *p == '?')
      break;
    if (*p == ':') {
      *p = 0;
      after_access = p+1;
    }
  }
 
  p = after_access;
  if (*p == '/'){
    if (p[1] == '/') {
      host_str = p+2;           /* host has been specified      */
 
      p = strchr(host_str, '/');/* look for end of host name if any */
      if (p) {
        absolute.init(p);       /* Root has been found */
        *p=0;                   /* Terminate host */
      } else {
	absolute.init("/");
      }
 
      p = strchr(host_str,':'); /* look for end of host name with port if any */
      if (p) {
        port_str = p+1;
        *p = 0;
      }
 
      if (host_str != NULL) host.init(host_str);
      if (port_str != NULL) port = atoi(port_str);
      else port = 80;
    } else {
      absolute.init(p);         /* Root found but no host */
    }
  }
  __FREE(name);
}

void URL::getURL(bool abs, char *buf, int maxlen)
{
  if (abs) {
    snprintf(buf, maxlen, "http://%s:%d%s", 
	     host.data(), port, absolute.data());
  } else {
    snprintf(buf, maxlen, "%s", absolute.data());
  }
}

bool URL::makeRequest(char *buf, int len,
		      const char *method, bool useProxy, 
		      time_t ims, const char *header)
{
  if (isNull()) return false;

  char ims_str[512], final[1024];
  const char *hdr = (header != NULL) ? header : "";
  *ims_str = '\0';

  if (ims > 0) {
    char date[64];
    http_time_to_date(date, sizeof(date), ims);
    snprintf(ims_str, sizeof(ims_str), "If-Modified-Since: %s\r\n", date);
  }

  getURL(useProxy, final, sizeof(final));
  
  snprintf(buf, len,
	   "%s %s HTTP/1.0\r\n"
	   "User-Agent: Pandora\r\n"
	   "Host: %s\r\n"
	   "Accept: */*\r\n"
	   "%s"
	   "%s"
	   "X-Pandora: 1\r\n"
	   "\r\n",
	   method, final, host.data(), ims_str, hdr);  

  return true;
}

in_addr_t URL::getHostAddr(void)
{
  return get_addr(host.data());
}

int URL::fetchString(char **str, in_addr_t proxy_addr, u_short proxy_port)
{
  FILE *f = get(proxy_addr, proxy_port);
  if (f == NULL) return 0;

  int bytes = 0, n = 0;

  if (str == NULL) {
    char buf[BUFSIZ];
    while((n = fread(buf, sizeof(char), sizeof(buf), f)) > 0) {
      bytes += n;
    }
  } else {
    int size = 1024 * 64;
    (*str) = (char *)xmalloc(size*sizeof(char));

    while((n = fread((*str) + bytes, sizeof(char), size - bytes, f)) > 0) {
      bytes += n;
      if ((size - bytes) < (size/5)) {
	size *= 2;
	(*str) = (char *)xrealloc((*str), size*sizeof(char));
      }
    }
    (*str) = (char *)xrealloc((*str), bytes*sizeof(char));
  }

  fclose(f);

  return bytes;
}

int URL::fetchFile(const char *dir, char *name, int len,
		   in_addr_t proxy_addr, u_short proxy_port)
{
  if (name == NULL) return -1;
  
  FILE *f = get(proxy_addr, proxy_port);
  if (f == NULL) return 0;

  char *base = NULL;
  base = strrchr(absolute.data(), '/');
  if (base == NULL) {
    fclose(f);
    return -1;
  }

  base++;
  if (*base == '\0') base = "index.html";

  snprintf(name, len, "%s/%s", 
	   ((dir != NULL) ? dir : "."), base);


  FILE *f2 = fopen(name, "w");
  if (f2 == NULL) {
    fclose(f);
    return -1;
  }

  int bytes = 0, n = 0;
  char respbuf[BUFSIZ];

  while((n = fread(respbuf, sizeof(char), sizeof(respbuf), f)) > 0) {
    respbuf[n] = '\0';
    bytes += fwrite(respbuf, sizeof(char), n, f2);
  }

  fclose(f2);
  fclose(f);

  return bytes;
}

bool URL::ifModifiedSince(time_t t, in_addr_t proxy_addr, u_short proxy_port)
{
  FILE *f = NULL;
  int fd = sendRequest("HEAD", proxy_addr, proxy_port, t);
  if (fd < 0) return false;
  fdopen_socket(fd, &f);
  if (f == NULL) return false;
  bool res = (fetchHeader(f) == 200);
  fclose(f);
  return res;
}

FILE *URL::get(in_addr_t proxy_addr, u_short proxy_port)
{
  FILE *f = NULL;
  int fd = sendRequest("GET", proxy_addr, proxy_port);
  fdopen_socket(fd, &f);
  if (f == NULL) return NULL;
  if (fetchHeader(f) != 200) {
    fclose(f);
    return NULL;
  }

  return f;
}

int URL::sendRequest(const char *method, 
		       in_addr_t proxy_addr, u_short proxy_port,  
		       time_t ims, const char *hdr)
{
  char reqbuf[2048];
  bool useProxy = (proxy_addr != 0);

  escape();
  if (!makeRequest(reqbuf, sizeof(reqbuf), method, useProxy, ims, hdr))
    return -1;

  int fd = openclient((useProxy ? proxy_port : getHostPort()),
		      (useProxy ? proxy_addr : getHostAddr()),
		      true, true);
  if (fd < 0) return -1;

  if (net_writeblock(fd, reqbuf, strlen(reqbuf)) <= 0) {
    pandora_pwarning("net_writeblock");
    close(fd);
    return -1;
  }

  return fd;
}


int URL::fetchHeader(FILE *f)
{

  if (f == NULL) return -1;

  bool found = false;
  char buf[BUFSIZ];
  int  code = -1;

  if (fgets(buf, sizeof(buf), f) == NULL) 	return -1;
  if (strncmp(buf, "HTTP/", 5) != 0) 		return -1;
  code = atoi(buf+9);

  while(fgets(buf, sizeof(buf), f) != NULL) {
    if (*buf == '\r' || *buf == '\n') {
      found = true;
      break;
    }
  }
  if (!found) return -1;

  return code;
}
