/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef TIMEVAL_H
#define TIMEVAL_H

#include <libpandora/global.h>

#include <libpandora/conf/time.h>
#include <iostream>
#include <iomanip>

inline int compareTimeStamp(const timeval &, const timeval &);
inline long diffTimeStamp(const timeval &, const timeval &);
inline void incTimeStamp(struct timeval &tv, long ms);

inline ostream &operator<< (ostream &, const timeval &);
inline bool operator< (const timeval&, const timeval &);
inline bool operator> (const timeval&, const timeval &);
inline bool operator== (const timeval&, const timeval &);
inline bool operator!= (const timeval&, const timeval &);
inline timeval operator+ (const timeval&, const timeval &);
inline timeval operator+ (const timeval&, long);
inline timeval operator- (const timeval&, const timeval &);
inline timeval operator- (const timeval&, long);



int compareTimeStamp(const timeval &ts1, const timeval &ts2)
{
  if (ts1.tv_sec > ts2.tv_sec)   return  1;
  if (ts1.tv_sec < ts2.tv_sec)   return -1;
  if (ts1.tv_usec > ts2.tv_usec) return  1;
  if (ts1.tv_usec < ts2.tv_usec) return -1;
  return 0;
}

long diffTimeStamp(const timeval &ts1, const timeval &ts2)
{
  return ((ts1.tv_sec - ts2.tv_sec)*1000
	  +(ts1.tv_usec - ts2.tv_usec)/1000);
}

void incTimeStamp(timeval &ts, long ms)
{
  long n = ts.tv_usec + (ms * 1000);
  ts.tv_usec = n % 1000000;
  ts.tv_sec += n / 1000000;
}

ostream &operator<< (ostream &f, const timeval &ts)
{
  return f << (int)ts.tv_sec << "." << setw(3) << setfill('0')
	   << (int)ts.tv_usec/1000 ;
}

bool operator< (const timeval &ts1, const timeval &ts2)
{
  return ((ts1.tv_sec < ts2.tv_sec) 
	  || ((ts1.tv_sec == ts2.tv_sec) && (ts1.tv_usec < ts2.tv_usec)));
}

bool operator> (const timeval &ts1, const timeval &ts2)
{
  return ((ts1.tv_sec > ts2.tv_sec) 
	  || ((ts1.tv_sec == ts2.tv_sec) && (ts1.tv_usec > ts2.tv_usec)));
}

bool operator== (const timeval &ts1, const timeval &ts2)
{
  return ((ts1.tv_sec == ts2.tv_sec) && (ts1.tv_usec == ts2.tv_usec));
}

bool operator!= (const timeval &ts1, const timeval &ts2)
{
  return !operator== (ts1, ts2);
}

timeval operator+ (const timeval &ts1, const timeval &ts2)
{
  timeval ts;
  ts.tv_sec = ts1.tv_sec + ts2.tv_sec;
  ts.tv_usec = ts1.tv_usec + ts2.tv_usec;
  if (ts.tv_usec >= 1000000) {
      ++(ts.tv_sec);
      ts.tv_usec -= 1000000;
  }
  return ts;
}

timeval operator+ (const timeval &ts1, long ms)
{
  long n = ts1.tv_usec + (ms * 1000);

  timeval ts;
  ts.tv_usec = n % 1000000;
  ts.tv_sec  = ts1.tv_sec + (n / 1000000);

  return ts;
}

timeval operator- (const timeval &ts1, const timeval &ts2)
{
  timeval ts;
  ts.tv_sec = ts1.tv_sec - ts2.tv_sec;
  ts.tv_usec = ts1.tv_usec - ts2.tv_usec;
  if (ts.tv_usec < 0) {
      --(ts.tv_sec);
      ts.tv_usec += 1000000;
  }
  return ts;
}

timeval operator- (const timeval &ts1, long ms)
{
  long n = ts1.tv_usec - (ms * 1000);

  timeval ts;
  ts.tv_usec = n % 1000000;
  ts.tv_sec  = ts1.tv_sec - (n / 1000000);

  return ts;
}

#endif /* TIMEVAL_H */
