/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <libpandora/timeout_scheduler.h>
#include <libpandora/component.h>

time_t timeout_scheduler::schedule(time_t t, const sched_task_t &st)
{
  if (t <= localTime) {
    //pandora_warning("scheduled in the past!");
    t = localTime + 1;
  }
  if (t > (localTime + size)) {
    //pandora_warning("scheduled too far away in the future!");
    t = localTime + (size - 1);
  }

  int nindex = (index + (t - localTime)) % size;

#if TDEBUG
  pandora_debug("task is STORED with index #" << nindex);
#endif
  slot_t *slot = &(tab[nindex]);
  slot->add(st);
#if DEBUG_SCHED
  ++inserted; ++remaining;
#endif

  return t;
}

void timeout_scheduler::remove(time_t t, const sched_task_t &st)
{
  if ( t < localTime ) {
    pandora_error("task should have been removed: " 
		  << t << " < " << localTime);
  }
  pandora_assert( t <= localTime + size );
  
  int nindex = (index + (t - localTime)) % size;

#if TDEBUG
  pandora_debug("task is REMOVED from index #" << nindex);
#endif
  slot_t *slot = &(tab[nindex]);
  pandora_assert(slot != NULL);
  slot->remove(st);
  st.removed(localTime);
#if DEBUG_SCHED
  --remaining; ++removed;
#endif
}

void timeout_scheduler::expire(time_t t)
{
  pandora_assert(localTime != 0);
  pandora_assert(index >= 0);

  for (/* empty */; localTime < t; ++localTime) {
    slot_t *slot = &(tab[index]);
    
    sched_task_t st;
    elementsDo(*slot, st) {
      st.expired(localTime);
#if DEBUG_SCHED
      --remaining; ++expired;
#endif
    }
    slot->clear();
    index = (index + 1) % size;
  }
}

void timeout_scheduler::cleanup(void)
{
  for (int i = 0; i < size; ++i) {
    slot_t *slot = &(tab[index]);      
    sched_task_t st;
    
    elementsDo(*slot, st) {
      slot->remove(st);
#if DEBUG_SCHED
      --remaining; ++expired;
#endif
    }
    slot->clear();
  }
  
  localTime = 0;
  index = 0;
}
