/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef TEXT_H
#define TEXT_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
}

#include <iostream>
#include <libpandora/serialize.h>
#include <libpandora/util.h>

class text {
private:
  char *_data;
  int _len;

public:
  text(void): _data(NULL), _len(0) { }
  inline text(int);
  inline text(const char *) ;
  inline text(const char *, int) ;
  inline text(const text &x) ;
  inline text &operator=(const text &x);
  ~text(void) { __FREE(_data); _len = 0; }

  inline char *data(void) 		{ return _data; }
  inline const char *data(void) const 	{ return (const char *)_data; }
  inline int  length(void) const 	{ return _len; }

  inline void init(int);
  inline void init(const char *str);
  inline void init(const char *str, int l);
  inline void init(const text &x);

  inline void take(char *str);
  inline void take(char *str, int l);
  inline void take(text &x);

  inline void update(void) { _len = ((_data != NULL) ? strlen(_data) : 0); }
  inline void reset(void)  { _data = NULL; _len = 0; }

  inline operator long() const;
  inline bool operator ==(const text &) const;
  inline bool operator !=(const text &x) { return !operator ==(x); }

  bool isNull(void) const { return (_data == NULL); }

  friend void serialize(char *, size_t &, const size_t, const text *);
  friend void unserialize(const char *, size_t &, text *);
};

text::text(int siz)
{
  if (siz > 0) {
    _len = siz;
    _data = (char *) xmalloc((_len+1)*sizeof(char));
    _data[0] = '\0';
  } else {
    _len = 0; _data = NULL;
  }
}

text::text(const char *str) 
{
  if (str != NULL) {
    _len = strlen(str);
    _data = (char *) xmalloc((_len+1)*sizeof(char));
    memcpy(_data, str, _len*sizeof(char));
    _data[_len]='\0';
  } else {
    _len = 0; _data = NULL;
  }
}

text::text(const char *str, int l) 
{
  if ((str != NULL) & (l > 0)) {
    _len = l;
    _data = (char *) xmalloc((_len+1)*sizeof(char));
    memcpy(_data, str, _len*sizeof(char));
    _data[_len]='\0';
  } else {
    _len = 0; _data = NULL;
  }
}

text::text(const text &x) : _len(x._len) 
{
  if (x.isNull()) {
    _data = NULL;
  } else {
    _data = (char *) xmalloc((_len+1) * sizeof(char));
    memcpy(_data, x._data, _len * sizeof(char));
    _data[_len]='\0';
  } 
}

text &text::operator=(const text &x) 
{
  init(x);
  return *this;
}

void text::init(int siz)
{
  __FREE(_data);
  if (siz <= 0) return;
  _len = siz;
  _data = (char *) xmalloc((_len+1)*sizeof(char));
  _data[0] = '\0';
}

void text::init(const char *str) 
{
  if (_data == str) {
    update();
    return;
  }
  __FREE(_data);
  if (str == NULL) return;
  _len = strlen(str);
  _data = (char *) xmalloc((_len+1)*sizeof(char));
  memcpy(_data, str, _len*sizeof(char));
  _data[_len]='\0';
}

void text::init(const char *str, int l) 
{
  if (_data == str) {
    _len = l;
    return;
  }
  __FREE(_data);
  if ((str == NULL) || (l < 0)) return;
  _len = l;
  _data = (char *) xmalloc((_len+1)*sizeof(char));
  memcpy(_data, str, _len*sizeof(char));
  _data[_len]='\0';
}

void text::init(const text &x) 
{
  if (&x == this) return;
  __FREE(_data);
  if (x.isNull()) return;
  _len = x._len;
  _data = (char *) xmalloc((_len+1)*sizeof(char));
  memcpy(_data, x._data, _len*sizeof(char));
  _data[_len]='\0';
}

void text::take(char *str) 
{
  if (_data == str) return;
  __FREE(_data);
  if (str == NULL) return;
  _len = strlen(str);
  _data = str;
  str = NULL;
}

void text::take(char *str, int l) 
{
  if (_data == str) {
    _len = l;
    return;
  }
  __FREE(_data);
  if ((str == NULL) || (l < 0)) return;
  _len = l;
  _data = str;
  str = NULL;
}

void text::take(text &x) 
{
  if (&x == this) return;
  __FREE(_data);
  if (x.isNull()) return;
  _len = x._len;
  _data = x._data;
  x.reset();
}

text::operator long() const
{
  return string_hash(_data);
}

bool text::operator ==(const text &x) const 
{ 
  return (!_data && !x._data) 
    || (_data && x._data 
	&& (_len == x._len) && !strncmp(_data, x._data, _len)); 
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const text *var)
{
  short _len = var->_len;
  serialize(str, count, maxlen, &_len);
  if (_len > 0) {
    pandora_assert(maxlen > (count + _len));
    memcpy(str + count, (char *)var->_data, _len);
    count += _len;
  }
}

inline void unserialize(const char *str, size_t &count, text *var)
{
  short _len;
  unserialize(str, count, &_len);
  var->_len = _len;
  __FREE(var->_data);
  if (var->_len > 0) {
    var->_data = (char *) xmalloc((var->_len + 1) * sizeof(char));
    memcpy(var->_data, str + count, var->_len);
    var->_data[var->_len] = '\0';
    count += var->_len;
  }
}

extern void initTextFromChar(const char *, void *);
extern ostream &operator<< (ostream &, const struct text &);

#endif /* TEXT_H */
