/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef STACK_ENTRY_H
#define STACK_ENTRY_H

#include <libpandora/global.h>

#include <iostream>
#include <libpandora/compentry.h>
#include <libpandora/pandora_limits.h>
#include <libpandora/text.h>
#include <libpandora/error.h>

class StackEntry {
private:
  bool needSetup;
  CompEntry compList[MAX_COMPONENTS];
  int nbComps;

  text paramList[MAX_OPTIONS];
  int nbParams;

public:
  text id;
  bool run;

  inline StackEntry(void);
  inline StackEntry(const text &name);
  StackEntry(const StackEntry &x);
  StackEntry &operator=(const StackEntry &x);
  inline ~StackEntry(void) { }

  size_t write(char *, size_t) const;
  size_t read(const char *);
  void print(ostream *f = &cout);
  void print_graph(ostream *f = &cout);

  bool setup(void);
  int macro(void);

  inline int getNbComps(void) const 		{ return nbComps; }
  inline int getNbParams(void) const 		{ return nbParams; }
  bool includes(const text &cid);

  bool insertStack(StackEntry *se, int where);

  inline void init(const char *name) 		{ id.init(name); }

  inline CompEntry *getCompEntry(int index);
  inline CompEntry *getCompEntry(const text &id);
  inline bool pushComponent(const CompEntry &ce);
  inline void popComponent(void);
  inline void clear(void) 			{ clearComp(); clearParam(); }

  inline void clearComp(void) 			{ nbComps = 0; }
  inline void clearParam(void) 			{ nbParams = 0; }

  inline text &getParam(int index);
  inline bool pushParam(const text &id);

  bool getOrigName(text &orig, const text &alias);

  inline void noRun(void) 			{ run = false; }

  bool setOption(const text &comp, const text &op, const MultiValue &val);
  bool getOption(const text &comp, const text &op, MultiValue *val);

  inline bool lookup(void);

private:
  void setBranches(void);
  void _setBranches(int, int *, int *);
  void setMux(void);
  void setSW(void);  

  void _print_graph(ostream *f, int idx, int b, int mux, int bmux);
};

inline void serialize(char *str, size_t &count, 
		 const size_t maxlen, const StackEntry *se) 
{
  count += se->write(str + count, maxlen - count);
}

inline void unserialize(const char *str, size_t &count, StackEntry *se)
{
  count += se->read(str + count);
}

StackEntry::StackEntry(void) : 
  needSetup(true), nbComps(0), nbParams(0), run(true) 
{
}

StackEntry::StackEntry(const text &name) : 
  needSetup(true), nbComps(0), nbParams(0), id(name), run(true) 
{
}

bool StackEntry::pushComponent(const CompEntry &ce) 
{
  if (nbComps >= MAX_COMPONENTS) return false;
  compList[nbComps] = ce;
  ++nbComps;
  needSetup = true;
  return true;
}

bool StackEntry::pushParam(const text &id) 
{
  if (nbParams >= MAX_OPTIONS) return false;
  paramList[nbParams] = id;
  ++nbParams;
  needSetup = true;
  return true;
}

text &StackEntry::getParam(int index) 
{
  pandora_assert(index >= 0 && index < nbParams);
  return paramList[index];
}

CompEntry *StackEntry::getCompEntry(int index) 
{
  pandora_assert(index >= 0 && index < nbComps);
  return &(compList[index]);
}

CompEntry *StackEntry::getCompEntry(const text &id) 
{
  int i;
  for (i = 0; i < nbComps; ++i)
    if ((compList[i].id == id) 
	||  (compList[i].alias == id))
      return &(compList[i]);

  return NULL;
}

bool StackEntry::lookup(void)
{
  bool res = true;

  for (int i = 0; i < nbComps; ++i)
    res &= compList[i].lookup();

  return res;
}

#endif /* STACK_ENTRY_H */
