/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef SERIALIZE_H
#define SERIALIZE_H

#include <libpandora/global.h>

extern "C" {
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <libpandora/conf/string.h>
#include <libpandora/conf/time.h>
}

#include <libpandora/error.h>

#define serialVar(v) 	serialize(str, count, maxlen, &(v))
#define unserialVar(v) 	unserialize(str, count, &(v))

#define serialPtr(v) 	serialize(str, count, maxlen, v)
#define unserialPtr(v) 	unserialize(str, count, v)

#define serialEnum(typ_t)						\
inline void serialize(char *str, size_t &count,				\
		      const size_t maxlen, const typ_t *var)		\
{									\
  int tmp = (int) *var;							\
  serialize(str, count, maxlen, &tmp);					\
}									\
									\
inline void unserialize(const char *str, size_t &count, typ_t *var)	\
{									\
  int tmp;								\
  unserialize(str, count, &tmp);					\
  *var = (typ_t) tmp;							\
}

#define beginLevel(n)	if (level > (n)) { return count; }

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const int *var)
{
  pandora_assert(maxlen > (count + 4));
  int32_t tmpl = htonl(*var);
  memcpy(str + count, (char *)&tmpl, 4);
  count += 4;
}

inline void unserialize(const char *str, size_t &count, int *var)
{
  int32_t tmpl;
  memcpy((char *)&tmpl, str + count, 4);
  *var = ntohl(tmpl);
  count += 4;
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const in_addr *var)
{
  pandora_assert(maxlen > (count + 4));
  memcpy(str + count, (char *)&(var->s_addr), 4);
  count += 4;
}

inline void unserialize(const char *str, size_t &count, in_addr *var)
{
  int32_t tmpl;
  memcpy((char *)&(var->s_addr), str + count, 4);
  count += 4;
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const short *var)
{
  pandora_assert(maxlen > (count + 2));
  short tmps = htons(*var);
  memcpy(str + count, (char *)&tmps, 2);
  count += 2;
}

inline void unserialize(const char *str, size_t &count, short *var)
{
  short tmps;
  memcpy((char *)&tmps, str + count, 2);
  *var = ntohs(tmps);
  count += 2;
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const char *var)
{
  pandora_assert(maxlen > (count + 1));
  *(str + count) = *var;
  count++;
}

inline void unserialize(const char *str, size_t &count, char *var)
{
  *var = *(str + count);
  count++;
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const float *var)
{
  int ipart = (int) *var;
  int fpart = (int)(1000000*(float)(*var - ipart));
  serialize(str, count, maxlen, &ipart);
  serialize(str, count, maxlen, &fpart);
}

inline void unserialize(const char *str, size_t &count, float *var)
{
  int ipart = 0, fpart = 0;
  unserialize(str, count, &ipart);
  unserialize(str, count, &fpart);
  *var = ((float)ipart + ((float)fpart)/1000000);
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const bool *var)
{
  int tmp = (int) *var;
  serialize(str, count, maxlen, &tmp);
}

inline void unserialize(const char *str, size_t &count, bool *var)
{
  int tmp;
  unserialize(str, count, &tmp);
  *var = tmp;
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const long *var)
{
#if SIZEOF_LONG > SIZEOF_INT
  int l = (int)((*var) >> (8 * SIZEOF_INT));
  int r = (int)(*var);
  serialize(str, count, maxlen, &l);
  serialize(str, count, maxlen, &r);  
#else
  serialize(str, count, maxlen, (const int *) var);
#endif
}

inline void unserialize(const char *str, size_t &count, long *var)
{
#if SIZEOF_LONG > SIZEOF_INT
  int l;
  int r;
  unserialize(str, count, &l);
  unserialize(str, count, &r);
  *var = ((long) l << (8 * SIZEOF_INT)) + r;
#else
  unserialize(str, count, (int *) var);
#endif
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const u_long *var)
{
  serialize(str, count, maxlen, (const long *) var);
}

inline void unserialize(const char *str, size_t &count, u_long *var)
{
  unserialize(str, count, (long *) var);
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const u_int *var)
{
  serialize(str, count, maxlen, (const int *) var);
}

inline void unserialize(const char *str, size_t &count, u_int *var)
{
  unserialize(str, count, (int *) var);
}


inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const u_short *var)
{
  serialize(str, count, maxlen, (const short *) var);
}

inline void unserialize(const char *str, size_t &count, u_short *var)
{
  unserialize(str, count, (short *) var);
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const signed char *var)
{
  serialize(str, count, maxlen, (const char *) var);
}

inline void unserialize(const char *str, size_t &count, signed char *var)
{
  unserialize(str, count, (char *) var);
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const u_char *var)
{
  serialize(str, count, maxlen, (const char *) var);
}

inline void unserialize(const char *str, size_t &count, u_char *var)
{
  unserialize(str, count, (char *) var);
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const timeval *var)
{
  serialize(str, count, maxlen, &(var->tv_sec));
  serialize(str, count, maxlen, &(var->tv_usec));
}

inline void unserialize(const char *str, size_t &count, timeval *var)
{
  unserialize(str, count, &(var->tv_sec));
  unserialize(str, count, &(var->tv_usec));
}

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, char *const *var)
{
  int len = ((*var != NULL) ? strlen(*var) : 0);
  serialize(str, count, maxlen, &len);
  if ((len > 0) && (maxlen > (count + len))) {
    memcpy(str + count, *var, len);
    count += len;
  }
}

inline void unserialize(const char *str, size_t &count, char **var)
{
  int len = 0;
  unserialize(str, count, &len);
  if (len > 0) {
    int truelen;
    if (*var == NULL) {
      *var = (char *) xmalloc((len+1)*sizeof(char));
      truelen = len;
    } else {
      // XXX fix an arbitraty limit so that it cannot be overflow'ed
      truelen = (len < 2047) ? len : 2047;
    }
    memcpy(*var, str + count, truelen);
    (*var)[truelen] = '\0';
    count += truelen;
  } else {
    *var = NULL;
  }
}

#endif /* SERIALIZE_H */
