/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef SELECT_H
#define SELECT_H

#include <libpandora/global.h>

extern "C" {
#include <stdlib.h>
#include <ctype.h>
#include <libpandora/conf/time.h>
#include <libpandora/conf/fcntl.h>
#include <sys/select.h>
           }

#include <iostream>
#include <libpandora/error.h>

#define DEBUG_SELECT 0

typedef bool sel_new_func_t(int, void *);
typedef void sel_end_func_t(int, void *);

struct select_funcs_t {
  int index;
  sel_new_func_t *new_conn;
  sel_end_func_t *end_conn;
  void *udata;

  select_funcs_t(void) : 
    index(-1), new_conn(NULL), end_conn(NULL), udata(NULL) {}
  select_funcs_t(select_funcs_t &x) : index(x.index), 
    new_conn(x.new_conn), end_conn(x.end_conn), udata(x.udata) {}
  select_funcs_t &operator=(select_funcs_t &x) {
    index = x.index;
    new_conn = x.new_conn; end_conn = x.end_conn; udata = x.udata;
    return *this;
  }
  void reset(void) {
    index = -1;
    new_conn = NULL; end_conn = NULL; udata = NULL;
  }
};

class Select {
public:
  enum kind_t { READ, WRITE, EXCEPT };

private:
  fd_set *readfds, *writefds, *exceptfds;
  fd_set fdset;
  
  int fds[FD_SETSIZE];
  select_funcs_t funcs[FD_SETSIZE];
  struct timeval tv, *timeout;
  int ms;

protected:
  int nfds;
  int max_fds;

public:
  inline Select(int milli = -1, kind_t type = READ);
  inline virtual ~Select(void);
  
  inline int select(void) ;
  inline int select(int) ;
  inline void cleanup(void);
  inline void registerFd(int fd, void *user = NULL,
			 sel_new_func_t *nc = &fakeNewConn, 
			 sel_end_func_t *ec = &fakeEndConn);
  inline void unregisterFd(int fd);
  inline int size(void) { return max_fds; }
  inline void setTimeout(int);

private:
  inline void initfds(void);
  static bool fakeNewConn(int, void *) {return true; }
  static void fakeEndConn(int, void *) {}

  inline void printFds(void);
};


Select::Select(int milli, kind_t type) 
  : readfds(NULL), writefds(NULL), exceptfds(NULL),
    timeout(NULL), ms(milli), nfds(0), max_fds(0) 
{
  if (ms >= 0) timeout = &tv;
  memset((char *)funcs, 0, FD_SETSIZE*sizeof(select_funcs_t));
  switch (type) {
  case READ:	readfds = &fdset;	break;
  case WRITE:	writefds = &fdset;	break;
  case EXCEPT:	exceptfds = &fdset;	break;
  }
}
  
Select::~Select(void) 
{
  cleanup();
}

int Select::select(void) 
{
  if (max_fds == 0) return 0;

  initfds();
  int n = ::select(nfds+1, readfds, writefds, exceptfds, timeout);

  switch (n) {
  case -1:	if (errno != EINTR) pandora_pwarning("select"); 	break;
  case  0:	/* timeout expired */ 					break;
  default:
    for (int i = 0, m = n; (m > 0) && (i < max_fds); ++i) {
      int fd = fds[i];
      if (FD_ISSET(fd, &fdset)) {
	--m;
	select_funcs_t *f = &(funcs[fd]);
	if ((*(f->new_conn))(fd, f->udata)) {
	  (*f->end_conn)(fd, f->udata);
	  unregisterFd(fd);
	}
      }
    }
    break;
  }

  return n;
}

int Select::select(int milli) 
{
  setTimeout(milli);
  return select();
}

void Select::cleanup(void) 
{
  for (int i = 0; i < max_fds; ++i) {
    select_funcs_t *f = &(funcs[fds[i]]);
    (*f->end_conn)(fds[i], f->udata);
    f->reset();
  }
  max_fds = 0;
}
  
void Select::registerFd(int fd, void *user, 
			sel_new_func_t *nc, 
			sel_end_func_t *ec) 
{
#if DEBUG_SELECT
  pandora_debug("[select] adding file desc   #" << fd);
#endif

#if 1
  int flags = fcntl(fd, F_GETFL);
  if (flags != -1) {
    int nflags = flags | O_NONBLOCK;
    if (flags != nflags)
      flags = fcntl(fd, F_SETFL, flags);
  }
  if (flags == -1) {
    pandora_pwarning("fcntl");
  }  
#endif

  select_funcs_t *f = &(funcs[fd]);
  if (f->end_conn != NULL) {
    (*(f->end_conn))(fd, f->udata);
    unregisterFd(fd);
  }

  fds[max_fds] = fd;

  f->index = max_fds;
  f->new_conn = (nc != NULL) ? nc : &fakeNewConn;
  f->end_conn = (ec != NULL) ? ec : &fakeEndConn;
  f->udata = user;

  ++max_fds;
  if (fd > nfds) nfds = fd;

#if DEBUG_SELECT
  printFds();
#endif
}

void Select::unregisterFd(int fd) 
{
#if DEBUG_SELECT
  pandora_debug("[select] removing file desc #" << fd);
#endif

  select_funcs_t *f = &(funcs[fd]);
  int index = f->index;
  if (index < 0) return;

  f->reset();

  for (int i = index; i < (max_fds - 1); ++i) {
    fds[i] = fds[i+1];
    (funcs[fds[i+1]]).index = i;
  }

  --max_fds;
  if (fd == nfds) {
    nfds = 0;
    for (int i = 0; i < max_fds; ++i) {
      if (fds[i] > nfds) nfds = fds[i];
    }
  }

#if DEBUG_SELECT
  printFds();
#endif
}

void Select::initfds(void) 
{
  FD_ZERO(&fdset);
  for (int i = 0; i < max_fds; ++i) FD_SET(fds[i], &fdset);
  if (ms >= 0) {
    tv.tv_sec = ms/1000;
    tv.tv_usec = (ms - (1000*tv.tv_sec))*1000;
  }
}

void Select::setTimeout(int milli)
{
  ms = milli; 
  timeout = (ms >= 0) ? &tv : NULL;
}

void Select::printFds(void) 
{
  CERR(0) << "[select] ";
  for (int i = 0; i < max_fds; ++i) cerr << fds[i] << " ";
  cerr << "\n";  
}

#endif /* SELECT_H */
