/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef SCHED_TASK_H
#define SCHED_TASK_H

#include <libpandora/global.h>

class scheduler;

typedef void (sched_task_func_t)(void *, time_t);

class sched_task_t {
private:
  scheduler		*sched;
  time_t		sched_time;
  sched_task_func_t 	*exp_func;
  sched_task_func_t 	*rem_func;
  void 			*data;

public:
  inline sched_task_t(void);
  inline sched_task_t(sched_task_func_t *, void *);
  inline sched_task_t(sched_task_func_t *, sched_task_func_t *, void *);
  inline sched_task_t(const sched_task_t &x);
  inline sched_task_t &operator =(const sched_task_t &x);
  inline ~sched_task_t(void) { }
  
  inline      operator long(void) const;
  inline bool operator ==(const sched_task_t &x) const;
  inline bool operator !=(const sched_task_t &x) const;

  inline sched_task_t &init(sched_task_func_t *f, void *d);
  inline sched_task_t &init(sched_task_func_t *, sched_task_func_t *, void *);
  inline sched_task_func_t *getExpFunc(void) const	{ return exp_func; }
  inline sched_task_func_t *getRemFunc(void) const	{ return rem_func; }
  inline void *getData(void) const			{ return data; }

  inline void expired(time_t) const;
  inline void removed(time_t) const;

  inline void setScheduler(scheduler *s) 	{ sched = s; }

  inline bool   isScheduled(void) const		{ return (sched_time != 0); }
  inline time_t getSchedule(void) const		{ return sched_time; }

  inline void   reset(void) 			{ sched_time = 0; }
  void   schedule(time_t t);
  void   remove(void);  
};

inline sched_task_t::sched_task_t(void) 
  : sched(NULL), sched_time(0) ,
    exp_func(NULL), rem_func(NULL), data(NULL) 		
{ 
}

inline sched_task_t::sched_task_t(sched_task_func_t *f, void *d) 
  : sched(NULL), sched_time(0), 
    exp_func(f), rem_func(NULL), data(d)
{ 
}

inline sched_task_t::sched_task_t(sched_task_func_t *f1, 
				  sched_task_func_t *f2, 
				  void *d) 
  : sched(NULL), sched_time(0), 
    exp_func(f1), rem_func(f2), data(d)
{ 
}

inline sched_task_t::sched_task_t(const sched_task_t &x) 
  : sched(x.sched), sched_time(x.sched_time), 
    exp_func(x.exp_func), rem_func(x.rem_func), data(x.data)
{
}

inline sched_task_t &sched_task_t::operator =(const sched_task_t &x) 
{
  sched = x.sched; sched_time = x.sched_time; 
  exp_func = x.exp_func; rem_func = x.rem_func; data = x.data;
  return *this;
}

inline sched_task_t::operator long(void) const
{
  return ((long)data ^ (long)exp_func);
}

inline bool sched_task_t::operator ==(const sched_task_t &x) const
{
  return (data == x.data) 
    && (exp_func == x.exp_func);
}

inline bool sched_task_t::operator !=(const sched_task_t &x) const
{
  return !operator==(x);
}

inline sched_task_t &sched_task_t::init(sched_task_func_t *f, void *d) 
{
  exp_func = f; data = d;
  return *this;
}

inline sched_task_t &sched_task_t::init(sched_task_func_t *f1, 
					sched_task_func_t *f2, 
					void *d) 
{
  exp_func = f1; rem_func = f2; data = d;
  return *this;
}

inline void sched_task_t::expired(time_t t) const
{
  if (exp_func == NULL) return;
  (*exp_func)(data, t);
}

inline void sched_task_t::removed(time_t t) const
{
  if (rem_func == NULL) return;
  (*rem_func)(data, t);
}


#endif /* SCHED_TASK_H */
