/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/sched_manager.h>
#include <libpandora/scheduler.h>

#define SCHED_MANAGER_DEBUG	0

static void smgr_expire(sched_manager_t::event_t *, time_t);

void sched_manager_t::cleanup(void)
{
#if SCHED_MANAGER_DEBUG
  pandora_debug("[sched_manager cleanup]");
#endif

  sched_handle_t handle;
  sched_task_t task;

  while(events.size() > 0) {
    keysValuesDo(events, handle, task) {
      remove(handle, task);
    }
  }
  sched = NULL;
  curhandle = NIL_SCHED_HANDLE;
}

sched_handle_t sched_manager_t::schedule(const sched_task_t &task, 
					 int timeout,
					 bool resched)
{
  if (sched == NULL) return NIL_SCHED_HANDLE;
  if (timeout < 1)   return NIL_SCHED_HANDLE;

  sched_handle_t h = nextHandle();
  event_t *ev = new event_t;
  ev->smgr =     this;
  ev->handle =   h;
  ev->exp_func = task.getExpFunc();
  ev->rem_func = task.getRemFunc();
  ev->data =     task.getData();
  ev->timeout =  timeout;
  ev->resched =  resched;
  ev->expired =  false;

  sched_task_t ntask((sched_task_func_t *) &smgr_expire, ev);

#if SCHED_MANAGER_DEBUG
  pandora_debug("[scheduling   task #" << (long)ntask << " (" << h << ")]");
#endif

  ntask.setScheduler(sched);
  ntask.schedule(sched->get_time() + timeout);
  events.atPut(h, ntask);

  return h;
}

void sched_manager_t::reschedule(sched_handle_t h, int timeout, bool resched)
{
  sched_task_t task = events.atOrNil(h);
  if (task == sched_task_t()) return;
  event_t *ev = (event_t *)task.getData();
  pandora_assert(ev != NULL);
  pandora_assert(sched != NULL);

#if SCHED_MANAGER_DEBUG
  pandora_debug("[rescheduling task #" << (long)task << " (" << h << ")]");
#endif

  (ev->expired ? task.reset() : task.remove());
  ev->timeout =  timeout;
  ev->resched =  resched;
  ev->expired =  false;
  task.schedule(sched->get_time() + timeout);
  events.atPut(h, task);
}

void sched_manager_t::remove(sched_handle_t h)
{
  sched_task_t task = events.atOrNil(h);
  if (task == sched_task_t()) return;
  remove(h, task);
}

void sched_manager_t::remove(sched_handle_t h, sched_task_t &task)
{
#if SCHED_MANAGER_DEBUG
  pandora_debug("[removing     task #" << (long)task << " (" << h << ")]");
#endif

  events.removeKey(h);
  event_t *ev = (event_t *)task.getData();
  if (!ev->expired) {
    if (ev->rem_func != NULL)
      (*(ev->rem_func))(ev->data, sched->get_time());
    task.remove();
  }
  __DELETE(ev);
}

static void smgr_expire(sched_manager_t::event_t *ev, time_t t)
{
  pandora_assert (ev != NULL);
  ev->expired = true;
  if (ev->exp_func != NULL)
    (*(ev->exp_func))(ev->data, t);

  pandora_assert (ev->smgr != NULL);
  if (ev->resched) {
    ev->smgr->reschedule(ev->handle, ev->timeout, true);
  } else {
    ev->smgr->remove(ev->handle);
  }
}
